Package["core-runtime"].queue("accounts-password",function () {/* Imports */
var Meteor = Package.meteor.Meteor;
var global = Package.meteor.global;
var meteorEnv = Package.meteor.meteorEnv;
var EmitterPromise = Package.meteor.EmitterPromise;
var Accounts = Package['accounts-base'].Accounts;
var SHA256 = Package.sha.SHA256;
var EJSON = Package.ejson.EJSON;
var DDP = Package['ddp-client'].DDP;
var DDPServer = Package['ddp-server'].DDPServer;
var Email = Package.email.Email;
var EmailInternals = Package.email.EmailInternals;
var Random = Package.random.Random;
var check = Package.check.check;
var Match = Package.check.Match;
var ECMAScript = Package.ecmascript.ECMAScript;
var meteorInstall = Package.modules.meteorInstall;
var Promise = Package.promise.Promise;

var require = meteorInstall({"node_modules":{"meteor":{"accounts-password":{"email_templates.js":function module(require,exports,module){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// packages/accounts-password/email_templates.js                                                                    //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
!module.wrapAsync(async function (module, __reifyWaitForDeps__, __reify_async_result__) {
  "use strict";
  try {
    let _objectSpread;
    module.link("@babel/runtime/helpers/objectSpread2", {
      default(v) {
        _objectSpread = v;
      }
    }, 0);
    if (__reifyWaitForDeps__()) (await __reifyWaitForDeps__())();
    const greet = welcomeMsg => (user, url) => {
      const greeting = user.profile && user.profile.name ? "Hello ".concat(user.profile.name, ",") : 'Hello,';
      return "".concat(greeting, "\n\n").concat(welcomeMsg, ", simply click the link below.\n\n").concat(url, "\n\nThank you.\n");
    };

    /**
     * @summary Options to customize emails sent from the Accounts system.
     * @locus Server
     * @importFromPackage accounts-base
     */
    Accounts.emailTemplates = _objectSpread(_objectSpread({}, Accounts.emailTemplates || {}), {}, {
      from: 'Accounts Example <no-reply@example.com>',
      siteName: Meteor.absoluteUrl().replace(/^https?:\/\//, '').replace(/\/$/, ''),
      resetPassword: {
        subject: () => "How to reset your password on ".concat(Accounts.emailTemplates.siteName),
        text: greet('To reset your password')
      },
      verifyEmail: {
        subject: () => "How to verify email address on ".concat(Accounts.emailTemplates.siteName),
        text: greet('To verify your account email')
      },
      enrollAccount: {
        subject: () => "An account has been created for you on ".concat(Accounts.emailTemplates.siteName),
        text: greet('To start using the service')
      }
    });
    __reify_async_result__();
  } catch (_reifyError) {
    return __reify_async_result__(_reifyError);
  }
  __reify_async_result__()
}, {
  self: this,
  async: false
});
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"password_server.js":function module(require,exports,module){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// packages/accounts-password/password_server.js                                                                    //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
!module.wrapAsync(async function (module, __reifyWaitForDeps__, __reify_async_result__) {
  "use strict";
  try {
    let _objectSpread;
    module.link("@babel/runtime/helpers/objectSpread2", {
      default(v) {
        _objectSpread = v;
      }
    }, 0);
    let argon2;
    module.link("argon2", {
      default(v) {
        argon2 = v;
      }
    }, 0);
    let bcryptHash, bcryptCompare;
    module.link("bcrypt", {
      hash(v) {
        bcryptHash = v;
      },
      compare(v) {
        bcryptCompare = v;
      }
    }, 1);
    let Accounts;
    module.link("meteor/accounts-base", {
      Accounts(v) {
        Accounts = v;
      }
    }, 2);
    if (__reifyWaitForDeps__()) (await __reifyWaitForDeps__())();
    // Utility for grabbing user
    const getUserById = async (id, options) => await Meteor.users.findOneAsync(id, Accounts._addDefaultFieldSelector(options));

    // User records have two fields that are used for password-based login:
    // - 'services.password.bcrypt', which stores the bcrypt password, which will be deprecated
    // - 'services.password.argon2', which stores the argon2 password
    //
    // When the client sends a password to the server, it can either be a
    // string (the plaintext password) or an object with keys 'digest' and
    // 'algorithm' (must be "sha-256" for now). The Meteor client always sends
    // password objects { digest: *, algorithm: "sha-256" }, but DDP clients
    // that don't have access to SHA can just send plaintext passwords as
    // strings.
    //
    // When the server receives a plaintext password as a string, it always
    // hashes it with SHA256 before passing it into bcrypt / argon2. When the server
    // receives a password as an object, it asserts that the algorithm is
    // "sha-256" and then passes the digest to bcrypt / argon2.

    Accounts._bcryptRounds = () => Accounts._options.bcryptRounds || 10;
    Accounts._argon2Enabled = () => Accounts._options.argon2Enabled || false;
    const ARGON2_TYPES = {
      argon2i: argon2.argon2i,
      argon2d: argon2.argon2d,
      argon2id: argon2.argon2id
    };
    Accounts._argon2Type = () => ARGON2_TYPES[Accounts._options.argon2Type] || argon2.argon2id;
    Accounts._argon2TimeCost = () => Accounts._options.argon2TimeCost || 2;
    Accounts._argon2MemoryCost = () => Accounts._options.argon2MemoryCost || 19456;
    Accounts._argon2Parallelism = () => Accounts._options.argon2Parallelism || 1;

    /**
     * Extracts the string to be encrypted using bcrypt or Argon2 from the given `password`.
     *
     * @param {string|Object} password - The password provided by the client. It can be:
     *  - A plaintext string password.
     *  - An object with the following properties:
     *      @property {string} digest - The hashed password.
     *      @property {string} algorithm - The hashing algorithm used. Must be "sha-256".
     *
     * @returns {string} - The resulting password string to encrypt.
     *
     * @throws {Error} - If the `algorithm` in the password object is not "sha-256".
     */
    const getPasswordString = password => {
      if (typeof password === "string") {
        password = SHA256(password);
      } else {
        // 'password' is an object
        if (password.algorithm !== "sha-256") {
          throw new Error("Invalid password hash algorithm. " + "Only 'sha-256' is allowed.");
        }
        password = password.digest;
      }
      return password;
    };

    /**
     * Encrypt the given `password` using either bcrypt or Argon2.
     * @param password can be a string (in which case it will be run through SHA256 before encryption) or an object with properties `digest` and `algorithm` (in which case we bcrypt or Argon2 `password.digest`).
     * @returns {Promise<string>} The encrypted password.
     */
    const hashPassword = async password => {
      password = getPasswordString(password);
      if (Accounts._argon2Enabled() === true) {
        return await argon2.hash(password, {
          type: Accounts._argon2Type(),
          timeCost: Accounts._argon2TimeCost(),
          memoryCost: Accounts._argon2MemoryCost(),
          parallelism: Accounts._argon2Parallelism()
        });
      } else {
        return await bcryptHash(password, Accounts._bcryptRounds());
      }
    };

    // Extract the number of rounds used in the specified bcrypt hash.
    const getRoundsFromBcryptHash = hash => {
      let rounds;
      if (hash) {
        const hashSegments = hash.split("$");
        if (hashSegments.length > 2) {
          rounds = parseInt(hashSegments[2], 10);
        }
      }
      return rounds;
    };
    Accounts._getRoundsFromBcryptHash = getRoundsFromBcryptHash;

    /**
     * Extract readable parameters from an Argon2 hash string.
     * @param {string} hash - The Argon2 hash string.
     * @returns {object} An object containing the parsed parameters.
     * @throws {Error} If the hash format is invalid.
     */
    function getArgon2Params(hash) {
      const regex = /^\$(argon2(?:i|d|id))\$v=\d+\$m=(\d+),t=(\d+),p=(\d+)/;
      const match = hash.match(regex);
      if (!match) {
        throw new Error("Invalid Argon2 hash format.");
      }
      const [, type, memoryCost, timeCost, parallelism] = match;
      return {
        type: ARGON2_TYPES[type],
        timeCost: parseInt(timeCost, 10),
        memoryCost: parseInt(memoryCost, 10),
        parallelism: parseInt(parallelism, 10)
      };
    }
    Accounts._getArgon2Params = getArgon2Params;
    const getUserPasswordHash = user => {
      var _user$services, _user$services$passwo, _user$services2, _user$services2$passw;
      return ((_user$services = user.services) === null || _user$services === void 0 ? void 0 : (_user$services$passwo = _user$services.password) === null || _user$services$passwo === void 0 ? void 0 : _user$services$passwo.argon2) || ((_user$services2 = user.services) === null || _user$services2 === void 0 ? void 0 : (_user$services2$passw = _user$services2.password) === null || _user$services2$passw === void 0 ? void 0 : _user$services2$passw.bcrypt);
    };
    Accounts._checkPasswordUserFields = {
      _id: 1,
      services: 1
    };
    const isBcrypt = hash => {
      // bcrypt hashes start with $2a$ or $2b$
      return hash.startsWith("$2");
    };
    const isArgon = hash => {
      // argon2 hashes start with $argon2i$, $argon2d$ or $argon2id$
      return hash.startsWith("$argon2");
    };
    const updateUserPasswordDefered = (user, formattedPassword) => {
      Meteor.defer(async () => {
        await updateUserPassword(user, formattedPassword);
      });
    };

    /**
     * Hashes the provided password and returns an object that can be used to update the user's password.
     * @param formattedPassword
     * @returns {Promise<{$set: {"services.password.bcrypt": string}}|{$unset: {"services.password.bcrypt": number}, $set: {"services.password.argon2": string}}>}
     */
    const getUpdatorForUserPassword = async formattedPassword => {
      const encryptedPassword = await hashPassword(formattedPassword);
      if (Accounts._argon2Enabled() === false) {
        return {
          $set: {
            "services.password.bcrypt": encryptedPassword
          },
          $unset: {
            "services.password.argon2": 1
          }
        };
      } else if (Accounts._argon2Enabled() === true) {
        return {
          $set: {
            "services.password.argon2": encryptedPassword
          },
          $unset: {
            "services.password.bcrypt": 1
          }
        };
      }
    };
    const updateUserPassword = async (user, formattedPassword) => {
      const updator = await getUpdatorForUserPassword(formattedPassword);
      await Meteor.users.updateAsync({
        _id: user._id
      }, updator);
    };

    /**
     * Checks whether the provided password matches the hashed password stored in the user's database record.
     *
     * @param {Object} user - The user object containing at least:
     *   @property {string} _id - The user's unique identifier.
     *   @property {Object} services - The user's services data.
     *   @property {Object} services.password - The user's password object.
     *   @property {string} [services.password.argon2] - The Argon2 hashed password.
     *   @property {string} [services.password.bcrypt] - The bcrypt hashed password, deprecated
     *
     * @param {string|Object} password - The password provided by the client. It can be:
     *   - A plaintext string password.
     *   - An object with the following properties:
     *       @property {string} digest - The hashed password.
     *       @property {string} algorithm - The hashing algorithm used. Must be "sha-256".
     *
     * @returns {Promise<Object>} - A result object with the following properties:
     *   @property {string} userId - The user's unique identifier.
     *   @property {Object} [error] - An error object if the password does not match or an error occurs.
     *
     * @throws {Error} - If an unexpected error occurs during the process.
     */
    const checkPasswordAsync = async (user, password) => {
      const result = {
        userId: user._id
      };
      const formattedPassword = getPasswordString(password);
      const hash = getUserPasswordHash(user);
      const argon2Enabled = Accounts._argon2Enabled();
      if (argon2Enabled === false) {
        if (isArgon(hash)) {
          // this is a rollback feature, enabling to switch back from argon2 to bcrypt if needed
          // TODO : deprecate this
          console.warn("User has an argon2 password and argon2 is not enabled, rolling back to bcrypt encryption");
          const match = await argon2.verify(hash, formattedPassword);
          if (!match) {
            result.error = Accounts._handleError("Incorrect password", false);
          } else {
            // The password checks out, but the user's stored password needs to be updated to argon2
            updateUserPasswordDefered(user, {
              digest: formattedPassword,
              algorithm: "sha-256"
            });
          }
        } else {
          const hashRounds = getRoundsFromBcryptHash(hash);
          const match = await bcryptCompare(formattedPassword, hash);
          if (!match) {
            result.error = Accounts._handleError("Incorrect password", false);
          } else if (hash) {
            const paramsChanged = hashRounds !== Accounts._bcryptRounds();
            // The password checks out, but the user's bcrypt hash needs to be updated
            // to match current bcrypt settings
            if (paramsChanged === true) {
              updateUserPasswordDefered(user, {
                digest: formattedPassword,
                algorithm: "sha-256"
              });
            }
          }
        }
      } else if (argon2Enabled === true) {
        if (isBcrypt(hash)) {
          // migration code from bcrypt to argon2
          const match = await bcryptCompare(formattedPassword, hash);
          if (!match) {
            result.error = Accounts._handleError("Incorrect password", false);
          } else {
            // The password checks out, but the user's stored password needs to be updated to argon2
            updateUserPasswordDefered(user, {
              digest: formattedPassword,
              algorithm: "sha-256"
            });
          }
        } else {
          // argon2 password
          const argon2Params = getArgon2Params(hash);
          const match = await argon2.verify(hash, formattedPassword);
          if (!match) {
            result.error = Accounts._handleError("Incorrect password", false);
          } else if (hash) {
            const paramsChanged = argon2Params.memoryCost !== Accounts._argon2MemoryCost() || argon2Params.timeCost !== Accounts._argon2TimeCost() || argon2Params.parallelism !== Accounts._argon2Parallelism() || argon2Params.type !== Accounts._argon2Type();
            if (paramsChanged === true) {
              // The password checks out, but the user's argon2 hash needs to be updated with the right params
              updateUserPasswordDefered(user, {
                digest: formattedPassword,
                algorithm: "sha-256"
              });
            }
          }
        }
      }
      return result;
    };
    Accounts._checkPasswordAsync = checkPasswordAsync;

    ///
    /// LOGIN
    ///

    // XXX maybe this belongs in the check package
    const NonEmptyString = Match.Where(x => {
      check(x, String);
      return x.length > 0;
    });
    const passwordValidator = Match.OneOf(Match.Where(str => {
      var _Meteor$settings, _Meteor$settings$pack, _Meteor$settings$pack2;
      return Match.test(str, String) && str.length <= ((_Meteor$settings = Meteor.settings) === null || _Meteor$settings === void 0 ? void 0 : (_Meteor$settings$pack = _Meteor$settings.packages) === null || _Meteor$settings$pack === void 0 ? void 0 : (_Meteor$settings$pack2 = _Meteor$settings$pack.accounts) === null || _Meteor$settings$pack2 === void 0 ? void 0 : _Meteor$settings$pack2.passwordMaxLength) || 256;
    }), {
      digest: Match.Where(str => Match.test(str, String) && str.length === 64),
      algorithm: Match.OneOf('sha-256')
    });

    // Handler to login with a password.
    //
    // The Meteor client sets options.password to an object with keys
    // 'digest' (set to SHA256(password)) and 'algorithm' ("sha-256").
    //
    // For other DDP clients which don't have access to SHA, the handler
    // also accepts the plaintext password in options.password as a string.
    //
    // (It might be nice if servers could turn the plaintext password
    // option off. Or maybe it should be opt-in, not opt-out?
    // Accounts.config option?)
    //
    // Note that neither password option is secure without SSL.
    //
    Accounts.registerLoginHandler("password", async options => {
      var _Accounts$_check2faEn, _Accounts;
      if (!options.password) return undefined; // don't handle

      check(options, {
        user: Accounts._userQueryValidator,
        password: passwordValidator,
        code: Match.Optional(NonEmptyString)
      });
      const user = await Accounts._findUserByQuery(options.user, {
        fields: _objectSpread({
          services: 1
        }, Accounts._checkPasswordUserFields)
      });
      if (!user) {
        Accounts._handleError("User not found");
      }
      if (!getUserPasswordHash(user)) {
        Accounts._handleError("User has no password set");
      }
      const result = await checkPasswordAsync(user, options.password);
      // This method is added by the package accounts-2fa
      // First the login is validated, then the code situation is checked
      if (!result.error && (_Accounts$_check2faEn = (_Accounts = Accounts)._check2faEnabled) !== null && _Accounts$_check2faEn !== void 0 && _Accounts$_check2faEn.call(_Accounts, user)) {
        if (!options.code) {
          Accounts._handleError('2FA code must be informed', true, 'no-2fa-code');
        }
        if (!Accounts._isTokenValid(user.services.twoFactorAuthentication.secret, options.code)) {
          Accounts._handleError('Invalid 2FA code', true, 'invalid-2fa-code');
        }
      }
      return result;
    });

    ///
    /// CHANGING
    ///

    /**
     * @summary Change a user's username asynchronously. Use this instead of updating the
     * database directly. The operation will fail if there is an existing user
     * with a username only differing in case.
     * @locus Server
     * @param {String} userId The ID of the user to update.
     * @param {String} newUsername A new username for the user.
     * @importFromPackage accounts-base
     */
    Accounts.setUsername = async (userId, newUsername) => {
      check(userId, NonEmptyString);
      check(newUsername, NonEmptyString);
      const user = await getUserById(userId, {
        fields: {
          username: 1
        }
      });
      if (!user) {
        Accounts._handleError("User not found");
      }
      const oldUsername = user.username;

      // Perform a case insensitive check for duplicates before update
      await Accounts._checkForCaseInsensitiveDuplicates('username', 'Username', newUsername, user._id);
      await Meteor.users.updateAsync({
        _id: user._id
      }, {
        $set: {
          username: newUsername
        }
      });

      // Perform another check after update, in case a matching user has been
      // inserted in the meantime
      try {
        await Accounts._checkForCaseInsensitiveDuplicates('username', 'Username', newUsername, user._id);
      } catch (ex) {
        // Undo update if the check fails
        await Meteor.users.updateAsync({
          _id: user._id
        }, {
          $set: {
            username: oldUsername
          }
        });
        throw ex;
      }
    };

    // Let the user change their own password if they know the old
    // password. `oldPassword` and `newPassword` should be objects with keys
    // `digest` and `algorithm` (representing the SHA256 of the password).
    Meteor.methods({
      changePassword: async function (oldPassword, newPassword) {
        check(oldPassword, passwordValidator);
        check(newPassword, passwordValidator);
        if (!this.userId) {
          throw new Meteor.Error(401, "Must be logged in");
        }
        const user = await getUserById(this.userId, {
          fields: _objectSpread({
            services: 1
          }, Accounts._checkPasswordUserFields)
        });
        if (!user) {
          Accounts._handleError("User not found");
        }
        if (!getUserPasswordHash(user)) {
          Accounts._handleError("User has no password set");
        }
        const result = await checkPasswordAsync(user, oldPassword);
        if (result.error) {
          throw result.error;
        }

        // It would be better if this removed ALL existing tokens and replaced
        // the token for the current connection with a new one, but that would
        // be tricky, so we'll settle for just replacing all tokens other than
        // the one for the current connection.
        const currentToken = Accounts._getLoginToken(this.connection.id);
        const updator = await getUpdatorForUserPassword(newPassword);
        await Meteor.users.updateAsync({
          _id: this.userId
        }, {
          $set: updator.$set,
          $pull: {
            "services.resume.loginTokens": {
              hashedToken: {
                $ne: currentToken
              }
            }
          },
          $unset: _objectSpread({
            "services.password.reset": 1
          }, updator.$unset)
        });
        return {
          passwordChanged: true
        };
      }
    });

    // Force change the users password.

    /**
     * @summary Forcibly change the password for a user.
     * @locus Server
     * @param {String} userId The id of the user to update.
     * @param {String} newPlaintextPassword A new password for the user.
     * @param {Object} [options]
     * @param {Object} options.logout Logout all current connections with this userId (default: true)
     * @importFromPackage accounts-base
     */
    Accounts.setPasswordAsync = async (userId, newPlaintextPassword, options) => {
      check(userId, String);
      check(newPlaintextPassword, Match.Where(str => {
        var _Meteor$settings2, _Meteor$settings2$pac, _Meteor$settings2$pac2;
        return Match.test(str, String) && str.length <= ((_Meteor$settings2 = Meteor.settings) === null || _Meteor$settings2 === void 0 ? void 0 : (_Meteor$settings2$pac = _Meteor$settings2.packages) === null || _Meteor$settings2$pac === void 0 ? void 0 : (_Meteor$settings2$pac2 = _Meteor$settings2$pac.accounts) === null || _Meteor$settings2$pac2 === void 0 ? void 0 : _Meteor$settings2$pac2.passwordMaxLength) || 256;
      }));
      check(options, Match.Maybe({
        logout: Boolean
      }));
      options = _objectSpread({
        logout: true
      }, options);
      const user = await getUserById(userId, {
        fields: {
          _id: 1
        }
      });
      if (!user) {
        throw new Meteor.Error(403, "User not found");
      }
      let updator = await getUpdatorForUserPassword(newPlaintextPassword);
      updator.$unset = updator.$unset || {};
      updator.$unset["services.password.reset"] = 1;
      if (options.logout) {
        updator.$unset["services.resume.loginTokens"] = 1;
      }
      await Meteor.users.updateAsync({
        _id: user._id
      }, updator);
    };

    ///
    /// RESETTING VIA EMAIL
    ///

    // Utility for plucking addresses from emails
    const pluckAddresses = function () {
      let emails = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
      return emails.map(email => email.address);
    };

    // Method called by a user to request a password reset email. This is
    // the start of the reset process.
    Meteor.methods({
      forgotPassword: async options => {
        check(options, {
          email: String
        });
        const user = await Accounts.findUserByEmail(options.email, {
          fields: {
            emails: 1
          }
        });
        if (!user) {
          Accounts._handleError("User not found");
        }
        const emails = pluckAddresses(user.emails);
        const caseSensitiveEmail = emails.find(email => email.toLowerCase() === options.email.toLowerCase());
        await Accounts.sendResetPasswordEmail(user._id, caseSensitiveEmail);
      }
    });

    /**
     * @summary Asynchronously generates a reset token and saves it into the database.
     * @locus Server
     * @param {String} userId The id of the user to generate the reset token for.
     * @param {String} email Which address of the user to generate the reset token for. This address must be in the user's `emails` list. If `null`, defaults to the first email in the list.
     * @param {String} reason `resetPassword` or `enrollAccount`.
     * @param {Object} [extraTokenData] Optional additional data to be added into the token record.
     * @returns {Promise<Object>} Promise of an object with {email, user, token} values.
     * @importFromPackage accounts-base
     */
    Accounts.generateResetToken = async (userId, email, reason, extraTokenData) => {
      // Make sure the user exists, and email is one of their addresses.
      // Don't limit the fields in the user object since the user is returned
      // by the function and some other fields might be used elsewhere.
      const user = await getUserById(userId);
      if (!user) {
        Accounts._handleError("Can't find user");
      }

      // pick the first email if we weren't passed an email.
      if (!email && user.emails && user.emails[0]) {
        email = user.emails[0].address;
      }

      // make sure we have a valid email
      if (!email || !pluckAddresses(user.emails).includes(email)) {
        Accounts._handleError("No such email for user.");
      }
      const token = Random.secret();
      const tokenRecord = {
        token,
        email,
        when: new Date()
      };
      if (reason === 'resetPassword') {
        tokenRecord.reason = 'reset';
      } else if (reason === 'enrollAccount') {
        tokenRecord.reason = 'enroll';
      } else if (reason) {
        // fallback so that this function can be used for unknown reasons as well
        tokenRecord.reason = reason;
      }
      if (extraTokenData) {
        Object.assign(tokenRecord, extraTokenData);
      }
      // if this method is called from the enroll account work-flow then
      // store the token record in 'services.password.enroll' db field
      // else store the token record in in 'services.password.reset' db field
      if (reason === "enrollAccount") {
        await Meteor.users.updateAsync({
          _id: user._id
        }, {
          $set: {
            "services.password.enroll": tokenRecord
          }
        });
        // before passing to template, update user object with new token
        Meteor._ensure(user, "services", "password").enroll = tokenRecord;
      } else {
        await Meteor.users.updateAsync({
          _id: user._id
        }, {
          $set: {
            "services.password.reset": tokenRecord
          }
        });
        // before passing to template, update user object with new token
        Meteor._ensure(user, "services", "password").reset = tokenRecord;
      }
      return {
        email,
        user,
        token
      };
    };

    /**
     * @summary Generates asynchronously an e-mail verification token and saves it into the database.
     * @locus Server
     * @param {String} userId The id of the user to generate the  e-mail verification token for.
     * @param {String} email Which address of the user to generate the e-mail verification token for. This address must be in the user's `emails` list. If `null`, defaults to the first unverified email in the list.
     * @param {Object} [extraTokenData] Optional additional data to be added into the token record.
     * @returns {Promise<Object>} Promise of an object with {email, user, token} values.
     * @importFromPackage accounts-base
     */
    Accounts.generateVerificationToken = async (userId, email, extraTokenData) => {
      // Make sure the user exists, and email is one of their addresses.
      // Don't limit the fields in the user object since the user is returned
      // by the function and some other fields might be used elsewhere.
      const user = await getUserById(userId);
      if (!user) {
        Accounts._handleError("Can't find user");
      }

      // pick the first unverified email if we weren't passed an email.
      if (!email) {
        const emailRecord = (user.emails || []).find(e => !e.verified);
        email = (emailRecord || {}).address;
        if (!email) {
          Accounts._handleError("That user has no unverified email addresses.");
        }
      }

      // make sure we have a valid email
      if (!email || !pluckAddresses(user.emails).includes(email)) {
        Accounts._handleError("No such email for user.");
      }
      const token = Random.secret();
      const tokenRecord = {
        token,
        // TODO: This should probably be renamed to "email" to match reset token record.
        address: email,
        when: new Date()
      };
      if (extraTokenData) {
        Object.assign(tokenRecord, extraTokenData);
      }
      await Meteor.users.updateAsync({
        _id: user._id
      }, {
        $push: {
          'services.email.verificationTokens': tokenRecord
        }
      });

      // before passing to template, update user object with new token
      Meteor._ensure(user, 'services', 'email');
      if (!user.services.email.verificationTokens) {
        user.services.email.verificationTokens = [];
      }
      user.services.email.verificationTokens.push(tokenRecord);
      return {
        email,
        user,
        token
      };
    };

    // send the user an email with a link that when opened allows the user
    // to set a new password, without the old password.

    /**
     * @summary Send an email asynchronously with a link the user can use to reset their password.
     * @locus Server
     * @param {String} userId The id of the user to send email to.
     * @param {String} [email] Optional. Which address of the user's to send the email to. This address must be in the user's `emails` list. Defaults to the first email in the list.
     * @param {Object} [extraTokenData] Optional additional data to be added into the token record.
     * @param {Object} [extraParams] Optional additional params to be added to the reset url.
     * @returns {Promise<Object>} Promise of an object with {email, user, token, url, options} values.
     * @importFromPackage accounts-base
     */
    Accounts.sendResetPasswordEmail = async (userId, email, extraTokenData, extraParams) => {
      const {
        email: realEmail,
        user,
        token
      } = await Accounts.generateResetToken(userId, email, 'resetPassword', extraTokenData);
      const url = await Accounts._resolvePromise(Accounts.urls.resetPassword(token, extraParams));
      const options = await Accounts.generateOptionsForEmail(realEmail, user, url, 'resetPassword');
      await Email.sendAsync(options);
      if (Meteor.isDevelopment && !Meteor.isPackageTest) {
        console.log("\nReset password URL: ".concat(url));
      }
      return {
        email: realEmail,
        user,
        token,
        url,
        options
      };
    };

    // send the user an email informing them that their account was created, with
    // a link that when opened both marks their email as verified and forces them
    // to choose their password. The email must be one of the addresses in the
    // user's emails field, or undefined to pick the first email automatically.
    //
    // This is not called automatically. It must be called manually if you
    // want to use enrollment emails.

    /**
     * @summary Send an email asynchronously with a link the user can use to set their initial password.
     * @locus Server
     * @param {String} userId The id of the user to send email to.
     * @param {String} [email] Optional. Which address of the user's to send the email to. This address must be in the user's `emails` list. Defaults to the first email in the list.
     * @param {Object} [extraTokenData] Optional additional data to be added into the token record.
     * @param {Object} [extraParams] Optional additional params to be added to the enrollment url.
     * @returns {Promise<Object>} Promise of an object {email, user, token, url, options} values.
     * @importFromPackage accounts-base
     */
    Accounts.sendEnrollmentEmail = async (userId, email, extraTokenData, extraParams) => {
      const {
        email: realEmail,
        user,
        token
      } = await Accounts.generateResetToken(userId, email, 'enrollAccount', extraTokenData);
      const url = await Accounts._resolvePromise(Accounts.urls.enrollAccount(token, extraParams));
      const options = await Accounts.generateOptionsForEmail(realEmail, user, url, 'enrollAccount');
      await Email.sendAsync(options);
      if (Meteor.isDevelopment && !Meteor.isPackageTest) {
        console.log("\nEnrollment email URL: ".concat(url));
      }
      return {
        email: realEmail,
        user,
        token,
        url,
        options
      };
    };

    // Take token from sendResetPasswordEmail or sendEnrollmentEmail, change
    // the users password, and log them in.
    Meteor.methods({
      resetPassword: async function () {
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
          args[_key] = arguments[_key];
        }
        const token = args[0];
        const newPassword = args[1];
        return await Accounts._loginMethod(this, "resetPassword", args, "password", async () => {
          var _Accounts$_check2faEn2, _Accounts2;
          check(token, String);
          check(newPassword, passwordValidator);
          let user = await Meteor.users.findOneAsync({
            "services.password.reset.token": token
          }, {
            fields: {
              services: 1,
              emails: 1
            }
          });
          let isEnroll = false;
          // if token is in services.password.reset db field implies
          // this method is was not called from enroll account workflow
          // else this method is called from enroll account workflow
          if (!user) {
            user = await Meteor.users.findOneAsync({
              "services.password.enroll.token": token
            }, {
              fields: {
                services: 1,
                emails: 1
              }
            });
            isEnroll = true;
          }
          if (!user) {
            throw new Meteor.Error(403, "Token expired");
          }
          let tokenRecord = {};
          if (isEnroll) {
            tokenRecord = user.services.password.enroll;
          } else {
            tokenRecord = user.services.password.reset;
          }
          const {
            when,
            email
          } = tokenRecord;
          let tokenLifetimeMs = Accounts._getPasswordResetTokenLifetimeMs();
          if (isEnroll) {
            tokenLifetimeMs = Accounts._getPasswordEnrollTokenLifetimeMs();
          }
          const currentTimeMs = Date.now();
          if (currentTimeMs - when > tokenLifetimeMs) throw new Meteor.Error(403, "Token expired");
          if (!pluckAddresses(user.emails).includes(email)) return {
            userId: user._id,
            error: new Meteor.Error(403, "Token has invalid email address")
          };

          // NOTE: We're about to invalidate tokens on the user, who we might be
          // logged in as. Make sure to avoid logging ourselves out if this
          // happens. But also make sure not to leave the connection in a state
          // of having a bad token set if things fail.
          const oldToken = Accounts._getLoginToken(this.connection.id);
          Accounts._setLoginToken(user._id, this.connection, null);
          const resetToOldToken = () => Accounts._setLoginToken(user._id, this.connection, oldToken);
          const updator = await getUpdatorForUserPassword(newPassword);
          try {
            // Update the user record by:
            // - Changing the password to the new one
            // - Forgetting about the reset token or enroll token that was just used
            // - Verifying their email, since they got the password reset via email.
            let affectedRecords = {};
            // if reason is enroll then check services.password.enroll.token field for affected records
            if (isEnroll) {
              affectedRecords = await Meteor.users.updateAsync({
                _id: user._id,
                "emails.address": email,
                "services.password.enroll.token": token
              }, {
                $set: _objectSpread({
                  "emails.$.verified": true
                }, updator.$set),
                $unset: _objectSpread({
                  "services.password.enroll": 1
                }, updator.$unset)
              });
            } else {
              affectedRecords = await Meteor.users.updateAsync({
                _id: user._id,
                "emails.address": email,
                "services.password.reset.token": token
              }, {
                $set: _objectSpread({
                  "emails.$.verified": true
                }, updator.$set),
                $unset: _objectSpread({
                  "services.password.reset": 1
                }, updator.$unset)
              });
            }
            if (affectedRecords !== 1) return {
              userId: user._id,
              error: new Meteor.Error(403, "Invalid email")
            };
          } catch (err) {
            resetToOldToken();
            throw err;
          }

          // Replace all valid login tokens with new ones (changing
          // password should invalidate existing sessions).
          await Accounts._clearAllLoginTokens(user._id);
          if ((_Accounts$_check2faEn2 = (_Accounts2 = Accounts)._check2faEnabled) !== null && _Accounts$_check2faEn2 !== void 0 && _Accounts$_check2faEn2.call(_Accounts2, user)) {
            return {
              userId: user._id,
              error: Accounts._handleError('Changed password, but user not logged in because 2FA is enabled', false, '2fa-enabled')
            };
          }
          return {
            userId: user._id
          };
        });
      }
    });

    ///
    /// EMAIL VERIFICATION
    ///

    // send the user an email with a link that when opened marks that
    // address as verified

    /**
     * @summary Send an email asynchronously with a link the user can use verify their email address.
     * @locus Server
     * @param {String} userId The id of the user to send email to.
     * @param {String} [email] Optional. Which address of the user's to send the email to. This address must be in the user's `emails` list. Defaults to the first unverified email in the list.
     * @param {Object} [extraTokenData] Optional additional data to be added into the token record.
     * @param {Object} [extraParams] Optional additional params to be added to the verification url.
     * @returns {Promise<Object>} Promise of an object with {email, user, token, url, options} values.
     * @importFromPackage accounts-base
     */
    Accounts.sendVerificationEmail = async (userId, email, extraTokenData, extraParams) => {
      // XXX Also generate a link using which someone can delete this
      // account if they own said address but weren't those who created
      // this account.

      const {
        email: realEmail,
        user,
        token
      } = await Accounts.generateVerificationToken(userId, email, extraTokenData);
      const url = await Accounts._resolvePromise(Accounts.urls.verifyEmail(token, extraParams));
      const options = await Accounts.generateOptionsForEmail(realEmail, user, url, 'verifyEmail');
      await Email.sendAsync(options);
      if (Meteor.isDevelopment && !Meteor.isPackageTest) {
        console.log("\nVerification email URL: ".concat(url));
      }
      return {
        email: realEmail,
        user,
        token,
        url,
        options
      };
    };

    // Take token from sendVerificationEmail, mark the email as verified,
    // and log them in.
    Meteor.methods({
      verifyEmail: async function () {
        for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }
        const token = args[0];
        return await Accounts._loginMethod(this, "verifyEmail", args, "password", async () => {
          var _Accounts$_check2faEn3, _Accounts3;
          check(token, String);
          const user = await Meteor.users.findOneAsync({
            'services.email.verificationTokens.token': token
          }, {
            fields: {
              services: 1,
              emails: 1
            }
          });
          if (!user) throw new Meteor.Error(403, "Verify email link expired");
          const tokenRecord = await user.services.email.verificationTokens.find(t => t.token == token);
          if (!tokenRecord) return {
            userId: user._id,
            error: new Meteor.Error(403, "Verify email link expired")
          };
          const emailsRecord = user.emails.find(e => e.address == tokenRecord.address);
          if (!emailsRecord) return {
            userId: user._id,
            error: new Meteor.Error(403, "Verify email link is for unknown address")
          };

          // By including the address in the query, we can use 'emails.$' in the
          // modifier to get a reference to the specific object in the emails
          // array. See
          // http://www.mongodb.org/display/DOCS/Updating/#Updating-The%24positionaloperator)
          // http://www.mongodb.org/display/DOCS/Updating#Updating-%24pull
          await Meteor.users.updateAsync({
            _id: user._id,
            'emails.address': tokenRecord.address
          }, {
            $set: {
              'emails.$.verified': true
            },
            $pull: {
              'services.email.verificationTokens': {
                address: tokenRecord.address
              }
            }
          });
          if ((_Accounts$_check2faEn3 = (_Accounts3 = Accounts)._check2faEnabled) !== null && _Accounts$_check2faEn3 !== void 0 && _Accounts$_check2faEn3.call(_Accounts3, user)) {
            return {
              userId: user._id,
              error: Accounts._handleError('Email verified, but user not logged in because 2FA is enabled', false, '2fa-enabled')
            };
          }
          return {
            userId: user._id
          };
        });
      }
    });

    /**
     * @summary Asynchronously replace an email address for a user. Use this instead of directly
     * updating the database. The operation will fail if there is a different user
     * with an email only differing in case. If the specified user has an existing
     * email only differing in case however, we replace it.
     * @locus Server
     * @param {String} userId The ID of the user to update.
     * @param {String} oldEmail The email address to replace.
     * @param {String} newEmail The new email address to use.
     * @param {Boolean} [verified] Optional - whether the new email address should
     * be marked as verified. Defaults to false.
     * @importFromPackage accounts-base
     */
    Accounts.replaceEmailAsync = async (userId, oldEmail, newEmail, verified) => {
      check(userId, NonEmptyString);
      check(oldEmail, NonEmptyString);
      check(newEmail, NonEmptyString);
      check(verified, Match.Optional(Boolean));
      if (verified === void 0) {
        verified = false;
      }
      const user = await getUserById(userId, {
        fields: {
          _id: 1
        }
      });
      if (!user) throw new Meteor.Error(403, "User not found");

      // Ensure no user already has this new email
      await Accounts._checkForCaseInsensitiveDuplicates("emails.address", "Email", newEmail, user._id);
      const result = await Meteor.users.updateAsync({
        _id: user._id,
        'emails.address': oldEmail
      }, {
        $set: {
          'emails.$.address': newEmail,
          'emails.$.verified': verified
        }
      });
      if (result.modifiedCount === 0) {
        throw new Meteor.Error(404, "No user could be found with old email");
      }
    };

    /**
     * @summary Asynchronously add an email address for a user. Use this instead of directly
     * updating the database. The operation will fail if there is a different user
     * with an email only differing in case. If the specified user has an existing
     * email only differing in case however, we replace it.
     * @locus Server
     * @param {String} userId The ID of the user to update.
     * @param {String} newEmail A new email address for the user.
     * @param {Boolean} [verified] Optional - whether the new email address should
     * be marked as verified. Defaults to false.
     * @importFromPackage accounts-base
     */
    Accounts.addEmailAsync = async (userId, newEmail, verified) => {
      check(userId, NonEmptyString);
      check(newEmail, NonEmptyString);
      check(verified, Match.Optional(Boolean));
      if (verified === void 0) {
        verified = false;
      }
      const user = await getUserById(userId, {
        fields: {
          emails: 1
        }
      });
      if (!user) throw new Meteor.Error(403, "User not found");

      // Allow users to change their own email to a version with a different case

      // We don't have to call checkForCaseInsensitiveDuplicates to do a case
      // insensitive check across all emails in the database here because: (1) if
      // there is no case-insensitive duplicate between this user and other users,
      // then we are OK and (2) if this would create a conflict with other users
      // then there would already be a case-insensitive duplicate and we can't fix
      // that in this code anyway.
      const caseInsensitiveRegExp = new RegExp("^".concat(Meteor._escapeRegExp(newEmail), "$"), "i");

      // TODO: This is a linear search. If we have a lot of emails.
      //  we should consider using a different data structure.
      const updatedEmail = async function () {
        let emails = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
        let _id = arguments.length > 1 ? arguments[1] : undefined;
        let updated = false;
        for (const email of emails) {
          if (caseInsensitiveRegExp.test(email.address)) {
            await Meteor.users.updateAsync({
              _id: _id,
              "emails.address": email.address
            }, {
              $set: {
                "emails.$.address": newEmail,
                "emails.$.verified": verified
              }
            });
            updated = true;
          }
        }
        return updated;
      };
      const didUpdateOwnEmail = await updatedEmail(user.emails, user._id);

      // In the other updates below, we have to do another call to
      // checkForCaseInsensitiveDuplicates to make sure that no conflicting values
      // were added to the database in the meantime. We don't have to do this for
      // the case where the user is updating their email address to one that is the
      // same as before, but only different because of capitalization. Read the
      // big comment above to understand why.

      if (didUpdateOwnEmail) {
        return;
      }

      // Perform a case insensitive check for duplicates before update
      await Accounts._checkForCaseInsensitiveDuplicates("emails.address", "Email", newEmail, user._id);
      await Meteor.users.updateAsync({
        _id: user._id
      }, {
        $addToSet: {
          emails: {
            address: newEmail,
            verified: verified
          }
        }
      });

      // Perform another check after update, in case a matching user has been
      // inserted in the meantime
      try {
        await Accounts._checkForCaseInsensitiveDuplicates("emails.address", "Email", newEmail, user._id);
      } catch (ex) {
        // Undo update if the check fails
        await Meteor.users.updateAsync({
          _id: user._id
        }, {
          $pull: {
            emails: {
              address: newEmail
            }
          }
        });
        throw ex;
      }
    };

    /**
     * @summary Remove an email address asynchronously for a user. Use this instead of updating
     * the database directly.
     * @locus Server
     * @param {String} userId The ID of the user to update.
     * @param {String} email The email address to remove.
     * @importFromPackage accounts-base
     */
    Accounts.removeEmail = async (userId, email) => {
      check(userId, NonEmptyString);
      check(email, NonEmptyString);
      const user = await getUserById(userId, {
        fields: {
          _id: 1
        }
      });
      if (!user) throw new Meteor.Error(403, "User not found");
      await Meteor.users.updateAsync({
        _id: user._id
      }, {
        $pull: {
          emails: {
            address: email
          }
        }
      });
    };

    ///
    /// CREATING USERS
    ///

    // Shared createUser function called from the createUser method, both
    // if originates in client or server code. Calls user provided hooks,
    // does the actual user insertion.
    //
    // returns the user id
    const createUser = async options => {
      // Unknown keys allowed, because a onCreateUserHook can take arbitrary
      // options.
      check(options, Match.ObjectIncluding({
        username: Match.Optional(String),
        email: Match.Optional(String),
        password: Match.Optional(passwordValidator)
      }));
      const {
        username,
        email,
        password
      } = options;
      if (!username && !email) throw new Meteor.Error(400, "Need to set a username or email");
      const user = {
        services: {}
      };
      if (password) {
        const hashed = await hashPassword(password);
        const argon2Enabled = Accounts._argon2Enabled();
        if (argon2Enabled === false) {
          user.services.password = {
            bcrypt: hashed
          };
        } else {
          user.services.password = {
            argon2: hashed
          };
        }
      }
      return await Accounts._createUserCheckingDuplicates({
        user,
        email,
        username,
        options
      });
    };

    // method for create user. Requests come from the client.
    Meteor.methods({
      createUser: async function () {
        for (var _len3 = arguments.length, args = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
          args[_key3] = arguments[_key3];
        }
        const options = args[0];
        return await Accounts._loginMethod(this, "createUser", args, "password", async () => {
          // createUser() above does more checking.
          check(options, Object);
          if (Accounts._options.forbidClientAccountCreation) return {
            error: new Meteor.Error(403, "Signups forbidden")
          };
          const userId = await Accounts.createUserVerifyingEmail(options);

          // client gets logged in as the new user afterwards.
          return {
            userId: userId
          };
        });
      }
    });

    /**
     * @summary Creates an user asynchronously and sends an email if `options.email` is informed.
     * Then if the `sendVerificationEmail` option from the `Accounts` package is
     * enabled, you'll send a verification email if `options.password` is informed,
     * otherwise you'll send an enrollment email.
     * @locus Server
     * @param {Object} options The options object to be passed down when creating
     * the user
     * @param {String} options.username A unique name for this user.
     * @param {String} options.email The user's email address.
     * @param {String} options.password The user's password. This is __not__ sent in plain text over the wire.
     * @param {Object} options.profile The user's profile, typically including the `name` field.
     * @importFromPackage accounts-base
     * */
    Accounts.createUserVerifyingEmail = async options => {
      options = _objectSpread({}, options);
      // Create user. result contains id and token.
      const userId = await createUser(options);
      // safety belt. createUser is supposed to throw on error. send 500 error
      // instead of sending a verification email with empty userid.
      if (!userId) throw new Error("createUser failed to insert new user");

      // If `Accounts._options.sendVerificationEmail` is set, register
      // a token to verify the user's primary email, and send it to
      // that address.
      if (options.email && Accounts._options.sendVerificationEmail) {
        if (options.password) {
          await Accounts.sendVerificationEmail(userId, options.email);
        } else {
          await Accounts.sendEnrollmentEmail(userId, options.email);
        }
      }
      return userId;
    };

    // Create user directly on the server.
    //
    // Unlike the client version, this does not log you in as this user
    // after creation.
    //
    // returns Promise<userId> or throws an error if it can't create
    //
    // XXX add another argument ("server options") that gets sent to onCreateUser,
    // which is always empty when called from the createUser method? eg, "admin:
    // true", which we want to prevent the client from setting, but which a custom
    // method calling Accounts.createUser could set?
    //

    Accounts.createUserAsync = createUser;

    // Create user directly on the server.
    //
    // Unlike the client version, this does not log you in as this user
    // after creation.
    //
    // returns userId or throws an error if it can't create
    //
    // XXX add another argument ("server options") that gets sent to onCreateUser,
    // which is always empty when called from the createUser method? eg, "admin:
    // true", which we want to prevent the client from setting, but which a custom
    // method calling Accounts.createUser could set?
    //

    Accounts.createUser = Accounts.createUserAsync;

    ///
    /// PASSWORD-SPECIFIC INDEXES ON USERS
    ///
    await Meteor.users.createIndexAsync('services.email.verificationTokens.token', {
      unique: true,
      sparse: true
    });
    await Meteor.users.createIndexAsync('services.password.reset.token', {
      unique: true,
      sparse: true
    });
    await Meteor.users.createIndexAsync('services.password.enroll.token', {
      unique: true,
      sparse: true
    });
    __reify_async_result__();
  } catch (_reifyError) {
    return __reify_async_result__(_reifyError);
  }
  __reify_async_result__()
}, {
  self: this,
  async: true
});
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"node_modules":{"argon2":{"package.json":function module(require,exports,module){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// node_modules/meteor/accounts-password/node_modules/argon2/package.json                                           //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
module.exports = {
  "name": "argon2",
  "version": "0.41.1",
  "main": "argon2.cjs"
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"argon2.cjs":function module(require,exports,module){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// node_modules/meteor/accounts-password/node_modules/argon2/argon2.cjs                                             //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
module.useNode();
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}},"bcrypt":{"package.json":function module(require,exports,module){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// node_modules/meteor/accounts-password/node_modules/bcrypt/package.json                                           //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
module.exports = {
  "name": "bcrypt",
  "version": "5.0.1",
  "main": "./bcrypt"
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"bcrypt.js":function module(require,exports,module){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// node_modules/meteor/accounts-password/node_modules/bcrypt/bcrypt.js                                              //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
module.useNode();
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}}}}}}},{
  "extensions": [
    ".js",
    ".json"
  ]
});


/* Exports */
return {
  require: require,
  eagerModulePaths: [
    "/node_modules/meteor/accounts-password/email_templates.js",
    "/node_modules/meteor/accounts-password/password_server.js"
  ]
}});

//# sourceURL=meteor://💻app/packages/accounts-password.js
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
