import { IJsonSchemaCollection } from "./schemas/json/IJsonSchemaCollection";
import { IJsonSchemaUnit } from "./schemas/json/IJsonSchemaUnit";
import { IValidation } from "./IValidation";
import { Primitive } from "./Primitive";
import { TypeGuardError } from "./TypeGuardError";
/**
 * > You must configure the generic argument `Types`.
 *
 * JSON Schemas Generator.
 *
 * Creates a JSON schema list which contains both main JSON schemas and
 * components. Note that, all of the named types are stored in the
 * {@link IJsonSchemaCollection.components} property for the `$ref` referencing.
 *
 * Also, you can specify the OpenAPI version by configuring the second generic
 * argument `Version`. For reference, the default version is `"3.1"`, and the
 * key difference between `"3.0"` and `"3.1"` is whether supporting the tuple
 * type or not.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template Types Tuple of target types
 * @template Version Version of OpenAPI specification. Default is 3.1
 * @returns JSON schema collection
 */
export declare function schemas(): never;
/**
 * JSON Schemas Generator.
 *
 * Creates a JSON schema list which contains both main JSON schemas and
 * components. Note that, all of the named types are stored in the
 * {@link IJsonSchemaCollection.components} property for the `$ref` referencing.
 *
 * Also, you can specify the OpenAPI version by configuring the second generic
 * argument `Version`. For reference, the default version is `"3.1"`, and the
 * key difference between `"3.0"` and `"3.1"` is whether supporting the tuple
 * type or not.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template Types Tuple of target types
 * @template Version Version of OpenAPI specification. Default is 3.1
 * @returns JSON schema collection
 */
export declare function schemas<Types extends unknown[], Version extends "3.0" | "3.1" = "3.1">(): IJsonSchemaCollection<Version, Types>;
/**
 * > You must configure the generic argument `Type`.
 *
 * JSON schema generator.
 *
 * Creates a JSON schema unit which contains a main JSON schema and its
 * components. Note that, all of the named types are stored in the
 * {@link IJsonSchemaUnit.components} property for the `$ref` referencing.
 *
 * Also, you can specify the OpenAPI version by configuring the second generic
 * argument `Version`. For reference, the default version is `"3.1"`, and key
 * different of `"3.0"` and `"3.1"` is whether supporting the tuple type or
 * not.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template Type Target type
 * @template Version Version of OpenAPI specification. Default is 3.1
 * @returns JSON schema unit
 */
export declare function schema(): never;
/**
 * JSON schema generator.
 *
 * Creates a JSON schema unit which contains a main JSON schema and its
 * components. Note that, all of the named types are stored in the
 * {@link IJsonSchemaUnit.components} property for the `$ref` referencing.
 *
 * Also, you can specify the OpenAPI version by configuring the second generic
 * argument `Version`. For reference, the default version is `"3.1"`, and key
 * different of `"3.0"` and `"3.1"` is whether supporting the tuple type or
 * not.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template Type Target type
 * @template Version Version of OpenAPI specification. Default is 3.1
 * @returns JSON schema unit
 */
export declare function schema<Type extends unknown, Version extends "3.0" | "3.1" = "3.1">(): IJsonSchemaUnit<Version, Type>;
/**
 * > You must configure the generic argument `T`.
 *
 * Safe `JSON.parse()` function with type assertion.
 *
 * `typia.json.assertParse()` is a combination function of `JSON.parse()` and
 * {@link assert}. Therefore, it converts a JSON (JavaScript Object Notation)
 * string to a `T` typed instance with type assertion.
 *
 * In such reason, when parsed JSON string value is not matched with the type
 * `T`, it throws {@link TypeGuardError} or custom error generated by
 * _errorFactory_. Otherwise, if there's no problem with the parsed value, the
 * parsed value will be returned.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @param input JSON string
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns Parsed value
 */
export declare function assertParse(input: string, errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): never;
/**
 * Safe `JSON.parse()` function with type assertion.
 *
 * `typia.json.assertParse()` is a combination function of `JSON.parse()` and
 * {@link assert}. Therefore, it converts a JSON (JavaScript Object Notation)
 * string to a `T` typed instance with type assertion.
 *
 * In such reason, when parsed JSON string value is not matched with the type
 * `T`, it throws {@link TypeGuardError} or custom error generated by
 * _errorFactory_. Otherwise, there's no problem on the parsed value, the parsed
 * value would be returned.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @param input JSON string
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns Parsed value
 */
export declare function assertParse<T>(input: string, errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): Primitive<T>;
/**
 * > You must configure the generic argument `T`.
 *
 * Safe `JSON.parse()` function with type checking.
 *
 * `typia.json.isParse()` is a combination function of `JSON.parse()` and
 * {@link is}. Therefore, it converts a JSON (JavaScript Object Notation) string
 * to a `T` typed instance with type checking.
 *
 * In such reason, when parsed JSON string value is not matched with the type
 * `T`, it returns `null` value. Otherwise, there's no problem on the parsed
 * value, the parsed value will be returned.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @param input JSON string
 * @returns Parsed value when exact type, otherwise `null`
 */
export declare function isParse(input: string): never;
/**
 * Safe `JSON.parse()` function with type checking.
 *
 * `typia.json.isParse()` is a combination function of `JSON.parse()` and
 * {@link is}. Therefore, it converts a JSON (JavaScript Object Notation) string
 * to a `T` typed instance with type checking.
 *
 * In such reason, when parsed JSON string value is not matched with the type
 * `T`, it returns `null` value. Otherwise, there's no problem on the parsed
 * value, the parsed value will be returned.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @param input JSON string
 * @returns Parsed value when exact type, otherwise `null`
 */
export declare function isParse<T>(input: string): Primitive<T> | null;
/**
 * > You must configure the generic argument `T`.
 *
 * Safe `JSON.parse()` function with detailed type validation.
 *
 * `typia.json.validateParse()` is a combination function of `JSON.parse()` and
 * {@link validate}. Therefore, it converts a JSON (JavaScript Object Notation)
 * string to a `T` typed instance with detailed type validation.
 *
 * In such reason, when parsed JSON string value is not matched with the type
 * `T`, it returns {@link IValidation.IFailure} value with detailed error
 * reasons. Otherwise, there's no problem on the parsed value, the parsed value
 * will be stored in `data` property of the output {@link IValidation.ISuccess}
 * instance.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @param input JSON string
 * @returns Validation result with JSON parsed value
 */
export declare function validateParse(input: string): never;
/**
 * Safe `JSON.parse()` function with detailed type validation.
 *
 * `typia.json.validateParse()` is a combination function of `JSON.parse()` and
 * {@link validate}. Therefore, it converts a JSON (JavaScript Object Notation)
 * string to a `T` typed instance with detailed type validation.
 *
 * In such reason, when parsed JSON string value is not matched with the type
 * `T`, it returns {@link IValidation.IFailure} value with detailed error
 * reasons. Otherwise, there's no problem on the parsed value, the parsed value
 * will be stored in `data` property of the output {@link IValidation.ISuccess}
 * instance.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @param input JSON string
 * @returns Validation result with JSON parsed value
 */
export declare function validateParse<T>(input: string): IValidation<Primitive<T>>;
/**
 * 8x faster `JSON.stringify()` function.
 *
 * Converts an input value to a JSON (JavaScript Object Notation) string, about
 * 8x faster than the native `JSON.stringify()` function. The 5x faster
 * principle is because it writes an optimized JSON conversion plan, only for
 * the type `T`.
 *
 * For reference, this `typia.json.stringify()` does not validate the input
 * value type. It just believes that the input value is following the type `T`.
 * Therefore, if you can't ensure the input value type, it will be better to
 * call one of below functions instead.
 *
 * - {@link assertStringify}
 * - {@link isStringify}
 * - {@link validateStringify}
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param input A value to be converted
 * @returns JSON string value
 */
export declare function stringify<T>(input: T): string;
/**
 * 5x faster `JSON.stringify()` function with type assertion.
 *
 * `typia.json.assertStringify()` is a combination function of {@link assert} and
 * {@link stringify}. Therefore, it converts an input value to JSON (JavaScript
 * Object Notation) string, with type assertion.
 *
 * In such reason, when `input` value is not matched with the type `T`, it
 * throws an {@link TypeGuardError} or custom error generated by _errorFactory_.
 * Otherwise, there's no problem on the `input` value, JSON string will be
 * returned.
 *
 * For reference, with type assertion, it is even 5x times faster than the
 * native `JSON.stringify()` function. So, just enjoy the safe and fast JSON
 * conversion with confidence.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param input A value to be asserted and converted
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns JSON string value
 */
export declare function assertStringify<T>(input: T, errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): string;
/**
 * 5x faster `JSON.stringify()` function with type assertion.
 *
 * `typia.json.assertStringify()` is a combination function of {@link assert} and
 * {@link stringify}. Therefore, it converts an input value to JSON (JavaScript
 * Object Notation) string, with type assertion.
 *
 * In such reason, when `input` value is not matched with the type `T`, it
 * throws an {@link TypeGuardError} or custom error generated by _errorFactory_.
 * Otherwise, there's no problem on the `input` value, JSON string will be
 * returned.
 *
 * For reference, with type assertion, it is even 5x times faster than the
 * native `JSON.stringify()` function. So, just enjoy the safe and fast JSON
 * conversion with confidence.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param input A value to be asserted and converted
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns JSON string value
 */
export declare function assertStringify<T>(input: T, errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): unknown;
/**
 * 7x faster `JSON.stringify()` function with type checking.
 *
 * `typia.json.stringify()` is a combination function of {@link is} and
 * {@link stringify}. Therefore, it converts an input value to JSON (JavaScript
 * Object Notation) string, with type checking.
 *
 * In such reason, when `input` value is not matched with the type `T`, it
 * returns `null` value. Otherwise, there's no problem on the `input` value,
 * JSON string will be returned.
 *
 * For reference, with type checking, it is even 7x times faster than the native
 * `JSON.stringify()` function. So, just enjoy the safe and fast JSON conversion
 * with confidence.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param input A value to be checked and converted
 * @returns JSON string value when exact type, otherwise null
 */
export declare function isStringify<T>(input: T): string | null;
/**
 * 7x faster `JSON.stringify()` function with type checking.
 *
 * `typia.json.isStringify()` is a combination function of {@link is} and
 * {@link stringify}. Therefore, it converts an input value to JSON (JavaScript
 * Object Notation) string, with type checking.
 *
 * In such reason, when `input` value is not matched with the type `T`, it
 * returns `null` value. Otherwise, there's no problem on the `input` value,
 * JSON string will be returned.
 *
 * For reference, with type checking, it is even 7x times faster than the native
 * `JSON.stringify()` function. So, just enjoy the safe and fast JSON conversion
 * with confidence.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param input A value to be checked and converted
 * @returns JSON string value when exact type, otherwise null
 */
export declare function isStringify<T>(input: unknown): string | null;
/**
 * 5x faster `JSON.stringify()` function with detailed type validation.
 *
 * `typia.json.validateStringify()` is a combination function of {@link validate}
 * and {@link stringify}. Therefore, it converts an input value to JSON
 * (JavaScript Object Notation) string, with detailed type validation.
 *
 * In such reason, when `input` value is not matched with the type `T`, it
 * returns {@link IValidation.IFailure} value with detailed error reasons.
 * Otherwise, there's no problem on the `input` value, JSON string will be
 * stored in `data` property of the output {@link IValidation.ISuccess}
 * instance.
 *
 * For reference, with detailed type validation, it is even 5x times faster than
 * the native `JSON.stringify()` function. So, just enjoy the safe and fast JSON
 * conversion with confidence.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param input A value to be checked and converted
 * @returns Validation result with JSON string value
 */
export declare function validateStringify<T>(input: T): IValidation<string>;
/**
 * 5x faster `JSON.stringify()` function with detailed type validation.
 *
 * `typia.json.validateStringify()` is a combination function of {@link validate}
 * and {@link stringify}. Therefore, it converts an input value to JSON
 * (JavaScript Object Notation) string, with detailed type validation.
 *
 * In such reason, when `input` value is not matched with the type `T`, it
 * returns {@link IValidation.IFailure} value with detailed error reasons.
 * Otherwise, there's no problem on the `input` value, JSON string will be
 * stored in `data` property of the output {@link IValidation.ISuccess}
 * instance.
 *
 * For reference, with detailed type validation, it is even 5x times faster than
 * the native `JSON.stringify()` function. So, just enjoy the safe and fast JSON
 * conversion with confidence.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param input A value to be checked and converted
 * @returns Validation result with JSON string value
 */
export declare function validateStringify<T>(input: unknown): IValidation<string>;
/**
 * Creates a reusable {@link isParse} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @returns Nothing until you configure the generic argument `T`
 * @throws Compile error
 * @danger You must configure the generic argument `T`
 */
export declare function createIsParse(): never;
/**
 * Creates a reusable {@link isParse} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @returns A reusable `isParse` function
 */
export declare function createIsParse<T>(): (input: string) => Primitive<T> | null;
/**
 * Creates a reusable {@link assertParse} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns Nothing until you configure the generic argument `T`
 * @throws Compile error
 * @danger You must configure the generic argument `T`
 */
export declare function createAssertParse(errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): never;
/**
 * Creates a reusable {@link assertParse} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns A reusable `assertParse` function
 */
export declare function createAssertParse<T>(errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): (input: string) => Primitive<T>;
/**
 * Creates a reusable {@link validateParse} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @returns Nothing until you configure the generic argument `T`
 * @throws Compile error
 * @danger You must configure the generic argument `T`
 */
export declare function createValidateParse(): never;
/**
 * Creates a reusable {@link validateParse} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Expected type of parsed value
 * @returns A reusable `validateParse` function
 */
export declare function createValidateParse<T>(): (input: string) => IValidation<Primitive<T>>;
/**
 * Creates a reusable {@link stringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @returns Nothing until you configure the generic argument `T`
 * @throws Compile error
 * @danger You must configure the generic argument `T`
 */
export declare function createStringify(): never;
/**
 * Creates a reusable {@link stringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @returns A reusable `stringify` function
 */
export declare function createStringify<T>(): (input: T) => string;
/**
 * Creates a reusable {@link assertStringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns Nothing until you configure the generic argument `T`
 * @throws Compile error
 * @danger You must configure the generic argument `T`
 */
export declare function createAssertStringify(errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): never;
/**
 * Creates a reusable {@link assertStringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @param errorFactory Custom error factory. Default is `TypeGuardError`
 * @returns A reusable `assertStringify` function
 */
export declare function createAssertStringify<T>(errorFactory?: undefined | ((props: TypeGuardError.IProps) => Error)): (input: unknown) => string;
/**
 * Creates a reusable {@link isStringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @returns Nothing until you configure the generic argument `T`
 * @throws Compile error
 * @danger You must configure the generic argument `T`
 */
export declare function createIsStringify(): never;
/**
 * Creates a reusable {@link isStringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @returns A reusable `isStringify` function
 */
export declare function createIsStringify<T>(): (input: unknown) => string | null;
/**
 * Creates a reusable {@link validateStringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @returns Nothing until you configure the generic argument `T`
 * @throws Compile error
 * @danger You must configure the generic argument `T`
 */
export declare function createValidateStringify(): never;
/**
 * Creates a reusable {@link validateStringify} function.
 *
 * @author Jeongho Nam - https://github.com/samchon
 * @template T Type of the input value
 * @returns A reusable `validateStringify` function
 */
export declare function createValidateStringify<T>(): (input: unknown) => IValidation<string>;
