"use strict";
var __spreadArray = (this && this.__spreadArray) || function (to, from, pack) {
    if (pack || arguments.length === 2) for (var i = 0, l = from.length, ar; i < l; i++) {
        if (ar || !(i in from)) {
            if (!ar) ar = Array.prototype.slice.call(from, 0, i);
            ar[i] = from[i];
        }
    }
    return to.concat(ar || Array.prototype.slice.call(from));
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.HotPatcher = void 0;
var functions_1 = require("./functions");
var HOT_PATCHER_TYPE = "@@HOTPATCHER";
var NOOP = function () { };
function createNewItem(method) {
    return {
        original: method,
        methods: [method],
        final: false
    };
}
/**
 * Hot patching manager class
 */
var HotPatcher = /** @class */ (function () {
    function HotPatcher() {
        this._configuration = {
            registry: {},
            getEmptyAction: "null"
        };
        this.__type__ = HOT_PATCHER_TYPE;
    }
    Object.defineProperty(HotPatcher.prototype, "configuration", {
        /**
         * Configuration object reference
         * @readonly
         */
        get: function () {
            return this._configuration;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(HotPatcher.prototype, "getEmptyAction", {
        /**
         * The action to take when a non-set method is requested
         * Possible values: null/throw
         */
        get: function () {
            return this.configuration.getEmptyAction;
        },
        set: function (newAction) {
            this.configuration.getEmptyAction = newAction;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Control another hot-patcher instance
     * Force the remote instance to use patched methods from calling instance
     * @param target The target instance to control
     * @param allowTargetOverrides Allow the target to override patched methods on
     * the controller (default is false)
     * @returns Returns self
     * @throws {Error} Throws if the target is invalid
     */
    HotPatcher.prototype.control = function (target, allowTargetOverrides) {
        var _this = this;
        if (allowTargetOverrides === void 0) { allowTargetOverrides = false; }
        if (!target || target.__type__ !== HOT_PATCHER_TYPE) {
            throw new Error("Failed taking control of target HotPatcher instance: Invalid type or object");
        }
        Object.keys(target.configuration.registry).forEach(function (foreignKey) {
            if (_this.configuration.registry.hasOwnProperty(foreignKey)) {
                if (allowTargetOverrides) {
                    _this.configuration.registry[foreignKey] = Object.assign({}, target.configuration.registry[foreignKey]);
                }
            }
            else {
                _this.configuration.registry[foreignKey] = Object.assign({}, target.configuration.registry[foreignKey]);
            }
        });
        target._configuration = this.configuration;
        return this;
    };
    /**
     * Execute a patched method
     * @param key The method key
     * @param args Arguments to pass to the method (optional)
     * @see HotPatcher#get
     * @returns The output of the called method
     */
    HotPatcher.prototype.execute = function (key) {
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        var method = this.get(key) || NOOP;
        return method.apply(void 0, args);
    };
    /**
     * Get a method for a key
     * @param key The method key
     * @returns Returns the requested function or null if the function
     * does not exist and the host is configured to return null (and not throw)
     * @throws {Error} Throws if the configuration specifies to throw and the method
     * does not exist
     * @throws {Error} Throws if the `getEmptyAction` value is invalid
     */
    HotPatcher.prototype.get = function (key) {
        var item = this.configuration.registry[key];
        if (!item) {
            switch (this.getEmptyAction) {
                case "null":
                    return null;
                case "throw":
                    throw new Error("Failed handling method request: No method provided for override: ".concat(key));
                default:
                    throw new Error("Failed handling request which resulted in an empty method: Invalid empty-action specified: ".concat(this.getEmptyAction));
            }
        }
        return functions_1.sequence.apply(void 0, item.methods);
    };
    /**
     * Check if a method has been patched
     * @param key The function key
     * @returns True if already patched
     */
    HotPatcher.prototype.isPatched = function (key) {
        return !!this.configuration.registry[key];
    };
    /**
     * Patch a method name
     * @param key The method key to patch
     * @param method The function to set
     * @param opts Patch options
     * @returns Returns self
     */
    HotPatcher.prototype.patch = function (key, method, opts) {
        if (opts === void 0) { opts = {}; }
        var _a = opts.chain, chain = _a === void 0 ? false : _a;
        if (this.configuration.registry[key] && this.configuration.registry[key].final) {
            throw new Error("Failed patching '".concat(key, "': Method marked as being final"));
        }
        if (typeof method !== "function") {
            throw new Error("Failed patching '".concat(key, "': Provided method is not a function"));
        }
        if (chain) {
            // Add new method to the chain
            if (!this.configuration.registry[key]) {
                // New key, create item
                this.configuration.registry[key] = createNewItem(method);
            }
            else {
                // Existing, push the method
                this.configuration.registry[key].methods.push(method);
            }
        }
        else {
            // Replace the original
            if (this.isPatched(key)) {
                var original = this.configuration.registry[key].original;
                this.configuration.registry[key] = Object.assign(createNewItem(method), {
                    original: original
                });
            }
            else {
                this.configuration.registry[key] = createNewItem(method);
            }
        }
        return this;
    };
    /**
     * Patch a method inline, execute it and return the value
     * Used for patching contents of functions. This method will not apply a patched
     * function if it has already been patched, allowing for external overrides to
     * function. It also means that the function is cached so that it is not
     * instantiated every time the outer function is invoked.
     * @param key The function key to use
     * @param method The function to patch (once, only if not patched)
     * @param args Arguments to pass to the function
     * @returns The output of the patched function
     * @example
     *  function mySpecialFunction(a, b) {
     *      return hotPatcher.patchInline("func", (a, b) => {
     *          return a + b;
     *      }, a, b);
     *  }
     */
    HotPatcher.prototype.patchInline = function (key, method) {
        var args = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            args[_i - 2] = arguments[_i];
        }
        if (!this.isPatched(key)) {
            this.patch(key, method);
        }
        return this.execute.apply(this, __spreadArray([key], args, false));
    };
    /**
     * Patch a method (or methods) in sequential-mode
     * See `patch()` with the option `chain: true`
     * @see patch
     * @param key The key to patch
     * @param methods The methods to patch
     * @returns Returns self
     */
    HotPatcher.prototype.plugin = function (key) {
        var _this = this;
        var methods = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            methods[_i - 1] = arguments[_i];
        }
        methods.forEach(function (method) {
            _this.patch(key, method, { chain: true });
        });
        return this;
    };
    /**
     * Restore a patched method if it has been overridden
     * @param key The method key
     * @returns Returns self
     */
    HotPatcher.prototype.restore = function (key) {
        if (!this.isPatched(key)) {
            throw new Error("Failed restoring method: No method present for key: ".concat(key));
        }
        else if (typeof this.configuration.registry[key].original !== "function") {
            throw new Error("Failed restoring method: Original method not found or of invalid type for key: ".concat(key));
        }
        this.configuration.registry[key].methods = [this.configuration.registry[key].original];
        return this;
    };
    /**
     * Set a method as being final
     * This sets a method as having been finally overridden. Attempts at overriding
     * again will fail with an error.
     * @param key The key to make final
     * @returns Returns self
     */
    HotPatcher.prototype.setFinal = function (key) {
        if (!this.configuration.registry.hasOwnProperty(key)) {
            throw new Error("Failed marking '".concat(key, "' as final: No method found for key"));
        }
        this.configuration.registry[key].final = true;
        return this;
    };
    return HotPatcher;
}());
exports.HotPatcher = HotPatcher;
