import { Emitter } from '@rocket.chat/emitter';
import { MediaSignalingSession, MediaCallWebRTCProcessor } from '@rocket.chat/media-signaling';
import { useSetting, useStream, useWriteStream } from '@rocket.chat/ui-contexts';
import { useEffect, useSyncExternalStore, useCallback } from 'react';
import { MediaCallLogger } from './MediaCallLogger';
import { useIceServers } from '../hooks/useIceServers';
const randomStringFactory = () => {
    if (!window.crypto) {
        return Math.random().toString(36).substring(2, 15);
    }
    return window.crypto.randomUUID();
};
const getSessionIdKey = (userId) => {
    return `rcx-media-session-id-${userId}`;
};
class MediaSessionStore extends Emitter {
    constructor() {
        super();
        this.sessionInstance = null;
        this.sendSignalFn = null;
        this._webrtcProcessorFactory = null;
    }
    change() {
        this.emit('change');
    }
    onChange(callback) {
        return this.on('change', callback);
    }
    webrtcProcessorFactory(config) {
        if (!this._webrtcProcessorFactory) {
            throw new Error('WebRTC processor factory not set');
        }
        return this._webrtcProcessorFactory(config);
    }
    sendSignal(signal) {
        if (this.sendSignalFn) {
            return this.sendSignalFn(signal);
        }
        console.warn('Media Call - Tried to send signal, but no sendSignalFn was set');
        return Promise.resolve();
    }
    getOldSessionId(userId) {
        if (!window.sessionStorage) {
            return undefined;
        }
        const key = getSessionIdKey(userId);
        const oldSessionId = window.sessionStorage.getItem(key);
        if (!oldSessionId) {
            return undefined;
        }
        window.sessionStorage.removeItem(key);
        return oldSessionId;
    }
    makeInstance(userId) {
        if (this.sessionInstance !== null) {
            this.sessionInstance.endSession();
            this.sessionInstance = null;
        }
        if (!this._webrtcProcessorFactory || !this.sendSignalFn) {
            return null;
        }
        this.sessionInstance = new MediaSignalingSession({
            userId,
            transport: (signal) => this.sendSignal(signal),
            processorFactories: {
                webrtc: (config) => this.webrtcProcessorFactory(config),
            },
            mediaStreamFactory: (...args) => navigator.mediaDevices.getUserMedia(...args),
            randomStringFactory,
            oldSessionId: this.getOldSessionId(userId),
            logger: new MediaCallLogger(),
        });
        if (window.sessionStorage) {
            window.sessionStorage.setItem(getSessionIdKey(userId), this.sessionInstance.sessionId);
        }
        this.change();
        return this.sessionInstance;
    }
    getInstance(userId) {
        var _a;
        if (!userId) {
            return null;
        }
        if (((_a = this.sessionInstance) === null || _a === void 0 ? void 0 : _a.userId) === userId) {
            return this.sessionInstance;
        }
        return this.makeInstance(userId);
    }
    setSendSignalFn(sendSignalFn) {
        this.sendSignalFn = sendSignalFn;
        this.change();
        return () => {
            this.sendSignalFn = null;
        };
    }
    setWebRTCProcessorFactory(factory) {
        this._webrtcProcessorFactory = factory;
        this.change();
    }
    processSignal(signal, userId) {
        if (!this.sessionInstance || this.sessionInstance.userId !== userId) {
            return;
        }
        this.sessionInstance.processSignal(signal);
    }
}
const mediaSession = new MediaSessionStore();
export const useMediaSessionInstance = (userId) => {
    const iceServers = useIceServers();
    const iceGatheringTimeout = useSetting('VoIP_TeamCollab_Ice_Gathering_Timeout', 5000);
    const notifyUserStream = useStream('notify-user');
    const writeStream = useWriteStream('notify-user');
    useEffect(() => {
        mediaSession.setWebRTCProcessorFactory((config) => new MediaCallWebRTCProcessor(Object.assign(Object.assign({}, config), { rtc: Object.assign(Object.assign({}, config.rtc), { iceServers }), iceGatheringTimeout })));
    }, [iceServers, iceGatheringTimeout]);
    useEffect(() => {
        // TODO: This stream is not typed.
        return mediaSession.setSendSignalFn((signal) => writeStream(`${userId}/media-calls`, JSON.stringify(signal)));
    }, [writeStream, userId]);
    useEffect(() => {
        if (!userId) {
            return;
        }
        const unsubNotification = notifyUserStream(`${userId}/media-signal`, (signal) => mediaSession.processSignal(signal, userId));
        return () => {
            unsubNotification();
        };
    }, [userId, notifyUserStream]);
    const instance = useSyncExternalStore(useCallback((callback) => {
        return mediaSession.onChange(callback);
    }, []), useCallback(() => {
        return mediaSession.getInstance(userId);
    }, [userId]));
    return instance !== null && instance !== void 0 ? instance : undefined;
};
//# sourceMappingURL=useMediaSessionInstance.js.map