import { LayoutBlockType } from '../blocks/LayoutBlockType';
import { TextObjectType } from '../blocks/TextObjectType';
import { isActionsBlockElement } from '../blocks/isActionsBlockElement';
import { isContextBlockElement } from '../blocks/isContextBlockElement';
import { isInputBlockElement } from '../blocks/isInputBlockElement';
import { isSectionBlockAccessoryElement } from '../blocks/isSectionBlockAccessoryElement';
import { isTextObject } from '../blocks/isTextObject';
import { isNotNull } from '../isNotNull';
import { BlockContext } from './BlockContext';
import { renderBlockElement } from './renderBlockElement';
import { renderLayoutBlock } from './renderLayoutBlock';
import { renderTextObject } from './renderTextObject';
import { resolveConditionalBlocks } from './resolveConditionalBlocks';
var SurfaceRenderer = /** @class */ (function () {
    function SurfaceRenderer(allowedLayoutBlockTypes) {
        var _this = this;
        this.isAllowedLayoutBlock = function (block) {
            return _this.allowedLayoutBlockTypes.has(block.type);
        };
        this.allowedLayoutBlockTypes = new Set(allowedLayoutBlockTypes);
    }
    SurfaceRenderer.prototype.render = function (blocks, conditions) {
        if (!Array.isArray(blocks)) {
            return [];
        }
        return blocks
            .flatMap(resolveConditionalBlocks(conditions))
            .filter(this.isAllowedLayoutBlock)
            .map(renderLayoutBlock(this))
            .filter(isNotNull);
    };
    SurfaceRenderer.prototype.renderTextObject = function (textObject, index, context) {
        return renderTextObject(this, context)(textObject, index);
    };
    SurfaceRenderer.prototype.renderActionsBlockElement = function (block, index) {
        if (this.allowedLayoutBlockTypes.has(LayoutBlockType.ACTIONS) === false && !isActionsBlockElement(block)) {
            return null;
        }
        return renderBlockElement(this, BlockContext.ACTION)(block, index);
    };
    /** @deprecated */
    SurfaceRenderer.prototype.renderActions = function (element, _context, _, index) {
        return this.renderActionsBlockElement(element, index);
    };
    SurfaceRenderer.prototype.renderContextBlockElement = function (block, index) {
        if (this.allowedLayoutBlockTypes.has(LayoutBlockType.CONTEXT) === false && !isContextBlockElement(block)) {
            return null;
        }
        if (isTextObject(block)) {
            return renderTextObject(this, BlockContext.CONTEXT)(block, index);
        }
        return renderBlockElement(this, BlockContext.CONTEXT)(block, index);
    };
    /** @deprecated */
    SurfaceRenderer.prototype.renderContext = function (element, _context, _, index) {
        return this.renderContextBlockElement(element, index);
    };
    SurfaceRenderer.prototype.renderInputBlockElement = function (block, index) {
        if (this.allowedLayoutBlockTypes.has(LayoutBlockType.INPUT) === false && !isInputBlockElement(block)) {
            return null;
        }
        return renderBlockElement(this, BlockContext.FORM)(block, index);
    };
    /** @deprecated */
    SurfaceRenderer.prototype.renderInputs = function (element, _context, _, index) {
        return this.renderInputBlockElement(element, index);
    };
    SurfaceRenderer.prototype.renderSectionAccessoryBlockElement = function (block, index) {
        if (this.allowedLayoutBlockTypes.has(LayoutBlockType.SECTION) === false && !isSectionBlockAccessoryElement(block)) {
            return null;
        }
        return renderBlockElement(this, BlockContext.SECTION)(block, index);
    };
    /** @deprecated */
    SurfaceRenderer.prototype.renderAccessories = function (element, _context, _, index) {
        return this.renderSectionAccessoryBlockElement(element, index);
    };
    /** @deprecated */
    SurfaceRenderer.prototype.plainText = function (element, context, index) {
        if (context === void 0) { context = BlockContext.NONE; }
        if (index === void 0) { index = 0; }
        return this[TextObjectType.PLAIN_TEXT](element, context, index);
    };
    /** @deprecated */
    SurfaceRenderer.prototype.text = function (textObject, context, index) {
        if (context === void 0) { context = BlockContext.NONE; }
        if (index === void 0) { index = 0; }
        switch (textObject.type) {
            case TextObjectType.PLAIN_TEXT:
                return this.plain_text(textObject, context, index);
            case TextObjectType.MRKDWN:
                return this.mrkdwn(textObject, context, index);
            default:
                return null;
        }
    };
    return SurfaceRenderer;
}());
export { SurfaceRenderer };
//# sourceMappingURL=SurfaceRenderer.js.map