import { useMemo } from 'react';
/**
 * useSafeRefCallback will call a cleanup function (returned from the passed callback)
 * if the passed callback is called multiple times (similar to useEffect, but in a callbackRef)
 *
 * @example
 *   const callback = useSafeRefCallback(
 *       useCallback(
 *           (node: T) => {
 *               if (!node) {
 *                   return;
 *               }
 *               node.addEventListener('click', listener);
 *               return () => {
 *                   node.removeEventListener('click', listener);
 *               };
 *           },
 *           [listener],
 *       ),
 *   );
 *
 */
export const useSafeRefCallback = (callback) => {
    const callbackRef = useMemo(() => {
        let _cleanup;
        return (node) => {
            if (typeof _cleanup === 'function') {
                _cleanup();
            }
            const cleanup = callback(node);
            _cleanup = cleanup || null;
        };
    }, [callback]);
    return callbackRef;
};
//# sourceMappingURL=useSafeRefCallback.js.map