var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { useEffectEvent } from '@rocket.chat/fuselage-hooks';
import { useMemo, useState } from 'react';
import StepsLinkedList from './lib/StepsLinkedList';
/**
 * Custom hook to manage the state and navigation of a wizard.
 * It provides methods to register steps, navigate between them, and manage their state.
 *
 * @param {UseWizardProps} props - The properties for the wizard.
 * @returns {WizardAPI} The API for managing the wizard state and navigation.
 */
export const useWizard = ({ steps: stepsMetadata }) => {
    const [steps] = useState(new StepsLinkedList(stepsMetadata));
    const [currentStep, setCurrentStep] = useState(steps.head);
    /**
     * Registers a new step in the wizard.
     * If a step with the same ID already exists, it updates the existing step.
     */
    const register = useEffectEvent((stepMetadata) => {
        const step = steps.append(stepMetadata);
        return () => steps.remove(step.id);
    });
    /**
     * Navigates to a specific step in the wizard.
     * If the step is disabled, it does nothing.
     *
     * @param {StepNode} step - The step to navigate to.
     */
    const goTo = useEffectEvent((step) => __awaiter(void 0, void 0, void 0, function* () {
        if (step.disabled) {
            return;
        }
        setCurrentStep(step);
    }));
    /**
     * Navigates to the next step in the wizard.
     * If there is no next step, it does nothing.
     */
    const next = useEffectEvent(() => __awaiter(void 0, void 0, void 0, function* () {
        if (!(currentStep === null || currentStep === void 0 ? void 0 : currentStep.next)) {
            return;
        }
        steps.enableStep(currentStep.next);
        goTo(currentStep.next);
    }));
    /**
     * Navigates to the previous step in the wizard.
     * If there is no previous step, it does nothing.
     */
    const previous = useEffectEvent(() => __awaiter(void 0, void 0, void 0, function* () {
        if (!(currentStep === null || currentStep === void 0 ? void 0 : currentStep.prev)) {
            return;
        }
        steps.enableStep(currentStep.prev);
        goTo(currentStep.prev);
    }));
    /**
     * Resets the next steps in the wizard.
     * It disables all steps that come after the current step.
     */
    const resetNextSteps = useEffectEvent(() => {
        if (!currentStep) {
            return;
        }
        let step = currentStep;
        while (step.next) {
            steps.disableStep(step.next);
            step = step.next;
        }
    });
    return useMemo(() => ({
        steps,
        register,
        currentStep,
        next,
        previous,
        goTo,
        resetNextSteps,
    }), [currentStep, goTo, next, previous, register, steps, resetNextSteps]);
};
//# sourceMappingURL=useWizard.js.map