"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PresenceReaper = void 0;
const node_timers_1 = require("node:timers");
const models_1 = require("@rocket.chat/models");
const isNonEmptyArray = (arr) => arr.length > 0;
class PresenceReaper {
    constructor(options) {
        this.onUpdate = options.onUpdate;
        this.staleThresholdMs = options.staleThresholdMs;
        this.batchSize = options.batchSize;
        this.running = false;
    }
    start() {
        if (this.running)
            return;
        this.running = true;
        // Run every 1 minute
        this.intervalId = (0, node_timers_1.setInterval)(() => {
            this.run().catch((err) => console.error('[PresenceReaper] Error:', err));
        }, 60 * 1000);
    }
    stop() {
        if (!this.running)
            return;
        this.running = false;
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = undefined;
        }
    }
    async run() {
        const cutoffDate = new Date(Date.now() - this.staleThresholdMs);
        // 1. Find users with potentially stale connections
        const cursor = models_1.UsersSessions.find({ 'connections._updatedAt': { $lte: cutoffDate } }, {
            projection: { _id: 1, connections: 1 },
        });
        const userChangeSet = new Map();
        for await (const sessionDoc of cursor) {
            this.processDocument(sessionDoc, cutoffDate, userChangeSet);
            if (userChangeSet.size >= this.batchSize) {
                await this.flushBatch(userChangeSet);
                userChangeSet.clear();
            }
        }
        if (userChangeSet.size > 0) {
            await this.flushBatch(userChangeSet);
        }
    }
    processDocument(sessionDoc, cutoffDate, changeMap) {
        const userId = sessionDoc._id;
        const allConnections = sessionDoc.connections || [];
        // Filter connections based on the cutoff
        const staleConnections = allConnections.filter((c) => c._updatedAt <= cutoffDate);
        if (isNonEmptyArray(staleConnections)) {
            changeMap.set(userId, {
                userId,
                removeIds: staleConnections.map((c) => c.id),
                cutoffDate, // Keep reference for race-condition check
            });
        }
    }
    async flushBatch(changeMap) {
        const operations = [];
        for (const plan of changeMap.values()) {
            operations.push({
                updateOne: {
                    filter: { _id: plan.userId },
                    update: {
                        $pull: {
                            connections: {
                                id: { $in: plan.removeIds },
                                _updatedAt: { $lte: plan.cutoffDate },
                            },
                        },
                    },
                },
            });
        }
        if (isNonEmptyArray(operations)) {
            await models_1.UsersSessions.col.bulkWrite(operations);
            await this.onUpdate(operations.map((op) => op.updateOne.filter._id));
        }
    }
}
exports.PresenceReaper = PresenceReaper;
//# sourceMappingURL=PresenceReaper.js.map