"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OmnichannelTranscript = void 0;
const core_services_1 = require("@rocket.chat/core-services");
const core_typings_1 = require("@rocket.chat/core-typings");
const message_parser_1 = require("@rocket.chat/message-parser");
const message_types_1 = require("@rocket.chat/message-types");
const models_1 = require("@rocket.chat/models");
const pdf_worker_1 = require("@rocket.chat/pdf-worker");
const tools_1 = require("@rocket.chat/tools");
const localTypes_1 = require("./localTypes");
class OmnichannelTranscript extends core_services_1.ServiceClass {
    constructor(loggerConstructor, 
    // Instance of i18n. Should already be init'd and loaded with the translation files
    translator) {
        super();
        this.translator = translator;
        this.name = 'omnichannel-transcript';
        this.maxNumberOfConcurrentJobs = 25;
        this.currentJobNumber = 0;
        this.worker = new pdf_worker_1.PdfWorker('chat-transcript');
        // eslint-disable-next-line new-cap
        this.log = new loggerConstructor('OmnichannelTranscript');
    }
    async getTimezone(agent) {
        const reportingTimezone = await core_services_1.Settings.get('Default_Timezone_For_Reporting');
        switch (reportingTimezone) {
            case 'custom':
                return core_services_1.Settings.get('Default_Custom_Timezone');
            case 'user':
                if (agent?.utcOffset) {
                    return (0, tools_1.guessTimezoneFromOffset)(agent.utcOffset);
                }
                return (0, tools_1.guessTimezone)();
            default:
                return (0, tools_1.guessTimezone)();
        }
    }
    async getMessagesFromRoom({ rid }) {
        const showSystemMessages = await core_services_1.Settings.get('Livechat_transcript_show_system_messages');
        // Closing message should not appear :)
        return models_1.Messages.findLivechatMessagesWithoutTypes(rid, ['command'], showSystemMessages, {
            sort: { ts: 1 },
            projection: {
                _id: 1,
                msg: 1,
                u: 1,
                t: 1,
                ts: 1,
                attachments: 1,
                files: 1,
                md: 1,
                navigation: 1,
                requestData: 1,
                transferData: 1,
                webRtcCallEndTs: 1,
                comment: 1,
                priorityData: 1,
                slaData: 1,
                rid: 1,
            },
        }).toArray();
    }
    getQuotesFromMessage(message) {
        const quotes = [];
        if (!message.attachments) {
            return quotes;
        }
        for (const attachment of message.attachments) {
            if (!(0, core_typings_1.isQuoteAttachment)(attachment)) {
                continue;
            }
            const { text, author_name: name, md, ts } = attachment;
            if (text) {
                quotes.push({
                    name,
                    md: md ?? (0, message_parser_1.parse)(text),
                    ts,
                });
            }
            quotes.push(...this.getQuotesFromMessage({ attachments: attachment.attachments }));
        }
        return quotes;
    }
    getSystemMessage(message, t) {
        if (!message.t)
            return undefined;
        const systemMessageDefinition = message_types_1.MessageTypes.getType(message);
        if (!systemMessageDefinition)
            return undefined;
        return {
            ...message,
            msg: systemMessageDefinition.text(t, message),
        };
    }
    async getMessagesData(messages, t) {
        const messagesData = [];
        for await (const message of messages) {
            const systemMessage = this.getSystemMessage(message, t);
            if (systemMessage) {
                messagesData.push({
                    ...systemMessage,
                    files: systemMessage.files ?? [],
                    quotes: systemMessage.quotes ?? [],
                });
                continue;
            }
            if (!message.attachments?.length) {
                // If there's no attachment and no message, what was sent? lol
                messagesData.push({
                    ...message,
                    files: [],
                    quotes: [],
                });
                continue;
            }
            const files = [];
            const quotes = [];
            for await (const attachment of message.attachments) {
                if ((0, core_typings_1.isQuoteAttachment)(attachment)) {
                    quotes.push(...this.getQuotesFromMessage(message));
                    continue;
                }
                if (!(0, core_typings_1.isFileAttachment)(attachment)) {
                    this.log.error(`Invalid attachment type ${attachment.type} for file ${attachment.title} in room ${message.rid}!`);
                    // ignore other types of attachments
                    continue;
                }
                if (!(0, core_typings_1.isFileImageAttachment)(attachment)) {
                    this.log.error(`Invalid attachment type ${attachment.type} for file ${attachment.title} in room ${message.rid}!`);
                    // ignore other types of attachments
                    files.push({ name: attachment.title });
                    continue;
                }
                if (!this.worker.isMimeTypeValid(attachment.image_type)) {
                    this.log.error(`Invalid mime type ${attachment.image_type} for file ${attachment.title} in room ${message.rid}!`);
                    // ignore invalid mime types
                    files.push({ name: attachment.title });
                    continue;
                }
                let file = message.files?.map((v) => ({ _id: v._id, name: v.name })).find((file) => file.name === attachment.title);
                if (!file) {
                    this.log.warn(`File ${attachment.title} not found in room ${message.rid}!`);
                    // For some reason, when an image is uploaded from clipboard, it doesn't have a file :(
                    // So, we'll try to get the FILE_ID from the `title_link` prop which has the format `/file-upload/FILE_ID/FILE_NAME` using a regex
                    const fileId = attachment.title_link?.match(/\/file-upload\/(.*)\/.*/)?.[1];
                    if (!fileId) {
                        this.log.error(`File ${attachment.title} not found in room ${message.rid}!`);
                        // ignore attachments without file
                        files.push({ name: attachment.title });
                        continue;
                    }
                    file = { _id: fileId, name: attachment.title || 'upload' };
                }
                if (!file) {
                    this.log.warn(`File ${attachment.title} not found in room ${message.rid}!`);
                    // ignore attachments without file
                    files.push({ name: attachment.title });
                    continue;
                }
                const uploadedFile = await models_1.Uploads.findOneById(file._id);
                if (!uploadedFile) {
                    this.log.error(`Uploaded file ${file._id} not found in room ${message.rid}!`);
                    // ignore attachments without file
                    files.push({ name: file.name });
                    continue;
                }
                try {
                    const fileBuffer = await core_services_1.Upload.getFileBuffer({ file: uploadedFile });
                    files.push({ name: file.name, buffer: fileBuffer, extension: uploadedFile.extension });
                }
                catch (e) {
                    this.log.error(`Failed to get file ${file._id}`, e);
                    // Push empty buffer so parser processes this as "unsupported file"
                    files.push({ name: file.name });
                    // TODO: this is a NATS error message, even when we shouldn't tie it, since it's the only way we have right now we'll live with it for a while
                    if (e.message === 'MAX_PAYLOAD_EXCEEDED') {
                        this.log.error(`File is too big to be processed by NATS. See NATS config for allowing bigger messages to be sent between services`);
                    }
                }
            }
            // When you send a file message, the things you type in the modal are not "msg", they're in "description" of the attachment
            // So, we'll fetch the the msg, if empty, go for the first description on an attachment, if empty, empty string
            const msg = message.msg || message.attachments.find((attachment) => attachment.description)?.description || '';
            // Remove nulls from final array
            messagesData.push({
                msg,
                u: message.u,
                files,
                quotes,
                ts: message.ts,
                md: message.md,
            });
        }
        return messagesData;
    }
    async workOnPdf({ details }) {
        this.log.info(`Processing transcript for room ${details.rid} by user ${details.userId} - Received from queue`);
        if (this.maxNumberOfConcurrentJobs <= this.currentJobNumber) {
            this.log.error(`Processing transcript for room ${details.rid} by user ${details.userId} - Too many concurrent jobs, queuing again`);
            throw new Error('retry');
        }
        this.currentJobNumber++;
        // TODO: cache these with mem
        const [siteName, dateFormat, timeAndDateFormat, serverLanguage] = await Promise.all([
            core_services_1.Settings.get('Site_Name'),
            core_services_1.Settings.get('Message_DateFormat'),
            core_services_1.Settings.get('Message_TimeAndDateFormat'),
            core_services_1.Settings.get('Language'),
        ]);
        const user = await models_1.Users.findOneById(details.userId, { projection: { _id: 1, language: 1 } });
        if (!user)
            return;
        const language = user.language ?? serverLanguage;
        const i18n = this.translator.cloneInstance({ lng: language });
        try {
            const room = await models_1.LivechatRooms.findOneById(details.rid, {
                projection: { v: 1, servedBy: 1, pdfTranscriptFileId: 1, closedAt: 1 },
            });
            if (!room) {
                throw new Error('room-not-found');
            }
            if (room.pdfTranscriptFileId) {
                this.log.info(`Processing transcript for room ${details.rid} by user ${details.userId} - PDF already exists`);
                return;
            }
            const messages = await this.getMessagesFromRoom({ rid: room._id });
            const visitor = room.v
                ? await models_1.LivechatVisitors.findOneEnabledById(room.v._id, {
                    projection: { _id: 1, name: 1, username: 1, visitorEmails: 1 },
                })
                : null;
            const agent = room.servedBy
                ? await models_1.Users.findOneAgentById(room.servedBy._id, {
                    projection: { _id: 1, name: 1, username: 1, utcOffset: 1 },
                })
                : null;
            const messagesData = await this.getMessagesData(messages, i18n.t);
            const timezone = await this.getTimezone(agent);
            this.log.info({ msg: 'Loading translations', language });
            const data = {
                visitor,
                agent,
                closedAt: room.closedAt,
                siteName,
                messages: messagesData,
                dateFormat,
                timeAndDateFormat,
                timezone,
            };
            await this.doRender({ data, details, i18n });
        }
        catch (error) {
            await this.pdfFailed({ details, e: error, i18n });
        }
        finally {
            this.currentJobNumber--;
        }
    }
    async doRender({ data, details, i18n }) {
        const transcriptText = i18n.t('Transcript');
        const stream = await this.worker.renderToStream({ data, i18n });
        const outBuff = await (0, tools_1.streamToBuffer)(stream);
        try {
            const { rid } = await core_services_1.Room.createDirectMessage({ to: details.userId, from: 'rocket.cat' });
            const [rocketCatFile, transcriptFile] = await this.uploadFiles({
                details,
                buffer: outBuff,
                roomIds: [rid, details.rid],
                data,
                transcriptText,
            });
            await this.pdfComplete({ details, transcriptFile, rocketCatFile, i18n });
        }
        catch (error) {
            this.pdfFailed({ details, e: error, i18n });
        }
    }
    async pdfFailed({ details, e, i18n }) {
        this.log.error(`Transcript for room ${details.rid} by user ${details.userId} - Failed: ${e.message}`);
        const room = await models_1.LivechatRooms.findOneById(details.rid, { projection: { _id: 1 } });
        if (!room) {
            return;
        }
        const { rid } = await core_services_1.Room.createDirectMessage({ to: details.userId, from: 'rocket.cat' });
        this.log.info(`Transcript for room ${details.rid} by user ${details.userId} - Sending error message to user`);
        await core_services_1.Message.sendMessage({
            fromId: 'rocket.cat',
            rid,
            msg: `${i18n.t('pdf_error_message')}: ${e.message}`,
        });
    }
    async uploadFiles({ details, buffer, roomIds, data, transcriptText, }) {
        return Promise.all(roomIds.map((roomId) => {
            return core_services_1.Upload.uploadFile({
                userId: details.userId,
                buffer,
                details: {
                    // transcript_{company-name}_{date}_{hour}.pdf
                    name: `${transcriptText}_${data.siteName}_${new Intl.DateTimeFormat('en-US').format(new Date()).replace(/\//g, '-')}_${data.visitor?.name || data.visitor?.username || 'Visitor'}.pdf`,
                    type: 'application/pdf',
                    rid: roomId,
                    // Rocket.cat is the goat
                    userId: 'rocket.cat',
                    size: buffer.length,
                },
            });
        }));
    }
    async pdfComplete({ details, transcriptFile, rocketCatFile, i18n, }) {
        this.log.info(`Transcript for room ${details.rid} by user ${details.userId} - Complete`);
        // Send the file to the livechat room where this was requested, to keep it in context
        try {
            await models_1.LivechatRooms.setPdfTranscriptFileIdById(details.rid, transcriptFile._id);
            this.log.info(`Transcript for room ${details.rid} by user ${details.userId} - Sending success message to user`);
            const result = await Promise.allSettled([
                core_services_1.Upload.sendFileMessage({
                    roomId: details.rid,
                    userId: 'rocket.cat',
                    file: transcriptFile,
                    message: {
                        // Translate from service
                        msg: i18n.t('pdf_success_message'),
                    },
                }),
                // Send the file to the user who requested it, so they can download it
                core_services_1.Upload.sendFileMessage({
                    roomId: rocketCatFile.rid || '',
                    userId: 'rocket.cat',
                    file: rocketCatFile,
                    message: {
                        // Translate from service
                        msg: i18n.t('pdf_success_message'),
                    },
                }),
            ]);
            const e = result.find((r) => (0, localTypes_1.isPromiseRejectedResult)(r));
            if (e && (0, localTypes_1.isPromiseRejectedResult)(e)) {
                throw e.reason;
            }
        }
        catch (err) {
            this.log.error({ msg: `Transcript for room ${details.rid} by user ${details.userId} - Failed to send message`, err });
        }
    }
}
exports.OmnichannelTranscript = OmnichannelTranscript;
//# sourceMappingURL=OmnichannelTranscript.js.map