import { Authorization } from '@rocket.chat/core-services';
import { LivechatUnit, LivechatDepartmentAgents } from '@rocket.chat/models';
import mem from 'mem';

import { defaultLogger } from '../utils/logger';

async function getUnitsFromUserRoles(user: string): Promise<string[]> {
	return LivechatUnit.findByMonitorId(user);
}

async function getDepartmentsFromUserRoles(user: string): Promise<string[]> {
	return (await LivechatDepartmentAgents.findByAgentId(user).toArray()).map((department) => department.departmentId);
}

const memoizedGetUnitFromUserRoles = mem(getUnitsFromUserRoles, { maxAge: process.env.TEST_MODE ? 1 : 10000 });
const memoizedGetDepartmentsFromUserRoles = mem(getDepartmentsFromUserRoles, { maxAge: process.env.TEST_MODE ? 1 : 10000 });

async function hasUnits(): Promise<boolean> {
	// @ts-expect-error - this prop is injected dynamically on ee license
	return (await LivechatUnit.countUnits({ type: 'u' })) > 0;
}

// Units should't change really often, so we can cache the result
const memoizedHasUnits = mem(hasUnits, { maxAge: process.env.TEST_MODE ? 1 : 10000 });

export const getUnitsFromUser = async (userId?: string): Promise<string[] | undefined> => {
	if (!userId) {
		return;
	}

	if (!(await memoizedHasUnits())) {
		return;
	}

	// TODO: we can combine these 2 calls into one single query
	if (await Authorization.hasAnyRole(userId, ['admin', 'livechat-manager'])) {
		return;
	}

	if (!(await Authorization.hasAnyRole(userId, ['livechat-monitor', 'livechat-agent']))) {
		return;
	}

	const unitsAndDepartments = [...(await memoizedGetUnitFromUserRoles(userId)), ...(await memoizedGetDepartmentsFromUserRoles(userId))];
	defaultLogger.debug({ msg: 'Calculating units for monitor', user: userId, unitsAndDepartments });

	return unitsAndDepartments;
};
