"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.startBroker = startBroker;
const core_services_1 = require("@rocket.chat/core-services");
const ejson_1 = __importDefault(require("ejson"));
const moleculer_1 = require("moleculer");
const pino_1 = require("pino");
const NetworkBroker_1 = require("./NetworkBroker");
const { MS_NAMESPACE = '', TRANSPORTER = '', CACHE = 'Memory', 
// SERIALIZER = 'MsgPack',
SERIALIZER = 'EJSON', MOLECULER_LOG_LEVEL = 'warn', BALANCE_STRATEGY = 'RoundRobin', BALANCE_PREFER_LOCAL = 'true', RETRY_FACTOR = '2', RETRY_MAX_DELAY = '1000', RETRY_DELAY = '100', RETRY_RETRIES = '5', RETRY_ENABLED = 'false', REQUEST_TIMEOUT = '60', HEARTBEAT_INTERVAL = '10', HEARTBEAT_TIMEOUT = '30', BULKHEAD_ENABLED = 'false', BULKHEAD_CONCURRENCY = '10', BULKHEAD_MAX_QUEUE_SIZE = '10000', MS_METRICS = 'false', MS_METRICS_PORT = '9458', SKIP_PROCESS_EVENT_REGISTRATION = 'false', } = process.env;
const { Base } = moleculer_1.Serializers;
class CustomRegenerator extends moleculer_1.Errors.Regenerator {
    restoreCustomError(plainError) {
        const { message, reason, details, errorType, isClientSafe } = plainError;
        if (errorType === 'Meteor.Error') {
            const error = new core_services_1.MeteorError(message, reason, details);
            if (typeof isClientSafe !== 'undefined') {
                error.isClientSafe = isClientSafe;
            }
            return error;
        }
        return undefined;
    }
    extractPlainError(err) {
        return {
            ...super.extractPlainError(err),
            ...((0, core_services_1.isMeteorError)(err) && {
                isClientSafe: err.isClientSafe,
                errorType: err.errorType,
                reason: err.reason,
                details: err.details,
            }),
        };
    }
}
class EJSONSerializer extends Base {
    serialize(obj) {
        return Buffer.from(ejson_1.default.stringify(obj));
    }
    deserialize(buf) {
        return ejson_1.default.parse(buf.toString());
    }
}
function startBroker(options = {}) {
    const network = new moleculer_1.ServiceBroker({
        namespace: MS_NAMESPACE,
        skipProcessEventRegistration: SKIP_PROCESS_EVENT_REGISTRATION === 'true',
        transporter: TRANSPORTER,
        metrics: {
            enabled: MS_METRICS === 'true',
            reporter: [
                {
                    type: 'Prometheus',
                    options: {
                        port: MS_METRICS_PORT,
                    },
                },
            ],
        },
        cacher: CACHE,
        serializer: SERIALIZER === 'EJSON' ? new EJSONSerializer() : SERIALIZER,
        logger: {
            type: 'Pino',
            options: {
                level: MOLECULER_LOG_LEVEL,
                pino: {
                    options: {
                        timestamp: pino_1.pino.stdTimeFunctions.isoTime,
                        ...(process.env.NODE_ENV !== 'production'
                            ? {
                                transport: {
                                    target: 'pino-pretty',
                                    options: {
                                        colorize: true,
                                    },
                                },
                            }
                            : {}),
                    },
                },
            },
        },
        registry: {
            strategy: BALANCE_STRATEGY,
            preferLocal: BALANCE_PREFER_LOCAL !== 'false',
        },
        requestTimeout: parseInt(REQUEST_TIMEOUT) * 1000,
        retryPolicy: {
            enabled: RETRY_ENABLED === 'true',
            retries: parseInt(RETRY_RETRIES),
            delay: parseInt(RETRY_DELAY),
            maxDelay: parseInt(RETRY_MAX_DELAY),
            factor: parseInt(RETRY_FACTOR),
            check: (err) => err && !!err.retryable,
        },
        maxCallLevel: 100,
        heartbeatInterval: parseInt(HEARTBEAT_INTERVAL),
        heartbeatTimeout: parseInt(HEARTBEAT_TIMEOUT),
        // circuitBreaker: {
        // 	enabled: false,
        // 	threshold: 0.5,
        // 	windowTime: 60,
        // 	minRequestCount: 20,
        // 	halfOpenTime: 10 * 1000,
        // 	check: (err: any): boolean => err && err.code >= 500,
        // },
        bulkhead: {
            enabled: BULKHEAD_ENABLED === 'true',
            concurrency: parseInt(BULKHEAD_CONCURRENCY),
            maxQueueSize: parseInt(BULKHEAD_MAX_QUEUE_SIZE),
        },
        errorRegenerator: new CustomRegenerator(),
        started() {
            console.log('NetworkBroker started successfully.');
        },
        ...options,
    });
    return new NetworkBroker_1.NetworkBroker(network);
}
//# sourceMappingURL=index.js.map