"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkBroker = void 0;
const core_services_1 = require("@rocket.chat/core-services");
const tracing_1 = require("@rocket.chat/tracing");
const EnterpriseCheck_1 = require("./EnterpriseCheck");
const events = {
    onNodeConnected: '$node.connected',
    onNodeUpdated: '$node.updated',
    onNodeDisconnected: '$node.disconnected',
};
const lifecycle = {
    created: 'created',
    started: 'started',
    stopped: 'stopped',
};
class NetworkBroker {
    constructor(broker) {
        this.started = Promise.resolve(false);
        this.defaultDependencies = ['settings', 'license'];
        this.broker = broker;
        this.metrics = broker.metrics;
    }
    async call(method, data, options) {
        if (!(await this.started)) {
            return;
        }
        const context = core_services_1.asyncLocalStorage.getStore();
        if (context?.ctx?.call) {
            return context.ctx.call(method, data, options);
        }
        const services = await this.broker.call('$node.services', {
            onlyAvailable: true,
        });
        if (!services.find((service) => service.name === method.split('.')[0])) {
            return new Error('method-not-available');
        }
        return this.broker.call(method, data, {
            ...options,
            meta: {
                optl: (0, tracing_1.injectCurrentContext)(),
            },
        });
    }
    async destroyService(instance) {
        const name = instance.getName();
        if (!name) {
            return;
        }
        await this.broker.destroyService(name);
        instance.removeAllListeners();
    }
    createService(instance, serviceDependencies = []) {
        const methods = (instance.constructor?.name === 'Object'
            ? Object.getOwnPropertyNames(instance)
            : Object.getOwnPropertyNames(Object.getPrototypeOf(instance))).filter((name) => name !== 'constructor');
        const serviceInstance = instance;
        const name = instance.getName();
        if (!name) {
            return;
        }
        const instanceEvents = instance.getEvents();
        if (!instanceEvents && !methods.length) {
            return;
        }
        const dependencies = [...serviceDependencies, ...(name === 'settings' ? [] : this.defaultDependencies)].filter((dependency) => dependency !== name);
        const service = {
            name,
            actions: {},
            mixins: !instance.isInternal() ? [EnterpriseCheck_1.EnterpriseCheck] : [],
            ...(dependencies.length ? { dependencies } : {}),
            events: instanceEvents.reduce((map, { eventName }) => {
                map[eventName] = /^\$/.test(eventName)
                    ? (ctx) => {
                        // internal events params are not an array
                        instance.emit(eventName, ctx.params);
                    }
                    : (ctx) => {
                        instance.emit(eventName, ...ctx.params);
                    };
                return map;
            }, {}),
        };
        if (!service.events || !service.actions) {
            return;
        }
        for (const method of methods) {
            if (method.match(/^on[A-Z]/)) {
                service.events[events[method]] = serviceInstance[method].bind(serviceInstance);
                continue;
            }
            if (lifecycle[method]) {
                service[method] = () => {
                    core_services_1.asyncLocalStorage.run({
                        id: '',
                        nodeID: this.broker.nodeID,
                        requestID: null,
                        broker: this,
                    }, serviceInstance[method].bind(serviceInstance));
                };
                continue;
            }
            service.actions[method] = async (ctx) => {
                return (0, tracing_1.tracerSpan)(`action ${name}:${method}`, {}, () => {
                    return core_services_1.asyncLocalStorage.run({
                        id: ctx.id,
                        nodeID: ctx.nodeID,
                        requestID: ctx.requestID,
                        broker: this,
                        ctx,
                    }, () => serviceInstance[method](...ctx.params));
                }, ctx.meta?.optl);
            };
        }
        this.broker.createService(service);
    }
    async broadcast(event, ...args) {
        return this.broker.broadcast(event, args);
    }
    async broadcastLocal(event, ...args) {
        void this.broker.broadcastLocal(event, args);
    }
    async broadcastToServices(services, event, ...args) {
        void this.broker.broadcast(event, args, services);
    }
    async nodeList() {
        return this.broker.call('$node.list');
    }
    async start() {
        await this.broker.start();
        this.started = Promise.resolve(true);
    }
}
exports.NetworkBroker = NetworkBroker;
//# sourceMappingURL=NetworkBroker.js.map