"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VideoConferenceRaw = void 0;
const core_typings_1 = require("@rocket.chat/core-typings");
const BaseRaw_1 = require("./BaseRaw");
class VideoConferenceRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'video_conference', trash);
    }
    modelIndexes() {
        return [
            { key: { rid: 1, createdAt: 1 }, unique: false },
            { key: { type: 1, status: 1 }, unique: false },
            { key: { discussionRid: 1 }, unique: false },
        ];
    }
    findPaginatedByRoomId(rid, { offset, count } = {}) {
        return this.findPaginated({ rid }, {
            sort: { createdAt: -1 },
            skip: offset,
            limit: count,
            projection: {
                providerData: 0,
            },
        });
    }
    async findAllLongRunning(minDate) {
        return this.find({
            createdAt: {
                $lte: minDate,
            },
            endedAt: {
                $exists: false,
            },
        }, {
            projection: {
                _id: 1,
            },
        });
    }
    async countByTypeAndStatus(type, status, options) {
        return this.countDocuments({
            type,
            status,
        }, options);
    }
    async createDirect({ providerName, ...callDetails }) {
        const call = {
            type: 'direct',
            users: [],
            messages: {},
            status: core_typings_1.VideoConferenceStatus.CALLING,
            createdAt: new Date(),
            providerName: providerName.toLowerCase(),
            ringing: true,
            ...callDetails,
        };
        return (await this.insertOne(call)).insertedId;
    }
    async createGroup({ providerName, ...callDetails }) {
        const call = {
            type: 'videoconference',
            users: [],
            messages: {},
            status: core_typings_1.VideoConferenceStatus.STARTED,
            anonymousUsers: 0,
            createdAt: new Date(),
            providerName: providerName.toLowerCase(),
            ...callDetails,
        };
        return (await this.insertOne(call)).insertedId;
    }
    async createLivechat({ providerName, ...callDetails }) {
        const call = {
            type: 'livechat',
            users: [],
            messages: {},
            status: core_typings_1.VideoConferenceStatus.STARTED,
            createdAt: new Date(),
            providerName: providerName.toLowerCase(),
            ...callDetails,
        };
        return (await this.insertOne(call)).insertedId;
    }
    async createVoIP(call) {
        const { externalId, ...data } = call;
        const doc = await this.findOneAndUpdate({ externalId }, { $set: data }, { upsert: true, returnDocument: 'after' });
        return doc?._id;
    }
    updateOneById(_id, update, options) {
        return this.updateOne({ _id }, update, options);
    }
    async setEndedById(callId, endedBy, endedAt) {
        await this.updateOneById(callId, {
            $set: {
                endedBy,
                endedAt: endedAt || new Date(),
            },
        });
    }
    async setDataById(callId, data) {
        await this.updateOneById(callId, {
            $set: data,
        });
    }
    async setRingingById(callId, ringing) {
        await this.updateOneById(callId, {
            $set: {
                ringing,
            },
        });
    }
    async setStatusById(callId, status) {
        await this.updateOneById(callId, {
            $set: {
                status,
            },
        });
    }
    async setUrlById(callId, url) {
        await this.updateOneById(callId, {
            $set: {
                url,
            },
        });
    }
    async setProviderDataById(callId, providerData) {
        await this.updateOneById(callId, {
            ...(providerData
                ? {
                    $set: {
                        providerData,
                    },
                }
                : {
                    $unset: {
                        providerData: 1,
                    },
                }),
        });
    }
    async addUserById(callId, user) {
        await this.updateOneById(callId, {
            $addToSet: {
                users: {
                    _id: user._id,
                    username: user.username,
                    name: user.name,
                    avatarETag: user.avatarETag,
                    ts: user.ts || new Date(),
                },
            },
        });
    }
    async setMessageById(callId, messageType, messageId) {
        await this.updateOneById(callId, {
            $set: {
                [`messages.${messageType}`]: messageId,
            },
        }); // TODO: Remove this cast when TypeScript is updated
        // TypeScript is not smart enough to infer that `messages.${'start' | 'end'}` matches two keys of `VideoConference`
    }
    async updateUserReferences(userId, username, name) {
        await this.updateMany({
            'users._id': userId,
        }, {
            $set: {
                'users.$.name': name,
                'users.$.username': username,
            },
        });
        await this.updateMany({
            'createdBy._id': userId,
        }, {
            $set: {
                'createdBy.name': name,
                'createdBy.username': username,
            },
        });
        await this.updateMany({
            'endedBy._id': userId,
        }, {
            $set: {
                'endedBy.name': name,
                'endedBy.username': username,
            },
        });
    }
    async increaseAnonymousCount(callId) {
        await this.updateOne({ _id: callId }, {
            $inc: {
                anonymousUsers: 1,
            },
        });
    }
    async setDiscussionRidById(callId, discussionRid) {
        await this.updateOne({ _id: callId }, { $set: { discussionRid } });
    }
    async unsetDiscussionRidById(callId) {
        await this.updateOne({ _id: callId }, { $unset: { discussionRid: true } });
    }
    async unsetDiscussionRid(discussionRid) {
        await this.updateMany({
            discussionRid,
        }, {
            $unset: {
                discussionRid: 1,
            },
        });
    }
}
exports.VideoConferenceRaw = VideoConferenceRaw;
//# sourceMappingURL=VideoConference.js.map