"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RolesRaw = void 0;
const index_1 = require("../index");
const BaseRaw_1 = require("./BaseRaw");
class RolesRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'roles', trash);
    }
    findByUpdatedDate(updatedAfterDate, options) {
        const query = {
            _updatedAt: { $gte: new Date(updatedAfterDate) },
        };
        return options ? this.find(query, options) : this.find(query);
    }
    async isUserInRoles(userId, roles, scope) {
        if (process.env.NODE_ENV === 'development' && (scope === 'Users' || scope === 'Subscriptions')) {
            throw new Error('Roles.isUserInRoles method received a role scope instead of a scope value.');
        }
        for await (const roleId of roles) {
            const role = await this.findOneById(roleId, { projection: { scope: 1 } });
            if (!role) {
                continue;
            }
            switch (role.scope) {
                case 'Subscriptions':
                    if (await index_1.Subscriptions.isUserInRole(userId, roleId, scope)) {
                        return true;
                    }
                    break;
                case 'Users':
                default:
                    if (await index_1.Users.isUserInRole(userId, roleId)) {
                        return true;
                    }
            }
        }
        return false;
    }
    findOneByIdOrName(_idOrName, options) {
        const query = {
            $or: [
                {
                    _id: _idOrName,
                },
                {
                    name: _idOrName,
                },
            ],
        };
        return this.findOne(query, options);
    }
    async findOneByName(name, options) {
        const query = {
            name,
        };
        return this.findOne(query, options);
    }
    findInIds(ids, options) {
        const query = {
            _id: {
                $in: ids,
            },
        };
        return this.find(query, options || {});
    }
    findInIdsOrNames(_idsOrNames, options) {
        const query = {
            $or: [
                {
                    _id: {
                        $in: _idsOrNames,
                    },
                },
                {
                    name: {
                        $in: _idsOrNames,
                    },
                },
            ],
        };
        return this.find(query, options || {});
    }
    findAllExceptIds(ids, options) {
        const query = {
            _id: {
                $nin: ids,
            },
        };
        return this.find(query, options || {});
    }
    findByScope(scope, options) {
        const query = {
            scope,
        };
        return this.find(query, options || {});
    }
    countByScope(scope, options) {
        const query = {
            scope,
        };
        return this.countDocuments(query, options);
    }
    findCustomRoles(options) {
        const query = {
            protected: false,
        };
        return this.find(query, options || {});
    }
    countCustomRoles(options) {
        const query = {
            protected: false,
        };
        return this.countDocuments(query, options || {});
    }
    async updateById(_id, name, scope, description = '', mandatory2fa = false) {
        const response = await this.findOneAndUpdate({ _id }, {
            $set: {
                name,
                scope,
                description,
                mandatory2fa,
            },
        }, { upsert: true, returnDocument: 'after' });
        if (!response) {
            console.log('Role not found', _id, name);
            throw new Error('Role not found');
        }
        return response;
    }
    /** @deprecated function getUsersInRole should be used instead */
    async findUsersInRole(roleId, scope, options) {
        if (process.env.NODE_ENV === 'development' && (scope === 'Users' || scope === 'Subscriptions')) {
            throw new Error('Roles.findUsersInRole method received a role scope instead of a scope value.');
        }
        const role = await this.findOneById(roleId, { projection: { scope: 1 } });
        if (!role) {
            throw new Error('RolesRaw.findUsersInRole: role not found');
        }
        switch (role.scope) {
            case 'Subscriptions':
                return index_1.Subscriptions.findUsersInRoles([role._id], scope, options);
            case 'Users':
            default:
                return index_1.Users.findUsersInRoles([role._id], null, options);
        }
    }
    async countUsersInRole(roleId, scope) {
        const role = await this.findOneById(roleId, { projection: { scope: 1 } });
        if (!role) {
            throw new Error('RolesRaw.countUsersInRole: role not found');
        }
        switch (role.scope) {
            case 'Subscriptions':
                return index_1.Subscriptions.countUsersInRoles([role._id], scope);
            case 'Users':
            default:
                return index_1.Users.countUsersInRoles([role._id]);
        }
    }
    async createWithRandomId(name, scope = 'Users', description = '', protectedRole = true, mandatory2fa = false) {
        const role = {
            name,
            scope,
            description,
            protected: protectedRole,
            mandatory2fa,
        };
        const res = await this.insertOne(role);
        return {
            _id: res.insertedId,
            ...role,
        };
    }
    async canAddUserToRole(uid, roleId, scope) {
        if (process.env.NODE_ENV === 'development' && (scope === 'Users' || scope === 'Subscriptions')) {
            throw new Error('Roles.canAddUserToRole method received a role scope instead of a scope value.');
        }
        const role = await this.findOne({ _id: roleId }, { projection: { scope: 1 } });
        if (!role) {
            return false;
        }
        switch (role.scope) {
            case 'Subscriptions':
                return index_1.Subscriptions.isUserInRoleScope(uid, scope);
            case 'Users':
            default:
                return index_1.Users.isUserInRoleScope(uid);
        }
    }
}
exports.RolesRaw = RolesRaw;
//# sourceMappingURL=Roles.js.map