"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediaCallsRaw = void 0;
const BaseRaw_1 = require("./BaseRaw");
class MediaCallsRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'media_calls', trash);
    }
    modelIndexes() {
        return [
            { key: { createdAt: 1 }, unique: false },
            { key: { ended: 1, uids: 1, expiresAt: 1 }, unique: false },
            {
                key: { 'caller.type': 1, 'caller.id': 1, 'callerRequestedId': 1 },
                sparse: true,
            },
        ];
    }
    async findOneByCallerRequestedId(id, caller, options) {
        return this.findOne({
            'caller.type': caller.type,
            'caller.id': caller.id,
            'callerRequestedId': id,
        }, options);
    }
    updateOneById(_id, update, options) {
        return this.updateOne({ _id }, update, options);
    }
    async startRingingById(callId, expiresAt) {
        return this.updateOne({
            _id: callId,
            state: 'none',
        }, { $set: { state: 'ringing', expiresAt } });
    }
    async acceptCallById(callId, data, expiresAt) {
        const { calleeContractId } = data;
        return this.updateOne({
            _id: callId,
            state: { $in: ['none', 'ringing'] },
        }, {
            $set: {
                'state': 'accepted',
                'callee.contractId': calleeContractId,
                'acceptedAt': new Date(),
                expiresAt,
            },
        });
    }
    async activateCallById(callId, expiresAt) {
        return this.updateOne({
            _id: callId,
            state: 'accepted',
        }, {
            $set: {
                state: 'active',
                activatedAt: new Date(),
                expiresAt,
            },
        });
    }
    async hangupCallById(callId, params) {
        const { endedBy, reason } = params || {};
        return this.updateOne({
            _id: callId,
            ended: false,
        }, {
            $set: {
                state: 'hangup',
                ended: true,
                endedAt: new Date(),
                ...(endedBy && { endedBy }),
                ...(reason && { hangupReason: reason }),
            },
        });
    }
    async setExpiresAtById(callId, expiresAt) {
        return this.updateOne({
            _id: callId,
            ended: false,
        }, {
            $set: { expiresAt },
        });
    }
    async transferCallById(callId, params) {
        return this.updateOne({
            _id: callId,
            state: {
                $in: ['accepted', 'active'],
            },
            transferredAt: {
                $exists: false,
            },
        }, {
            $set: {
                transferredAt: new Date(),
                transferredBy: params.by,
                transferredTo: params.to,
            },
        });
    }
    findAllExpiredCalls(options) {
        return this.find({
            ended: false,
            expiresAt: {
                $lte: new Date(),
            },
        }, options);
    }
    findAllNotOverByUid(uid, options) {
        return this.find({
            ended: false,
            expiresAt: {
                $gt: new Date(),
            },
            uids: uid,
        }, options);
    }
    async hasUnfinishedCalls() {
        const count = await this.countDocuments({ ended: false }, { limit: 1 });
        return count > 0;
    }
    async hasUnfinishedCallsByUid(uid, exceptCallId) {
        const count = await this.countDocuments({
            ended: false,
            uids: uid,
            ...(exceptCallId && { _id: { $ne: exceptCallId } }),
        }, { limit: 1 });
        return count > 0;
    }
}
exports.MediaCallsRaw = MediaCallsRaw;
//# sourceMappingURL=MediaCalls.js.map