"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LivechatRoomsRaw = void 0;
const core_typings_1 = require("@rocket.chat/core-typings");
const string_helpers_1 = require("@rocket.chat/string-helpers");
const BaseRaw_1 = require("./BaseRaw");
const readSecondaryPreferred_1 = require("../readSecondaryPreferred");
/**
 * @extends BaseRaw<ILivechatRoom>
 */
class LivechatRoomsRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'room', trash);
    }
    // move indexes from constructor to here using IndexDescription as type
    modelIndexes() {
        return [
            { key: { open: 1 }, sparse: true },
            { key: { departmentId: 1 }, sparse: true },
            { key: { 'metrics.chatDuration': 1 }, sparse: true },
            { key: { 'metrics.serviceTimeDuration': 1 }, sparse: true },
            { key: { 'metrics.visitorInactivity': 1 }, sparse: true },
            { key: { 'omnichannel.predictedVisitorAbandonmentAt': 1 }, sparse: true },
            { key: { closedAt: 1 }, sparse: true },
            { key: { servedBy: 1 }, sparse: true },
            { key: { 'v.token': 1, 'email.thread': 1 }, sparse: true },
            { key: { 'v._id': 1 }, sparse: true },
            { key: { 'servedBy._id': 1, 'departmentId': 1, 't': 1, 'open': 1, 'ts': -1 } },
            { key: { t: 1, departmentId: 1, closedAt: 1 }, partialFilterExpression: { closedAt: { $exists: true } } },
            { key: { source: 1 }, sparse: true },
            { key: { departmentAncestors: 1 }, sparse: true },
            {
                key: { 't': 1, 'open': 1, 'source.type': 1, 'v.status': 1 },
                partialFilterExpression: {
                    't': { $eq: 'l' },
                    'open': { $eq: true },
                    'source.type': { $eq: 'widget' },
                },
            },
            { key: { 'livechatData.$**': 1 } },
            // TODO: Remove index on next major
            // { key: { pdfTranscriptRequested: 1 }, sparse: true },
            { key: { pdfTranscriptFileId: 1 }, sparse: true }, // used on statistics
            { key: { callStatus: 1 }, sparse: true }, // used on statistics
            { key: { priorityId: 1 }, sparse: true },
            { key: { slaId: 1 }, sparse: true },
            { key: { source: 1, ts: 1 }, partialFilterExpression: { source: { $exists: true }, t: 'l' } },
            { key: { departmentId: 1, ts: 1 }, partialFilterExpression: { departmentId: { $exists: true }, t: 'l' } },
            { key: { 'tags.0': 1, 'ts': 1 }, partialFilterExpression: { 'tags.0': { $exists: true }, 't': 'l' } },
            { key: { servedBy: 1, ts: 1 }, partialFilterExpression: { servedBy: { $exists: true }, t: 'l' } },
            { key: { 'v.activity': 1, 'ts': 1 }, partialFilterExpression: { 'v.activity': { $exists: true }, 't': 'l' } },
            { key: { contactId: 1 }, partialFilterExpression: { contactId: { $exists: true }, t: 'l' } },
        ];
    }
    async findOneById(_id, options) {
        const query = { _id, t: 'l' };
        if (options) {
            return this.findOne(query, options);
        }
        return this.findOne(query);
    }
    getQueueMetrics({ departmentId, agentId, includeOfflineAgents, options = {}, }) {
        const match = { $match: { t: 'l', open: true, servedBy: { $exists: true } } };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const departmentsLookup = {
            $lookup: {
                from: 'rocketchat_livechat_department',
                let: {
                    deptId: '$departmentId',
                },
                pipeline: [
                    {
                        $match: {
                            $expr: {
                                $eq: ['$_id', '$$deptId'],
                            },
                        },
                    },
                    {
                        $project: {
                            name: 1,
                        },
                    },
                ],
                as: 'departments',
            },
        };
        const departmentsUnwind = {
            $unwind: {
                path: '$departments',
                preserveNullAndEmptyArrays: true,
            },
        };
        const usersLookup = {
            $lookup: {
                from: 'users',
                let: {
                    servedById: '$servedBy._id',
                },
                pipeline: [
                    {
                        $match: {
                            $expr: {
                                $eq: ['$_id', '$$servedById'],
                            },
                            ...(!includeOfflineAgents && {
                                status: { $ne: 'offline' },
                                statusLivechat: 'available',
                            }),
                            ...(agentId && { _id: agentId }),
                        },
                    },
                    {
                        $project: {
                            _id: 1,
                            username: 1,
                            status: 1,
                        },
                    },
                ],
                as: 'user',
            },
        };
        const usersUnwind = {
            $unwind: {
                path: '$user',
            },
        };
        const usersGroup = {
            $group: {
                _id: {
                    userId: '$user._id',
                    username: '$user.username',
                    status: '$user.status',
                    departmentId: '$departmentId',
                    departmentName: '$departments.name',
                },
                chats: { $sum: 1 },
            },
        };
        const project = {
            $project: {
                _id: 0,
                user: {
                    _id: '$_id.userId',
                    username: '$_id.username',
                    status: '$_id.status',
                },
                department: {
                    _id: '$_id.departmentId',
                    name: '$_id.departmentName',
                },
                chats: 1,
            },
        };
        const firstParams = [match, departmentsLookup, departmentsUnwind, usersLookup, usersUnwind];
        const sort = { $sort: options.sort || { chats: -1 } };
        const pagination = [sort];
        if (options.offset) {
            pagination.push({ $skip: options.offset });
        }
        if (options.count) {
            pagination.push({ $limit: options.count });
        }
        const facet = {
            $facet: {
                sortedResults: pagination,
                totalCount: [{ $group: { _id: null, total: { $sum: 1 } } }],
            },
        };
        const params = [...firstParams, usersGroup, project, facet];
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)(), allowDiskUse: true }).toArray();
    }
    async findAllNumberOfAbandonedRooms({ start, end, departmentId, inactivityTimeout, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                't': 'l',
                'metrics.visitorInactivity': {
                    $gte: inactivityTimeout,
                },
                'ts': { $gte: new Date(start) },
                'closedAt': { $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departmentId',
                },
                abandonedRooms: { $sum: 1 },
            },
        };
        const project = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                abandonedRooms: 1,
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    async findPercentageOfAbandonedRooms({ start, end, inactivityTimeout, departmentId, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                t: 'l',
                ts: { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departmentId',
                },
                rooms: { $sum: 1 },
                abandonedChats: {
                    $sum: {
                        $cond: [
                            {
                                $and: [
                                    { $ifNull: ['$metrics.visitorInactivity', false] },
                                    {
                                        $gte: ['$metrics.visitorInactivity', inactivityTimeout],
                                    },
                                ],
                            },
                            1,
                            0,
                        ],
                    },
                },
            },
        };
        const project = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                percentageOfAbandonedChats: {
                    $floor: {
                        $cond: [{ $eq: ['$rooms', 0] }, 0, { $divide: [{ $multiply: ['$abandonedChats', 100] }, '$rooms'] }],
                    },
                },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findAllAverageOfChatDurationTime({ start, end, departmentId, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                t: 'l',
                ts: { $gte: new Date(start) },
                closedAt: { $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departmentId',
                },
                rooms: { $sum: 1 },
                chatsDuration: { $sum: '$metrics.chatDuration' },
            },
        };
        const project = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                averageChatDurationTimeInSeconds: {
                    $ceil: { $cond: [{ $eq: ['$rooms', 0] }, 0, { $divide: ['$chatsDuration', '$rooms'] }] },
                },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findAllAverageWaitingTime({ start, end, departmentId, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                t: 'l',
                ts: { $gte: new Date(start), $lte: new Date(end) },
                waitingResponse: { $ne: true },
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departmentId',
                },
                rooms: { $sum: 1 },
                chatsFirstResponses: { $sum: '$metrics.response.ft' },
            },
        };
        const project = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                averageWaitingTimeInSeconds: {
                    $ceil: {
                        $cond: [{ $eq: ['$rooms', 0] }, 0, { $divide: ['$chatsFirstResponses', '$rooms'] }],
                    },
                },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findAllRooms({ start, end, answered, departmentId, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                t: 'l',
                ts: { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        if (answered !== undefined) {
            match.$match.waitingResponse = { [answered ? '$ne' : '$eq']: true };
        }
        const group = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departmentId',
                },
                rooms: { $sum: 1 },
            },
        };
        const project = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                rooms: 1,
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findAllServiceTime({ start, end, departmentId, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                't': 'l',
                'ts': { $gte: new Date(start) },
                'closedAt': { $lte: new Date(end) },
                'metrics.serviceTimeDuration': { $exists: true },
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departmentId',
                },
                rooms: { $sum: 1 },
                serviceTimeDuration: { $sum: '$metrics.serviceTimeDuration' },
            },
        };
        const project = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                chats: '$rooms',
                serviceTimeDuration: { $ceil: '$serviceTimeDuration' },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findAllNumberOfTransferredRooms({ start, end, departmentId, options = {}, }) {
        const match = {
            $match: {
                t: 'l',
                ts: { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const departmentsLookup = {
            $lookup: {
                from: 'rocketchat_livechat_department',
                localField: 'departmentId',
                foreignField: '_id',
                as: 'departments',
            },
        };
        const departmentsUnwind = {
            $unwind: {
                path: '$departments',
                preserveNullAndEmptyArrays: true,
            },
        };
        const departmentsGroup = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departments._id',
                    name: '$departments.name',
                },
                rooms: { $push: '$$ROOT' },
            },
        };
        const departmentsProject = {
            $project: {
                _id: '$_id.departmentId',
                name: '$_id.name',
                rooms: 1,
            },
        };
        const roomsUnwind = {
            $unwind: {
                path: '$rooms',
                preserveNullAndEmptyArrays: true,
            },
        };
        const messagesLookup = {
            $lookup: {
                from: 'rocketchat_message',
                localField: 'rooms._id',
                foreignField: 'rid',
                as: 'messages',
            },
        };
        const messagesProject = {
            $project: {
                _id: 1,
                name: 1,
                messages: {
                    $filter: {
                        input: '$messages',
                        as: 'message',
                        cond: {
                            $and: [{ $eq: ['$$message.t', 'livechat_transfer_history'] }],
                        },
                    },
                },
            },
        };
        const transferProject = {
            $project: {
                name: 1,
                transfers: { $size: { $ifNull: ['$messages', []] } },
            },
        };
        const transferGroup = {
            $group: {
                _id: {
                    departmentId: '$_id',
                    name: '$name',
                },
                numberOfTransferredRooms: { $sum: '$transfers' },
            },
        };
        const presentationProject = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                name: { $ifNull: ['$_id.name', null] },
                numberOfTransferredRooms: 1,
            },
        };
        const firstParams = [match, departmentsLookup, departmentsUnwind];
        if (departmentId && departmentId !== 'undefined') {
            firstParams.push({
                $match: {
                    'departments._id': departmentId,
                },
            });
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [
            ...firstParams,
            departmentsGroup,
            departmentsProject,
            roomsUnwind,
            messagesLookup,
            messagesProject,
            transferProject,
            transferGroup,
            presentationProject,
            sort,
        ];
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { allowDiskUse: true, readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    countAllOpenChatsBetweenDate({ start, end, departmentId }) {
        const query = {
            't': 'l',
            'metrics.chatDuration': {
                $exists: false,
            },
            '$or': [
                {
                    onHold: {
                        $exists: false,
                    },
                },
                {
                    onHold: {
                        $exists: true,
                        $eq: false,
                    },
                },
            ],
            'servedBy': { $exists: true },
            'ts': { $gte: new Date(start), $lte: new Date(end) },
        };
        if (departmentId && departmentId !== 'undefined') {
            query.departmentId = departmentId;
        }
        return this.countDocuments(query);
    }
    countAllClosedChatsBetweenDate({ start, end, departmentId }) {
        const query = {
            't': 'l',
            'metrics.chatDuration': {
                $exists: true,
            },
            'ts': { $gte: new Date(start), $lte: new Date(end) },
        };
        if (departmentId && departmentId !== 'undefined') {
            query.departmentId = departmentId;
        }
        return this.countDocuments(query);
    }
    countAllQueuedChatsBetweenDate({ start, end, departmentId }) {
        const query = {
            t: 'l',
            servedBy: { $exists: false },
            open: true,
            ts: { $gte: new Date(start), $lte: new Date(end) },
        };
        if (departmentId && departmentId !== 'undefined') {
            query.departmentId = departmentId;
        }
        return this.countDocuments(query);
    }
    countAllOpenChatsByAgentBetweenDate({ start, end, departmentId }) {
        const match = {
            $match: {
                't': 'l',
                'servedBy._id': { $exists: true },
                'open': true,
                '$or': [
                    {
                        onHold: {
                            $exists: false,
                        },
                    },
                    {
                        onHold: {
                            $exists: true,
                            $eq: false,
                        },
                    },
                ],
                'ts': { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: '$servedBy.username',
                chats: { $sum: 1 },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        return this.col.aggregate([match, group], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    countAllOnHoldChatsByAgentBetweenDate({ start, end, departmentId }) {
        const match = {
            $match: {
                't': 'l',
                'servedBy._id': { $exists: true },
                'open': true,
                'onHold': {
                    $exists: true,
                    $eq: true,
                },
                'ts': { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: '$servedBy.username',
                chats: { $sum: 1 },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        return this.col.aggregate([match, group], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    countAllClosedChatsByAgentBetweenDate({ start, end, departmentId }) {
        const match = {
            $match: {
                't': 'l',
                'open': { $exists: false },
                'servedBy._id': { $exists: true },
                'ts': { $gte: new Date(start) },
                'closedAt': { $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: '$servedBy.username',
                chats: { $sum: 1 },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        return this.col.aggregate([match, group], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    countAllOpenChatsByDepartmentBetweenDate({ start, end, departmentId }) {
        const match = {
            $match: {
                t: 'l',
                open: true,
                departmentId: { $exists: true },
                ts: { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const lookup = {
            $lookup: {
                from: 'rocketchat_livechat_department',
                localField: 'departmentId',
                foreignField: '_id',
                as: 'departments',
            },
        };
        const unwind = {
            $unwind: {
                path: '$departments',
                preserveNullAndEmptyArrays: true,
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: '$departments._id',
                    name: '$departments.name',
                },
                chats: { $sum: 1 },
            },
        };
        const project = {
            $project: {
                _id: '$_id._id',
                name: '$_id.name',
                chats: 1,
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const params = [match, lookup, unwind, group, project];
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    countAllClosedChatsByDepartmentBetweenDate({ start, end, departmentId }) {
        const match = {
            $match: {
                t: 'l',
                open: { $exists: false },
                departmentId: { $exists: true },
                ts: { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const lookup = {
            $lookup: {
                from: 'rocketchat_livechat_department',
                localField: 'departmentId',
                foreignField: '_id',
                as: 'departments',
            },
        };
        const unwind = {
            $unwind: {
                path: '$departments',
                preserveNullAndEmptyArrays: true,
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: '$departments._id',
                    name: '$departments.name',
                },
                chats: { $sum: 1 },
            },
        };
        const project = {
            $project: {
                _id: '$_id._id',
                name: '$_id.name',
                chats: 1,
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const params = [match, lookup, unwind, group, project];
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    calculateResponseTimingsBetweenDates({ start, end, departmentId }) {
        const match = {
            $match: {
                t: 'l',
                ts: { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: null,
                sumResponseAvg: {
                    $sum: '$metrics.response.avg',
                },
                roomsWithResponseTime: {
                    $sum: {
                        $cond: [
                            {
                                $and: [{ $ifNull: ['$metrics.response.avg', false] }],
                            },
                            1,
                            0,
                        ],
                    },
                },
                maxFirstResponse: { $max: '$metrics.response.ft' },
            },
        };
        const project = {
            $project: {
                avg: {
                    $trunc: {
                        $cond: [{ $eq: ['$roomsWithResponseTime', 0] }, 0, { $divide: ['$sumResponseAvg', '$roomsWithResponseTime'] }],
                    },
                },
                longest: '$maxFirstResponse',
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        return this.col.aggregate([match, group, project], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    calculateReactionTimingsBetweenDates({ start, end, departmentId }) {
        const match = {
            $match: {
                t: 'l',
                ts: { $gte: new Date(start), $lte: new Date(end) },
            },
        };
        const group = {
            $group: {
                _id: null,
                sumReactionFirstResponse: {
                    $sum: '$metrics.reaction.ft',
                },
                roomsWithFirstReaction: {
                    $sum: {
                        $cond: [
                            {
                                $and: [{ $ifNull: ['$metrics.reaction.ft', false] }],
                            },
                            1,
                            0,
                        ],
                    },
                },
                maxFirstReaction: { $max: '$metrics.reaction.ft' },
            },
        };
        const project = {
            $project: {
                avg: {
                    $trunc: {
                        $cond: [{ $eq: ['$roomsWithFirstReaction', 0] }, 0, { $divide: ['$sumReactionFirstResponse', '$roomsWithFirstReaction'] }],
                    },
                },
                longest: '$maxFirstReaction',
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        return this.col.aggregate([match, group, project], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    calculateDurationTimingsBetweenDates({ start, end, departmentId }) {
        const match = {
            $match: {
                't': 'l',
                'ts': { $gte: new Date(start), $lte: new Date(end) },
                'metrics.chatDuration': { $exists: true },
            },
        };
        const group = {
            $group: {
                _id: null,
                sumChatDuration: {
                    $sum: '$metrics.chatDuration',
                },
                roomsWithChatDuration: {
                    $sum: {
                        $cond: [
                            {
                                $and: [{ $ifNull: ['$metrics.chatDuration', false] }],
                            },
                            1,
                            0,
                        ],
                    },
                },
                maxChatDuration: { $max: '$metrics.chatDuration' },
            },
        };
        const project = {
            $project: {
                avg: {
                    $trunc: {
                        $cond: [{ $eq: ['$roomsWithChatDuration', 0] }, 0, { $divide: ['$sumChatDuration', '$roomsWithChatDuration'] }],
                    },
                },
                longest: '$maxChatDuration',
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        return this.col.aggregate([match, group, project], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() }).toArray();
    }
    findAllAverageOfServiceTime({ start, end, departmentId, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                't': 'l',
                'ts': { $gte: new Date(start), $lte: new Date(end) },
                'responseBy.lastMessageTs': { $exists: true },
                'servedBy._id': { $exists: true },
            },
        };
        const group = {
            $group: {
                _id: {
                    _id: null,
                    departmentId: '$departmentId',
                },
                rooms: { $sum: 1 },
                allServiceTime: {
                    $sum: { $divide: [{ $subtract: ['$responseBy.lastMessageTs', '$servedBy.ts'] }, 1000] },
                },
            },
        };
        const project = {
            $project: {
                _id: { $ifNull: ['$_id.departmentId', null] },
                averageServiceTimeInSeconds: {
                    $ceil: { $cond: [{ $eq: ['$rooms', 0] }, 0, { $divide: ['$allServiceTime', '$rooms'] }] },
                },
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            match.$match.departmentId = departmentId;
        }
        const sort = { $sort: options.sort || { name: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findByVisitorId(visitorId, options, extraQuery = {}) {
        const query = {
            't': 'l',
            'v._id': visitorId,
            ...extraQuery,
        };
        return this.find(query, options);
    }
    findPaginatedByVisitorId(visitorId, options, extraQuery = {}) {
        const query = {
            't': 'l',
            'v._id': visitorId,
            ...extraQuery,
        };
        return this.findPaginated(query, options);
    }
    findRoomsByVisitorIdAndMessageWithCriteria({ visitorId, searchText, open, served, onlyCount = false, source, options = {}, }) {
        const match = {
            $match: {
                'v._id': visitorId,
                ...(open !== undefined && !open && { closedAt: { $exists: true } }),
                ...(served !== undefined && served && { servedBy: { $exists: served } }),
                ...(source && {
                    $or: [{ 'source.type': new RegExp((0, string_helpers_1.escapeRegExp)(source), 'i') }, { 'source.alias': new RegExp((0, string_helpers_1.escapeRegExp)(source), 'i') }],
                }),
            },
        };
        const lookup = {
            $lookup: {
                from: 'rocketchat_message',
                localField: '_id',
                foreignField: 'rid',
                as: 'messages',
            },
        };
        const matchMessages = searchText && {
            $match: { 'messages.msg': { $regex: `.*${(0, string_helpers_1.escapeRegExp)(searchText)}.*` } },
        };
        const params = [match, lookup];
        if (matchMessages) {
            params.push(matchMessages);
        }
        const project = {
            $project: {
                fname: 1,
                ts: 1,
                v: 1,
                msgs: 1,
                servedBy: 1,
                closedAt: 1,
                closedBy: 1,
                closer: 1,
                tags: 1,
                closingMessage: {
                    $filter: {
                        input: '$messages',
                        as: 'messages',
                        cond: { $eq: ['$$messages.t', 'livechat-close'] },
                    },
                },
            },
        };
        const unwindClosingMsg = {
            $unwind: { path: '$closingMessage', preserveNullAndEmptyArrays: true },
        };
        const sort = { $sort: options.sort || { ts: -1 } };
        params.push(project, unwindClosingMsg, sort);
        if (onlyCount) {
            params.push({ $count: 'count' });
            return this.col.aggregate(params);
        }
        if (options.skip) {
            params.push({ $skip: options.skip });
        }
        if (options.limit) {
            params.push({ $limit: options.limit });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findClosedRoomsByContactPaginated({ contactId, options = {}, }) {
        return this.findPaginated({
            contactId,
            closedAt: { $exists: true },
        }, options);
    }
    findRoomsWithCriteria({ agents, roomName, departmentId, open, served, createdAt, closedAt, tags, customFields, visitorId, roomIds, onhold, queued, options = {}, extraQuery = {}, }) {
        const isRoomNameExactTerm = roomName?.startsWith(`"`) && roomName?.endsWith(`"`);
        const roomNameQuery = isRoomNameExactTerm ? roomName?.slice(1, -1) : roomName;
        const query = {
            t: 'l',
            ...extraQuery,
            ...(agents && { 'servedBy._id': { $in: agents } }),
            ...(roomName && isRoomNameExactTerm
                ? { fname: roomNameQuery } // exact match
                : roomName && { fname: new RegExp((0, string_helpers_1.escapeRegExp)(roomName), 'i') }), // regex match
            ...(departmentId && departmentId !== 'undefined' && { departmentId: { $in: [].concat(departmentId) } }),
            ...(open !== undefined && { open: { $exists: open }, onHold: { $ne: true } }),
            ...(served !== undefined && { servedBy: { $exists: served } }),
            ...(visitorId && visitorId !== 'undefined' && { 'v._id': visitorId }),
        };
        if (open) {
            query.servedBy = { $exists: true };
        }
        if (createdAt) {
            query.ts = {};
            if (createdAt.start) {
                query.ts.$gte = new Date(createdAt.start);
            }
            if (createdAt.end) {
                query.ts.$lte = new Date(createdAt.end);
            }
        }
        if (closedAt) {
            query.closedAt = {};
            if (closedAt.start) {
                query.closedAt.$gte = new Date(closedAt.start);
            }
            if (closedAt.end) {
                query.closedAt.$lte = new Date(closedAt.end);
            }
        }
        if (tags) {
            query.tags = { $in: tags };
        }
        if (customFields && Object.keys(customFields).length) {
            query.$and = Object.keys(customFields).map((key) => ({
                [`livechatData.${key}`]: new RegExp(customFields[key], 'i'),
            }));
        }
        if (roomIds) {
            query._id = { $in: roomIds };
        }
        if (onhold) {
            query.onHold = {
                $exists: true,
                $eq: onhold,
            };
        }
        if (queued) {
            query.servedBy = { $exists: false };
            query.open = true;
            query.onHold = { $ne: true };
        }
        return this.findPaginated(query, {
            sort: options.sort || { name: 1 },
            skip: options.offset,
            limit: options.count,
        });
    }
    getOnHoldConversationsBetweenDate(from, to, departmentId) {
        const query = {
            onHold: {
                $exists: true,
                $eq: true,
            },
            ts: {
                $gte: new Date(from), // ISO Date, ts >= date.gte
                $lt: new Date(to), // ISODate, ts < date.lt
            },
        };
        if (departmentId && departmentId !== 'undefined') {
            query.departmentId = departmentId;
        }
        return this.countDocuments(query);
    }
    findAllServiceTimeByAgent({ start, end, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                't': 'l',
                'servedBy._id': { $exists: true },
                'metrics.serviceTimeDuration': { $exists: true },
                'ts': {
                    $gte: start,
                    $lte: end,
                },
            },
        };
        const group = {
            $group: {
                _id: { _id: '$servedBy._id', username: '$servedBy.username' },
                chats: { $sum: 1 },
                serviceTimeDuration: { $sum: '$metrics.serviceTimeDuration' },
            },
        };
        const project = {
            $project: {
                _id: '$_id._id',
                username: '$_id.username',
                chats: 1,
                serviceTimeDuration: { $ceil: '$serviceTimeDuration' },
            },
        };
        const sort = { $sort: options.sort || { username: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    findAllAverageServiceTimeByAgents({ start, end, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                't': 'l',
                'servedBy._id': { $exists: true },
                'metrics.serviceTimeDuration': { $exists: true },
                'ts': {
                    $gte: start,
                    $lte: end,
                },
            },
        };
        const group = {
            $group: {
                _id: { _id: '$servedBy._id', username: '$servedBy.username' },
                chats: { $sum: 1 },
                serviceTimeDuration: { $sum: '$metrics.serviceTimeDuration' },
            },
        };
        const project = {
            $project: {
                _id: '$_id._id',
                username: '$_id.username',
                name: '$_id.name',
                active: '$_id.active',
                averageServiceTimeInSeconds: {
                    $ceil: {
                        $cond: [{ $eq: ['$chats', 0] }, 0, { $divide: ['$serviceTimeDuration', '$chats'] }],
                    },
                },
            },
        };
        const sort = { $sort: options.sort || { username: 1 } };
        const params = [match, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    setDepartmentByRoomId(roomId, departmentId) {
        return this.updateOne({ _id: roomId }, { $set: { departmentId } });
    }
    findOpen(extraQuery = {}) {
        return this.find({ t: 'l', open: true, ...extraQuery });
    }
    setAutoTransferOngoingById(roomId) {
        const query = {
            _id: roomId,
        };
        const update = {
            $set: {
                autoTransferOngoing: true,
            },
        };
        return this.updateOne(query, update);
    }
    unsetAutoTransferOngoingById(roomId) {
        const query = {
            _id: roomId,
        };
        const update = {
            $unset: {
                autoTransferOngoing: 1,
            },
        };
        return this.updateOne(query, update);
    }
    setAutoTransferredAtById(roomId) {
        const query = {
            _id: roomId,
        };
        const update = {
            $set: {
                autoTransferredAt: new Date(),
            },
        };
        return this.updateOne(query, update);
    }
    findAvailableSources() {
        return this.col.aggregate([
            {
                $group: {
                    _id: 0,
                    types: {
                        $addToSet: {
                            $cond: {
                                if: {
                                    $eq: ['$source.type', 'app'],
                                },
                                then: '$$REMOVE',
                                else: { type: '$source.type' },
                            },
                        },
                    },
                    apps: {
                        $addToSet: {
                            $cond: {
                                if: {
                                    $eq: ['$source.type', 'app'],
                                },
                                else: '$$REMOVE',
                                then: {
                                    type: '$source.type',
                                    id: '$source.id',
                                    alias: '$source.alias',
                                    sidebarIcon: '$source.sidebarIcon',
                                    defaultIcon: '$source.defaultIcon',
                                },
                            },
                        },
                    },
                },
            },
            {
                $project: {
                    _id: 0,
                    fullTypes: { $setUnion: ['$types', '$apps'] },
                },
            },
        ]);
    }
    setPdfTranscriptFileIdById(rid, fileId) {
        return this.updateOne({
            _id: rid,
        }, {
            $set: { pdfTranscriptFileId: fileId },
        });
    }
    setEmailTranscriptRequestedByRoomId(roomId, transcriptInfo) {
        const { requestedAt, requestedBy, email, subject } = transcriptInfo;
        return this.updateOne({
            _id: roomId,
            t: 'l',
        }, {
            $set: {
                transcriptRequest: {
                    requestedAt,
                    requestedBy,
                    email,
                    subject,
                },
            },
        });
    }
    unsetEmailTranscriptRequestedByRoomId(roomId) {
        return this.updateOne({
            _id: roomId,
            t: 'l',
        }, {
            $unset: {
                transcriptRequest: 1,
            },
        });
    }
    closeRoomById(roomId, closeInfo, options) {
        const { closer, closedBy, closedAt, chatDuration, serviceTimeDuration, tags } = closeInfo;
        return this.updateOne({
            _id: roomId,
            t: 'l',
        }, {
            $set: {
                closedAt,
                'metrics.chatDuration': chatDuration,
                'metrics.serviceTimeDuration': serviceTimeDuration,
                'v.status': core_typings_1.UserStatus.OFFLINE,
                ...(closer && { closer }),
                ...(closedBy && { closedBy }),
                ...(tags && { tags }),
            },
            $unset: {
                open: 1,
            },
        }, options);
    }
    bulkRemoveDepartmentAndUnitsFromRooms(departmentId) {
        return this.updateMany({ departmentId }, { $unset: { departmentId: 1, departmentAncestors: 1 } });
    }
    findOneByIdOrName(_idOrName, options) {
        const query = {
            t: 'l',
            $or: [
                {
                    _id: _idOrName,
                },
                {
                    name: _idOrName,
                },
            ],
        };
        return this.findOne(query, options);
    }
    updateSurveyFeedbackById(_id, surveyFeedback) {
        const query = {
            _id,
        };
        const update = {
            $set: {
                surveyFeedback,
            },
        };
        return this.updateOne(query, update);
    }
    async updateDataByToken(token, key, value, overwrite = true) {
        const query = {
            'v.token': token,
            'open': true,
        };
        if (!overwrite) {
            const room = await this.findOne(query, { projection: { livechatData: 1 } });
            if (!room) {
                return false;
            }
            if (room.livechatData && typeof room.livechatData[key] !== 'undefined') {
                return true;
            }
        }
        const update = {
            $set: {
                [`livechatData.${key}`]: value,
            },
        };
        return this.updateMany(query, update);
    }
    async saveRoomById({ _id, topic, tags, livechatData, ...extra }) {
        const setData = { ...extra };
        const unsetData = {};
        if (topic != null) {
            const trimmedTopic = topic.trim();
            if (trimmedTopic.length) {
                setData.topic = trimmedTopic;
            }
            else {
                unsetData.topic = 1;
            }
        }
        if (Array.isArray(tags) && tags.length > 0) {
            setData.tags = tags;
        }
        else {
            unsetData.tags = 1;
        }
        if (extra.priorityId === '') {
            unsetData.priorityId = 1;
            delete setData.priorityId;
        }
        if (extra.slaId === '') {
            unsetData.slaId = 1;
            delete setData.slaId;
        }
        if (livechatData) {
            Object.keys(livechatData).forEach((key) => {
                const value = livechatData[key].trim();
                if (value) {
                    setData[`livechatData.${key}`] = value;
                }
                else {
                    unsetData[`livechatData.${key}`] = 1;
                }
            });
        }
        const update = {};
        if (Object.keys(setData).length > 0) {
            update.$set = setData;
        }
        if (Object.keys(unsetData).length > 0) {
            update.$unset = unsetData;
        }
        if (Object.keys(update).length === 0) {
            return;
        }
        return this.updateOne({ _id }, update);
    }
    findById(_id, fields) {
        const options = {};
        if (fields) {
            options.projection = fields;
        }
        const query = {
            t: 'l',
            _id,
        };
        return this.find(query, options);
    }
    findByIds(ids, fields, extraQuery = {}) {
        const options = {};
        if (fields) {
            options.projection = fields;
        }
        const query = {
            t: 'l',
            _id: { $in: ids },
            ...extraQuery,
        };
        return this.find(query, options);
    }
    findOneByIdAndVisitorToken(_id, visitorToken, fields) {
        const options = {};
        if (fields) {
            options.projection = fields;
        }
        const query = {
            't': 'l',
            _id,
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    findOneByVisitorTokenAndEmailThread(visitorToken, emailThread, options) {
        const query = {
            't': 'l',
            'v.token': visitorToken,
            '$or': [{ 'email.thread': { $elemMatch: { $in: emailThread } } }, { 'email.thread': new RegExp(emailThread.join('|')) }],
        };
        return this.findOne(query, options);
    }
    findOneByVisitorTokenAndEmailThreadAndDepartment(visitorToken, emailThread, departmentId, options) {
        const query = {
            't': 'l',
            'v.token': visitorToken,
            '$or': [
                { 'email.thread': { $elemMatch: { $in: emailThread } } },
                { 'email.thread': new RegExp(emailThread.map((t) => `"${t}"`).join('|')) },
            ],
            ...(departmentId && { departmentId }),
        };
        return this.findOne(query, options);
    }
    findOneOpenByVisitorTokenAndEmailThread(visitorToken, emailThread, options) {
        const query = {
            't': 'l',
            'open': true,
            'v.token': visitorToken,
            '$or': [{ 'email.thread': { $elemMatch: { $in: emailThread } } }, { 'email.thread': new RegExp(emailThread.join('|')) }],
        };
        return this.findOne(query, options);
    }
    updateEmailThreadByRoomId(roomId, threadIds) {
        const query = {
            $addToSet: {
                'email.thread': threadIds,
            },
        };
        return this.updateOne({ _id: roomId }, query);
    }
    findOneLastServedAndClosedByVisitorToken(visitorToken, options = {}) {
        const query = {
            't': 'l',
            'v.token': visitorToken,
            'closedAt': { $exists: true },
            'servedBy': { $exists: true },
        };
        options.sort = { closedAt: -1 };
        return this.findOne(query, options);
    }
    findOneByVisitorToken(visitorToken, fields) {
        const options = {};
        if (fields) {
            options.projection = fields;
        }
        const query = {
            't': 'l',
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    findOpenByVisitorToken(visitorToken, options = {}, extraQuery = {}) {
        const query = {
            't': 'l',
            'open': true,
            'v.token': visitorToken,
            ...extraQuery,
        };
        return this.find(query, options);
    }
    findOneOpenByContactChannelVisitor(association, options = {}) {
        const query = {
            't': 'l',
            'open': true,
            'v._id': association.visitorId,
            'source.type': association.source.type,
            ...(association.source.id ? { 'source.id': association.source.id } : {}),
        };
        return this.findOne(query, options);
    }
    findOneOpenByVisitorToken(visitorToken, options = {}) {
        const query = {
            't': 'l',
            'open': true,
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    findOneOpenByVisitorTokenAndDepartmentIdAndSource(visitorToken, departmentId, source, options = {}) {
        const query = {
            't': 'l',
            'open': true,
            'v.token': visitorToken,
            departmentId,
            ...(source && { 'source.type': source }),
        };
        return this.findOne(query, options);
    }
    findOpenByVisitorTokenAndDepartmentId(visitorToken, departmentId, options = {}, extraQuery = {}) {
        const query = {
            't': 'l',
            'open': true,
            'v.token': visitorToken,
            departmentId,
            ...extraQuery,
        };
        return this.find(query, options);
    }
    findByVisitorToken(visitorToken, extraQuery = {}, options) {
        const query = {
            't': 'l',
            'v.token': visitorToken,
            ...extraQuery,
        };
        return this.find(query, options);
    }
    findByVisitorIdAndAgentId(visitorId, agentId, options = {}, extraQuery = {}) {
        const query = {
            t: 'l',
            ...(visitorId && { 'v._id': visitorId }),
            ...(agentId && { 'servedBy._id': agentId }),
            ...extraQuery,
        };
        return this.find(query, options);
    }
    async findNewestByContactVisitorAssociation(association, options = {}) {
        const query = {
            't': 'l',
            'v._id': association.visitorId,
            'source.type': association.source.type,
            ...(association.source.id ? { 'source.id': association.source.id } : {}),
        };
        return this.findOne(query, {
            ...options,
            sort: { _updatedAt: -1 },
        });
    }
    findOneOpenByRoomIdAndVisitorToken(roomId, visitorToken, options = {}) {
        const query = {
            't': 'l',
            '_id': roomId,
            'open': true,
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    findClosedRooms(departmentIds, options = {}, extraQuery = {}) {
        const query = {
            t: 'l',
            open: { $exists: false },
            closedAt: { $exists: true },
            ...(Array.isArray(departmentIds) && departmentIds.length > 0 && { departmentId: { $in: departmentIds } }),
            ...extraQuery,
        };
        return this.find(query, options);
    }
    getResponseByRoomIdUpdateQuery(responseBy, updater = this.getUpdater()) {
        updater.set('responseBy', responseBy);
        updater.unset('waitingResponse');
        return updater;
    }
    getNotResponseByRoomIdUpdateQuery(updater = this.getUpdater()) {
        updater.set('waitingResponse', true);
        updater.unset('responseBy');
        return updater;
    }
    getAgentLastMessageTsUpdateQuery(updater = this.getUpdater()) {
        return updater.set('responseBy.lastMessageTs', new Date());
    }
    getAnalyticsUpdateQuery(analyticsData, updater = this.getUpdater()) {
        if (analyticsData) {
            updater.set('metrics.response.avg', analyticsData.avgResponseTime);
            updater.inc('metrics.response.total', 1);
            updater.inc('metrics.response.tt', analyticsData.responseTime);
            updater.inc('metrics.reaction.tt', analyticsData.reactionTime);
        }
        if (analyticsData?.firstResponseTime) {
            updater.set('metrics.reaction.fd', analyticsData.firstReactionDate);
            updater.set('metrics.reaction.ft', analyticsData.firstReactionTime);
            updater.set('metrics.response.fd', analyticsData.firstResponseDate);
            updater.set('metrics.response.ft', analyticsData.firstResponseTime);
        }
        return updater;
    }
    getAnalyticsUpdateQueryBySentByAgent(room, message, analyticsData, updater = this.getUpdater()) {
        // livechat analytics : update last message timestamps
        const visitorLastQuery = room.metrics?.v ? room.metrics.v.lq : room.ts;
        const agentLastReply = room.metrics?.servedBy ? room.metrics.servedBy.lr : room.ts;
        if (visitorLastQuery > agentLastReply) {
            return this.getAnalyticsUpdateQuery(analyticsData, updater).set('metrics.servedBy.lr', message.ts);
        }
        return this.getAnalyticsUpdateQuery(analyticsData, updater);
    }
    getAnalyticsUpdateQueryBySentByVisitor(room, message, updater = this.getUpdater()) {
        // livechat analytics : update last message timestamps
        const visitorLastQuery = room.metrics?.v ? room.metrics.v.lq : room.ts;
        const agentLastReply = room.metrics?.servedBy ? room.metrics.servedBy.lr : room.ts;
        // update visitor timestamp, only if its new inquiry and not continuing message
        if (agentLastReply >= visitorLastQuery) {
            return updater.set('metrics.v.lq', message.ts);
        }
        return updater;
    }
    getTotalConversationsBetweenDate(t, date, { departmentId } = {}) {
        const query = {
            t,
            ts: {
                $gte: new Date(date.gte), // ISO Date, ts >= date.gte
                $lte: new Date(date.lte), // ISODate, ts <= date.lte
            },
            ...(departmentId && departmentId !== 'undefined' && { departmentId }),
        };
        return this.countDocuments(query);
    }
    getAnalyticsMetricsBetweenDate(t, date, { departmentId } = {}, extraQuery = {}) {
        const query = {
            t,
            ts: {
                $gte: new Date(date.gte), // ISO Date, ts >= date.gte
                $lte: new Date(date.lte), // ISODate, ts <= date.lte
            },
            ...(departmentId && departmentId !== 'undefined' && { departmentId }),
            ...extraQuery,
        };
        return this.find(query, {
            projection: { ts: 1, departmentId: 1, open: 1, servedBy: 1, responseBy: 1, metrics: 1, msgs: 1 },
        });
    }
    getAnalyticsMetricsBetweenDateWithMessages(t, date, { departmentId } = {}, extraQuery = {}, extraMatchers = {}) {
        return this.col.aggregate([
            {
                $match: {
                    t,
                    ts: {
                        $gte: new Date(date.gte), // ISO Date, ts >= date.gte
                        $lte: new Date(date.lte), // ISODate, ts <= date.lte
                    },
                    ...(departmentId && departmentId !== 'undefined' && { departmentId }),
                    ...extraMatchers,
                },
            },
            { $addFields: { roomId: '$_id' } },
            {
                $lookup: {
                    from: 'rocketchat_message',
                    // mongo doesn't like _id as variable name here :(
                    let: { roomId: '$roomId' },
                    pipeline: [
                        {
                            $match: {
                                $expr: {
                                    $and: [
                                        {
                                            $eq: ['$$roomId', '$rid'],
                                        },
                                        {
                                            // this is similar to do { $exists: false }
                                            $lte: ['$t', null],
                                        },
                                        ...(extraQuery ? [extraQuery] : []),
                                    ],
                                },
                            },
                        },
                    ],
                    as: 'messages',
                },
            },
            {
                $unwind: {
                    path: '$messages',
                    preserveNullAndEmptyArrays: true,
                },
            },
            {
                $group: {
                    _id: {
                        _id: '$_id',
                        ts: '$ts',
                        departmentId: '$departmentId',
                        open: '$open',
                        servedBy: '$servedBy',
                        metrics: '$metrics',
                    },
                    messagesCount: {
                        $sum: 1,
                    },
                },
            },
            {
                $project: {
                    _id: '$_id._id',
                    ts: '$_id.ts',
                    departmentId: '$_id.departmentId',
                    open: '$_id.open',
                    servedBy: '$_id.servedBy',
                    metrics: '$_id.metrics',
                    msgs: '$messagesCount',
                },
            },
        ], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    getAnalyticsBetweenDate(date, { departmentId } = {}) {
        return this.col.aggregate([
            {
                $match: {
                    t: 'l',
                    ts: {
                        $gte: new Date(date.gte), // ISO Date, ts >= date.gte
                        $lte: new Date(date.lte), // ISODate, ts <= date.lte
                    },
                    ...(departmentId && departmentId !== 'undefined' && { departmentId }),
                },
            },
            { $addFields: { roomId: '$_id' } },
            {
                $lookup: {
                    from: 'rocketchat_message',
                    // mongo doesn't like _id as variable name here :(
                    let: { roomId: '$roomId' },
                    pipeline: [
                        {
                            $match: {
                                $expr: {
                                    $and: [
                                        {
                                            $eq: ['$$roomId', '$rid'],
                                        },
                                        {
                                            // this is similar to do { $exists: false }
                                            $lte: ['$t', null],
                                        },
                                    ],
                                },
                            },
                        },
                    ],
                    as: 'messages',
                },
            },
            {
                $unwind: {
                    path: '$messages',
                    preserveNullAndEmptyArrays: true,
                },
            },
            {
                $group: {
                    _id: {
                        _id: '$_id',
                        ts: '$ts',
                        departmentId: '$departmentId',
                        open: '$open',
                        servedBy: '$servedBy',
                        metrics: '$metrics',
                        onHold: '$onHold',
                    },
                    messagesCount: {
                        $sum: 1,
                    },
                },
            },
            {
                $project: {
                    _id: '$_id._id',
                    ts: '$_id.ts',
                    departmentId: '$_id.departmentId',
                    open: '$_id.open',
                    servedBy: '$_id.servedBy',
                    metrics: '$_id.metrics',
                    msgs: '$messagesCount',
                    onHold: '$_id.onHold',
                },
            },
        ], { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    countOpenByAgent(userId, extraQuery = {}) {
        const query = {
            't': 'l',
            'open': true,
            'servedBy._id': userId,
            ...extraQuery,
        };
        return this.countDocuments(query);
    }
    findOpenByAgent(userId, extraQuery = {}, options = {}) {
        const query = {
            't': 'l',
            'open': true,
            'servedBy._id': userId,
            ...extraQuery,
        };
        return this.find(query, options);
    }
    changeAgentByRoomId(roomId, newAgent) {
        const query = {
            _id: roomId,
            t: 'l',
        };
        const update = {
            $set: {
                servedBy: {
                    _id: newAgent.agentId,
                    username: newAgent.username,
                    ts: newAgent.ts ?? new Date(),
                },
            },
        };
        return this.updateOne(query, update);
    }
    changeDepartmentIdByRoomId(roomId, departmentId) {
        const query = {
            _id: roomId,
            t: 'l',
        };
        const update = {
            $set: {
                departmentId,
            },
        };
        return this.updateOne(query, update);
    }
    saveCRMDataByRoomId(roomId, crmData) {
        const query = {
            _id: roomId,
            t: 'l',
        };
        const update = {
            $set: {
                crmData,
            },
        };
        return this.updateOne(query, update);
    }
    updateVisitorStatus(token, status) {
        const query = {
            'v.token': token,
            'open': true,
            't': 'l',
        };
        const update = {
            $set: {
                'v.status': status,
            },
        };
        return this.updateMany(query, update);
    }
    removeAgentByRoomId(roomId) {
        const query = {
            _id: roomId,
            t: 'l',
        };
        const update = {
            $set: { queuedAt: new Date() },
            $unset: { servedBy: 1 },
        };
        return this.updateOne(query, update);
    }
    removeByVisitorToken(token) {
        const query = {
            't': 'l',
            'v.token': token,
        };
        return this.deleteMany(query);
    }
    removeByVisitorId(_id) {
        const query = {
            't': 'l',
            'v._id': _id,
        };
        return this.deleteMany(query);
    }
    removeById(_id) {
        const query = {
            _id,
            t: 'l',
        };
        return this.deleteOne(query);
    }
    getVisitorLastMessageTsUpdateQueryByRoomId(lastMessageTs, updater = this.getUpdater()) {
        return updater.set('v.lastMessageTs', lastMessageTs);
    }
    setVisitorInactivityInSecondsById(roomId, visitorInactivity) {
        const query = {
            _id: roomId,
        };
        const update = {
            $set: {
                'metrics.visitorInactivity': visitorInactivity,
            },
        };
        return this.updateOne(query, update);
    }
    changeVisitorByRoomId(roomId, { _id, username, token }) {
        const query = {
            _id: roomId,
            t: 'l',
        };
        const update = {
            $set: {
                'v._id': _id,
                'v.username': username,
                'v.token': token,
            },
        };
        return this.updateOne(query, update);
    }
    unarchiveOneById(roomId) {
        const query = {
            _id: roomId,
            t: 'l',
        };
        const update = {
            $set: {
                open: true,
            },
            $unset: {
                servedBy: 1,
                closedAt: 1,
                closedBy: 1,
                closer: 1,
            },
        };
        return this.updateOne(query, update);
    }
    getVisitorActiveForPeriodUpdateQuery(period, updater = this.getUpdater()) {
        return updater.addToSet('v.activity', period);
    }
    async getMACStatisticsForPeriod(period) {
        return this.col
            .aggregate([
            {
                $match: {
                    't': 'l',
                    'v.activity': period,
                },
            },
            {
                $group: {
                    _id: {
                        source: {
                            $ifNull: ['$source.alias', '$source.type'],
                        },
                    },
                    contactsCount: {
                        $addToSet: '$v._id',
                    },
                    conversationsCount: {
                        $sum: 1,
                    },
                },
            },
            {
                $group: {
                    _id: null,
                    sources: {
                        $push: {
                            source: '$_id.source',
                            contactsCount: {
                                $size: '$contactsCount',
                            },
                            conversationsCount: '$conversationsCount',
                        },
                    },
                    totalContactsCount: {
                        $sum: {
                            $size: '$contactsCount',
                        },
                    },
                    totalConversationsCount: {
                        $sum: '$conversationsCount',
                    },
                },
            },
            {
                $project: {
                    _id: 0,
                    contactsCount: '$totalContactsCount',
                    conversationsCount: '$totalConversationsCount',
                    sources: 1,
                },
            },
        ])
            .toArray();
    }
    async getMACStatisticsBetweenDates(start, end) {
        return this.col
            .aggregate([
            {
                $match: {
                    't': 'l',
                    'v.activity': { $exists: true },
                    'ts': {
                        $gte: start,
                        $lt: end,
                    },
                },
            },
            {
                $group: {
                    _id: {
                        source: {
                            $ifNull: ['$source.alias', '$source.type'],
                        },
                    },
                    contactsCount: {
                        $addToSet: '$v._id',
                    },
                    conversationsCount: {
                        $sum: 1,
                    },
                },
            },
            {
                $group: {
                    _id: null,
                    sources: {
                        $push: {
                            source: '$_id.source',
                            contactsCount: {
                                $size: '$contactsCount',
                            },
                            conversationsCount: '$conversationsCount',
                        },
                    },
                    totalContactsCount: {
                        $sum: {
                            $size: '$contactsCount',
                        },
                    },
                    totalConversationsCount: {
                        $sum: '$conversationsCount',
                    },
                },
            },
            {
                $project: {
                    _id: 0,
                    contactsCount: '$totalContactsCount',
                    conversationsCount: '$totalConversationsCount',
                    sources: 1,
                },
            },
        ])
            .toArray();
    }
    countLivechatRoomsWithDepartment() {
        return this.countDocuments({ departmentId: { $exists: true } });
    }
    async unsetAllPredictedVisitorAbandonment() {
        throw new Error('Method not implemented.');
    }
    setOnHoldByRoomId(_roomId) {
        throw new Error('Method not implemented.');
    }
    unsetOnHoldByRoomId(_roomId) {
        throw new Error('Method not implemented.');
    }
    unsetOnHoldAndPredictedVisitorAbandonmentByRoomId(_roomId) {
        throw new Error('Method not implemented.');
    }
    setSlaForRoomById(_roomId, _sla) {
        throw new Error('Method not implemented.');
    }
    removeSlaFromRoomById(_roomId) {
        throw new Error('Method not implemented.');
    }
    bulkRemoveSlaFromRoomsById(_slaId) {
        throw new Error('Method not implemented.');
    }
    findOpenBySlaId(_slaId, _options) {
        throw new Error('Method not implemented.');
    }
    async setPriorityByRoomId(_roomId, _priority) {
        throw new Error('Method not implemented.');
    }
    async unsetPriorityByRoomId(_roomId) {
        throw new Error('Method not implemented.');
    }
    findOpenRoomsByPriorityId(_priorityId) {
        throw new Error('Method not implemented.');
    }
    getPredictedVisitorAbandonmentByRoomIdUpdateQuery(_willBeAbandonedAt, _updater) {
        throw new Error('Method not implemented.');
    }
    setPredictedVisitorAbandonmentByRoomId(_rid, _willBeAbandonedAt) {
        throw new Error('Method not implemented.');
    }
    findAbandonedOpenRooms(_date) {
        throw new Error('Method not implemented.');
    }
    async unsetPredictedVisitorAbandonmentByRoomId(_roomId) {
        throw new Error('Method not implemented.');
    }
    async associateRoomsWithDepartmentToUnit(_departments, _unitId) {
        throw new Error('Method not implemented.');
    }
    async removeUnitAssociationFromRooms(_unitId) {
        throw new Error('Method not implemented.');
    }
    async updateDepartmentAncestorsById(_rid, _departmentAncestors) {
        throw new Error('Method not implemented.');
    }
    countPrioritizedRooms() {
        throw new Error('Method not implemented.');
    }
    countRoomsWithSla() {
        throw new Error('Method not implemented.');
    }
    countRoomsWithTranscriptSent() {
        throw new Error('Method not implemented.');
    }
    getConversationsBySource(_start, _end, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    getConversationsByStatus(_start, _end, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    getConversationsByDepartment(_start, _end, _sort, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    getConversationsByTags(_start, _end, _sort, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    getConversationsByAgents(_start, _end, _sort, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    getConversationsWithoutTagsBetweenDate(_start, _end, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    getTotalConversationsWithoutAgentsBetweenDate(_start, _end, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    getTotalConversationsWithoutDepartmentBetweenDates(_start, _end, _extraQuery) {
        throw new Error('Method not implemented.');
    }
    setContactByVisitorAssociation(association, contact) {
        return this.updateMany({
            't': 'l',
            'v._id': association.visitorId,
            'source.type': association.source.type,
            ...(association.source.id ? { 'source.id': association.source.id } : {}),
        }, {
            $set: {
                contactId: contact._id,
                ...(contact.name ? { fname: contact.name } : {}),
            },
        });
    }
    updateContactDataByContactId(oldContactId, contact) {
        const update = {
            ...(contact._id ? { contactId: contact._id } : {}),
            ...(contact.name ? { fname: contact.name } : {}),
        };
        if (!Object.keys(update).length) {
            throw new Error('error-invalid-operation');
        }
        return this.updateMany({
            t: 'l',
            contactId: oldContactId,
        }, {
            $set: update,
        });
    }
    updateMergedContactIds(_contactIdsThatWereMerged, _newContactId, _options) {
        throw new Error('Method not implemented.');
    }
    findClosedRoomsByContactAndSourcePaginated(_params) {
        throw new Error('Method not implemented.');
    }
    findOpenByContactId(contactId, options) {
        return this.find({ open: true, contactId }, options);
    }
    checkContactOpenRooms(contactId) {
        return this.findOne({ contactId, open: true }, { projection: { _id: 1 } });
    }
}
exports.LivechatRoomsRaw = LivechatRoomsRaw;
//# sourceMappingURL=LivechatRooms.js.map