"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LivechatInquiryRaw = void 0;
const core_typings_1 = require("@rocket.chat/core-typings");
const BaseRaw_1 = require("./BaseRaw");
const readSecondaryPreferred_1 = require("../readSecondaryPreferred");
const { INQUIRY_LOCK_TIMEOUT = '10000' } = process.env;
class LivechatInquiryRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'livechat_inquiry', trash);
    }
    modelIndexes() {
        return [
            {
                key: {
                    rid: 1,
                },
            },
            {
                key: {
                    name: 1,
                },
            },
            {
                key: {
                    message: 1,
                },
            },
            {
                key: {
                    ts: 1,
                },
            },
            {
                key: {
                    department: 1,
                },
            },
            {
                key: {
                    status: 1,
                },
            },
            {
                key: {
                    priorityId: 1,
                    priorityWeight: 1,
                },
                sparse: true,
            },
            {
                key: {
                    priorityWeight: 1,
                    ts: 1,
                },
                partialFilterExpression: {
                    status: { $eq: core_typings_1.LivechatInquiryStatus.QUEUED },
                },
            },
            {
                key: {
                    estimatedWaitingTimeQueue: 1,
                    ts: 1,
                },
                partialFilterExpression: {
                    status: { $eq: core_typings_1.LivechatInquiryStatus.QUEUED },
                },
            },
            {
                key: {
                    'v.token': 1,
                    'status': 1,
                },
            },
            {
                key: {
                    locked: 1,
                    lockedAt: 1,
                },
                sparse: true,
            },
            { key: { 'v._id': 1 } },
        ];
    }
    findOneQueuedByRoomId(rid) {
        const query = {
            rid,
            status: core_typings_1.LivechatInquiryStatus.QUEUED,
        };
        return this.findOne(query);
    }
    findOneByRoomId(rid, options) {
        const query = {
            rid,
        };
        return this.findOne(query, options);
    }
    findOneReadyByRoomId(rid, options) {
        const query = {
            rid,
            status: core_typings_1.LivechatInquiryStatus.READY,
        };
        return this.findOne(query, options);
    }
    findIdsByVisitorToken(token) {
        return this.find({ 'v.token': token }, { projection: { _id: 1 } });
    }
    findIdsByVisitorId(_id) {
        return this.find({ 'v._id': _id }, { projection: { _id: 1 } });
    }
    getDistinctQueuedDepartments(options) {
        return this.col
            .aggregate([
            { $match: { status: core_typings_1.LivechatInquiryStatus.QUEUED } },
            {
                $group: {
                    _id: '$department',
                },
            },
        ], options)
            .toArray();
    }
    async setDepartmentByInquiryId(inquiryId, department) {
        return this.findOneAndUpdate({ _id: inquiryId }, { $set: { department } }, { returnDocument: 'after' });
    }
    /**
     * Updates the `lastMessage` of inquiries that are not taken yet, after they're taken we only need to update room's `lastMessage`
     */
    async setLastMessageByRoomId(rid, message) {
        return this.findOneAndUpdate({ rid, status: { $ne: core_typings_1.LivechatInquiryStatus.TAKEN } }, { $set: { lastMessage: message } }, { returnDocument: 'after' });
    }
    async setLastMessageById(inquiryId, lastMessage) {
        return this.updateOne({ _id: inquiryId }, { $set: { lastMessage } });
    }
    async findNextAndLock(queueSortBy, department) {
        const date = new Date();
        return this.findOneAndUpdate({
            status: core_typings_1.LivechatInquiryStatus.QUEUED,
            ...(department ? { department } : { department: { $exists: false } }),
            $or: [
                {
                    locked: true,
                    lockedAt: {
                        $lte: new Date(date.getTime() - parseInt(INQUIRY_LOCK_TIMEOUT)),
                    },
                },
                {
                    locked: { $ne: true },
                },
            ],
        }, {
            $set: {
                locked: true,
                // apply INQUIRY_LOCK_TIMEOUT secs lock lifetime
                lockedAt: new Date(),
            },
        }, {
            sort: queueSortBy,
        });
    }
    async unlock(inquiryId) {
        return this.updateOne({ _id: inquiryId }, { $unset: { locked: 1, lockedAt: 1 } });
    }
    async unlockAll() {
        return this.updateMany({ locked: { $exists: true } }, { $unset: { locked: 1, lockedAt: 1 }, $set: { status: core_typings_1.LivechatInquiryStatus.QUEUED, queuedAt: new Date() } });
    }
    async getCurrentSortedQueueAsync({ inquiryId, department, queueSortBy, }) {
        const filter = [
            {
                $match: {
                    status: 'queued',
                    ...(department && { department }),
                },
            },
            { $sort: queueSortBy },
            {
                $group: {
                    _id: 1,
                    inquiry: {
                        $push: {
                            _id: '$_id',
                            rid: '$rid',
                            name: '$name',
                            ts: '$ts',
                            status: '$status',
                            department: '$department',
                        },
                    },
                },
            },
            {
                $unwind: {
                    path: '$inquiry',
                    includeArrayIndex: 'position',
                },
            },
            {
                $project: {
                    _id: '$inquiry._id',
                    rid: '$inquiry.rid',
                    name: '$inquiry.name',
                    ts: '$inquiry.ts',
                    status: '$inquiry.status',
                    department: '$inquiry.department',
                    position: 1,
                },
            },
        ];
        // To get the current room position in the queue, we need to apply the next $match after the $project
        if (inquiryId) {
            filter.push({ $match: { _id: inquiryId } });
        }
        return this.col
            .aggregate(filter, {
            readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)(),
        })
            .toArray();
    }
    setSlaForRoom(_rid, _data) {
        throw new Error('Method not implemented on the community edition.');
    }
    unsetSlaForRoom(_roomId) {
        throw new Error('Method not implemented on the community edition.');
    }
    bulkUnsetSla(_roomIds) {
        throw new Error('Method not implemented on the community edition.');
    }
    setPriorityForRoom(_rid, _priority) {
        throw new Error('Method not implemented on the community edition.');
    }
    unsetPriorityForRoom(_rid) {
        throw new Error('Method not implemented on the community edition.');
    }
    async removeByRoomId(rid, options) {
        return this.deleteOne({ rid }, options);
    }
    getQueuedInquiries(options) {
        return this.find({ status: core_typings_1.LivechatInquiryStatus.QUEUED }, options);
    }
    takeInquiry(inquiryId, lockedAt) {
        return this.updateOne({
            _id: inquiryId,
            ...(lockedAt && { lockedAt }),
        }, {
            $set: { status: core_typings_1.LivechatInquiryStatus.TAKEN, takenAt: new Date() },
            $unset: { defaultAgent: 1, estimatedInactivityCloseTimeAt: 1, queuedAt: 1 },
        });
    }
    openInquiry(inquiryId) {
        return this.updateOne({
            _id: inquiryId,
        }, {
            $set: { status: core_typings_1.LivechatInquiryStatus.OPEN },
        });
    }
    async queueInquiry(inquiryId, lastMessage, defaultAgent) {
        return this.findOneAndUpdate({
            _id: inquiryId,
        }, {
            $set: {
                status: core_typings_1.LivechatInquiryStatus.QUEUED,
                queuedAt: new Date(),
                ...(lastMessage && { lastMessage }),
                ...(defaultAgent && { defaultAgent }),
            },
            $unset: { takenAt: 1 },
        }, { returnDocument: 'after' });
    }
    queueInquiryAndRemoveDefaultAgent(inquiryId) {
        return this.updateOne({
            _id: inquiryId,
        }, {
            $set: { status: core_typings_1.LivechatInquiryStatus.QUEUED, queuedAt: new Date() },
            $unset: { takenAt: 1, defaultAgent: 1 },
        });
    }
    readyInquiry(inquiryId) {
        return this.updateOne({
            _id: inquiryId,
        }, {
            $set: {
                status: core_typings_1.LivechatInquiryStatus.READY,
            },
        });
    }
    async changeDepartmentIdByRoomId(rid, department) {
        const query = {
            rid,
        };
        const updateObj = {
            $set: {
                department,
            },
        };
        return this.updateOne(query, updateObj);
    }
    async getStatus(inquiryId) {
        return (await this.findOne({ _id: inquiryId }))?.status;
    }
    updateVisitorStatus(token, status) {
        const query = {
            'v.token': token,
            'status': core_typings_1.LivechatInquiryStatus.QUEUED,
        };
        const update = {
            $set: {
                'v.status': status,
            },
        };
        return this.updateOne(query, update);
    }
    setDefaultAgentById(inquiryId, defaultAgent) {
        return this.updateOne({
            _id: inquiryId,
        }, {
            $set: {
                defaultAgent,
            },
        });
    }
    async setStatusById(inquiryId, status) {
        const result = await this.findOneAndUpdate({ _id: inquiryId }, { $set: { status } }, {
            upsert: true,
            returnDocument: 'after',
        });
        if (!result) {
            throw new Error('error-failed-to-set-inquiry-status');
        }
        return result;
    }
    setNameByRoomId(rid, name) {
        const query = { rid };
        const update = {
            $set: {
                name,
            },
        };
        return this.updateOne(query, update);
    }
    findOneByToken(token) {
        const query = {
            'v.token': token,
            'status': core_typings_1.LivechatInquiryStatus.QUEUED,
        };
        return this.findOne(query);
    }
    removeDefaultAgentById(inquiryId) {
        return this.updateOne({
            _id: inquiryId,
        }, {
            $unset: { defaultAgent: 1 },
        });
    }
    async removeByVisitorToken(token) {
        const query = {
            'v.token': token,
        };
        await this.deleteMany(query);
    }
    async markInquiryActiveForPeriod(rid, period) {
        return this.findOneAndUpdate({ rid }, { $addToSet: { 'v.activity': period } });
    }
    updateNameByVisitorIds(visitorIds, name) {
        const query = { 'v._id': { $in: visitorIds } };
        const update = {
            $set: { name },
        };
        return this.updateMany(query, update);
    }
    findByVisitorIds(visitorIds, options) {
        return this.find({ 'v._id': { $in: visitorIds } }, options);
    }
}
exports.LivechatInquiryRaw = LivechatInquiryRaw;
//# sourceMappingURL=LivechatInquiry.js.map