"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LivechatAgentActivityRaw = void 0;
const date_fns_1 = require("date-fns");
const BaseRaw_1 = require("./BaseRaw");
const readSecondaryPreferred_1 = require("../readSecondaryPreferred");
class LivechatAgentActivityRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'livechat_agent_activity', trash);
    }
    modelIndexes() {
        return [{ key: { date: 1 } }, { key: { agentId: 1, date: 1 }, unique: true }];
    }
    findOneByAgendIdAndDate(agentId, date) {
        return this.findOne({ agentId, date });
    }
    async createOrUpdate(data = {}) {
        const { date, agentId, lastStartedAt } = data;
        if (!date || !agentId) {
            return null;
        }
        return this.findOneAndUpdate({ agentId, date }, {
            $unset: {
                lastStoppedAt: 1,
            },
            $set: {
                lastStartedAt: lastStartedAt || new Date(),
            },
            $setOnInsert: {
                date,
                agentId,
            },
        }, { upsert: true });
    }
    updateLastStoppedAt({ agentId, date, lastStoppedAt, availableTime, }) {
        const query = {
            agentId,
            date,
        };
        const update = {
            $inc: { availableTime },
            $set: {
                lastStoppedAt,
            },
        };
        return this.updateMany(query, update);
    }
    updateServiceHistory({ agentId, date, serviceHistory, }) {
        const query = {
            agentId,
            date,
        };
        const update = {
            $addToSet: {
                serviceHistory,
            },
        };
        return this.updateMany(query, update);
    }
    findOpenSessions() {
        const query = {
            lastStoppedAt: { $exists: false },
        };
        return this.find(query);
    }
    findAllAverageAvailableServiceTime({ date, departmentId }) {
        const match = { $match: { date } };
        const lookup = {
            $lookup: {
                from: 'rocketchat_livechat_department_agents',
                localField: 'agentId',
                foreignField: 'agentId',
                as: 'departments',
            },
        };
        const unwind = {
            $unwind: {
                path: '$departments',
                preserveNullAndEmptyArrays: true,
            },
        };
        const departmentsMatch = {
            $match: {
                'departments.departmentId': departmentId,
            },
        };
        const sumAvailableTimeWithCurrentTime = {
            $sum: [{ $divide: [{ $subtract: [new Date(), '$lastStartedAt'] }, 1000] }, '$availableTime'],
        };
        const group = {
            $group: {
                _id: null,
                allAvailableTimeInSeconds: {
                    $sum: {
                        $cond: [{ $ifNull: ['$lastStoppedAt', false] }, '$availableTime', sumAvailableTimeWithCurrentTime],
                    },
                },
                rooms: { $sum: 1 },
            },
        };
        const project = {
            $project: {
                averageAvailableServiceTimeInSeconds: {
                    $trunc: {
                        $cond: [{ $eq: ['$rooms', 0] }, 0, { $divide: ['$allAvailableTimeInSeconds', '$rooms'] }],
                    },
                },
            },
        };
        const params = [match];
        if (departmentId && (departmentId !== 'undefined' || departmentId !== undefined)) {
            params.push(lookup);
            params.push(unwind);
            params.push(departmentsMatch);
        }
        params.push(group);
        params.push(project);
        return this.col
            .aggregate(params, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() })
            .toArray();
    }
    findAvailableServiceTimeHistory({ start, end, fullReport, onlyCount = false, options = {}, }) {
        const match = {
            $match: {
                date: {
                    $gte: parseInt((0, date_fns_1.format)((0, date_fns_1.parseISO)(start), 'yyyyMMdd')),
                    $lte: parseInt((0, date_fns_1.format)((0, date_fns_1.parseISO)(end), 'yyyyMMdd')),
                },
            },
        };
        const lookup = {
            $lookup: {
                from: 'users',
                localField: 'agentId',
                foreignField: '_id',
                as: 'user',
            },
        };
        const unwind = {
            $unwind: {
                path: '$user',
            },
        };
        const group = {
            $group: {
                _id: { _id: '$user._id', username: '$user.username' },
                serviceHistory: { $first: '$serviceHistory' },
                availableTimeInSeconds: { $sum: '$availableTime' },
            },
        };
        const project = {
            $project: {
                _id: 0,
                username: '$_id.username',
                availableTimeInSeconds: 1,
                ...(fullReport && { serviceHistory: 1 }),
            },
        };
        const sort = { $sort: options.sort || { username: 1 } };
        const params = [match, lookup, unwind, group, project, sort];
        if (onlyCount) {
            params.push({ $count: 'total' });
            return this.col.aggregate(params);
        }
        if (options.offset) {
            params.push({ $skip: options.offset });
        }
        if (options.count) {
            params.push({ $limit: options.count });
        }
        return this.col.aggregate(params, { allowDiskUse: true, readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
}
exports.LivechatAgentActivityRaw = LivechatAgentActivityRaw;
//# sourceMappingURL=LivechatAgentActivity.js.map