"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SipServerSession = void 0;
const random_1 = require("@rocket.chat/random");
const drachtio_srf_1 = __importDefault(require("drachtio-srf"));
const errorCodes_1 = require("./errorCodes");
const logger_1 = require("../logger");
const IncomingSipCall_1 = require("./providers/IncomingSipCall");
const OutgoingSipCall_1 = require("./providers/OutgoingSipCall");
const getDefaultSettings_1 = require("../server/getDefaultSettings");
class SipServerSession {
    get sessionId() {
        return this._sessionId;
    }
    constructor() {
        this.wasEverEnabled = false;
        this._sessionId = random_1.Random.id();
        this.knownCalls = new Map();
        this.srf = new drachtio_srf_1.default();
        // Always instantiate it with the default settings as it stays disconnected until explicitly configured
        this.settings = (0, getDefaultSettings_1.getDefaultSettings)();
        this.initializeDrachtio();
    }
    reactToCallUpdate(params) {
        const { callId, ...otherParams } = params;
        const sipCall = this.knownCalls.get(callId);
        if (!sipCall) {
            // If we don't know this call, then it's probably being handled by a session in some other server instance
            return;
        }
        sipCall.reactToCallChanges(otherParams).catch((error) => {
            logger_1.logger.error({ msg: 'Failed to react to call changes', error, call: sipCall.call, otherParams });
        });
    }
    reportInternalCallUpdate(callId) {
        logger_1.logger.debug({ msg: 'SipServerSession.reportInternalCallUpdate', callId });
    }
    registerCall(call) {
        this.knownCalls.set(call.callId, call);
    }
    configure(settings) {
        this.settings = settings;
        if (!this.isEnabledOnSettings(settings)) {
            return;
        }
        if (!this.wasEverEnabled) {
            this.connectDrachtio();
        }
    }
    async createOutgoingCall(params) {
        return OutgoingSipCall_1.OutgoingSipCall.createCall(this, params);
    }
    async createSipDialog(sipExtension, opts, progressCallbacks) {
        const uri = this.getExtensionUri(sipExtension);
        return this.srf.createUAC(uri, opts, progressCallbacks);
    }
    geContactUri(contact) {
        const sipExtension = contact.sipExtension || (contact.type === 'sip' && contact.id) || null;
        const username = contact.username && `user-${contact.username}`;
        const userId = contact.id && `user-${contact.id}`;
        const sipUsername = sipExtension || username || userId;
        return this.getExtensionUri(sipUsername || 'unknown');
    }
    getExtensionUri(extension) {
        const { host, port } = this.settings.sip.sipServer;
        if (!host) {
            throw new Error('Sip Server Host is not configured');
        }
        const portStr = port ? `:${port}` : '';
        return `sip:${extension}@${host}${portStr}`;
    }
    stripDrachtioServerDetails(reqOrRes) {
        const { _agent, socket: _socket, _req, _res, ...data } = reqOrRes;
        return data;
    }
    isEnabledOnSettings(settings) {
        return Boolean(settings.enabled && settings.sip.enabled && settings.sip.drachtio.host && settings.sip.drachtio.secret);
    }
    initializeDrachtio() {
        logger_1.logger.debug('Initializing Drachtio');
        this.srf.on('connect', (err, hostport) => {
            if (err) {
                logger_1.logger.error({ msg: 'Drachtio Connection Failed', err });
                return;
            }
            logger_1.logger.debug({ msg: 'Connected to a drachtio server', hostport });
        });
        this.srf.on('error', (err, socket) => this.onDrachtioError(err, socket));
        this.srf.use((req, _res, next) => {
            logger_1.logger.debug({ msg: 'Incoming message from Drachtio', method: req.method, source: req.source_address });
            next();
        });
        this.srf.invite((req, res) => {
            logger_1.logger.debug('Received a call on drachtio.');
            void this.processInvite(req, res).catch((error) => {
                logger_1.logger.error({ msg: 'Error processing Drachtio Invite', error });
            });
        });
    }
    connectDrachtio() {
        if (this.wasEverEnabled) {
            return;
        }
        const { host, port = 9022, secret } = this.settings.sip.drachtio;
        logger_1.logger.info({ msg: 'Connecting to drachtio', host, port });
        this.wasEverEnabled = true;
        this.srf.connect({
            host,
            port,
            secret,
        });
    }
    async processInvite(req, res) {
        if (!this.isEnabledOnSettings(this.settings)) {
            res.send(errorCodes_1.SipErrorCodes.SERVICE_NOT_AVAILABLE);
            return;
        }
        const sipCall = await IncomingSipCall_1.IncomingSipCall.processInvite(this, this.srf, req, res).catch((e) => {
            this.forwardSipExceptionToResponse(e, res);
            throw e;
        });
        this.registerCall(sipCall);
    }
    forwardSipExceptionToResponse(exception, res) {
        if (!exception || typeof exception !== 'object') {
            return;
        }
        if (!(exception instanceof errorCodes_1.SipError)) {
            return;
        }
        res.send(exception.sipErrorCode);
    }
    onDrachtioError(error, socket) {
        logger_1.logger.error({ msg: 'Drachtio Service Error', error });
        if (this.isEnabledOnSettings(this.settings)) {
            return;
        }
        try {
            this.srf.disconnect(socket);
        }
        catch {
            // Supress errors on socket disconnection
        }
    }
}
exports.SipServerSession = SipServerSession;
//# sourceMappingURL=Session.js.map