"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediaCallServer = void 0;
const emitter_1 = require("@rocket.chat/emitter");
const media_signaling_1 = require("@rocket.chat/media-signaling");
const CallDirector_1 = require("./CallDirector");
const getDefaultSettings_1 = require("./getDefaultSettings");
const common_1 = require("../definition/common");
const InternalCallProvider_1 = require("../internal/InternalCallProvider");
const SignalProcessor_1 = require("../internal/SignalProcessor");
const logger_1 = require("../logger");
const Session_1 = require("../sip/Session");
/**
 * Class used as gateway to send and receive signals to/from clients
 * The actual function used to send the signals needs to be set by the server
 */
class MediaCallServer {
    constructor() {
        this.emitter = new emitter_1.Emitter();
        this.session = new Session_1.SipServerSession();
        this.settings = (0, getDefaultSettings_1.getDefaultSettings)();
        this.signalProcessor = new SignalProcessor_1.GlobalSignalProcessor();
        this.signalProcessor.emitter.on('signalRequest', ({ toUid, signal }) => {
            // Forward signal requests from the signal processor to the server
            this.sendSignal(toUid, signal);
        });
        this.signalProcessor.emitter.on('callRequest', ({ params }) => {
            this.requestCall(params).catch(() => null);
        });
    }
    receiveSignal(fromUid, signal) {
        logger_1.logger.debug({ msg: 'MediaCallServer.receiveSignal', type: signal.type, fromUid });
        if (!(0, media_signaling_1.isClientMediaSignal)(signal)) {
            logger_1.logger.error({ msg: 'The Media Signal Server received an invalid client signal object' });
            throw new Error('invalid-signal');
        }
        this.signalProcessor.processSignal(fromUid, signal).catch((error) => {
            logger_1.logger.error({ msg: 'Failed to process client signal', error, type: signal.type });
        });
    }
    sendSignal(toUid, signal) {
        logger_1.logger.debug({ msg: 'MediaCallServer.sendSignal', toUid, type: signal.type });
        this.emitter.emit('signalRequest', { toUid, signal });
    }
    reportCallUpdate(params) {
        logger_1.logger.debug({ msg: 'MediaCallServer.reportCallUpdate', params });
        this.emitter.emit('callUpdated', params);
    }
    updateCallHistory(params) {
        logger_1.logger.debug({ msg: 'MediaCallServer.updateCallHistory', params });
        this.emitter.emit('historyUpdate', params);
    }
    async requestCall(params) {
        try {
            const fullParams = await this.parseCallContacts(params);
            await this.createCall(fullParams);
        }
        catch (error) {
            let rejectionReason = 'unsupported';
            if (error && typeof error === 'object' && error instanceof common_1.CallRejectedError) {
                rejectionReason = error.callRejectedReason;
            }
            else {
                logger_1.logger.error({ msg: 'Failed to create a requested call', params, error });
            }
            const originalId = params.requestedCallId || params.parentCallId;
            if (originalId && params.requestedBy?.type === 'user') {
                logger_1.logger.info({ msg: 'Call Request Rejected', uid: params.requestedBy.id, rejectionReason });
                this.sendSignal(params.requestedBy.id, {
                    type: 'rejected-call-request',
                    callId: originalId,
                    toContractId: params.requestedBy.contractId,
                    reason: rejectionReason,
                });
            }
            else {
                throw error;
            }
        }
    }
    async createCall(params) {
        logger_1.logger.debug({ msg: 'MediaCallServer.createCall', params });
        if (params.callee.type === 'sip') {
            await this.session.createOutgoingCall(params);
            return;
        }
        await InternalCallProvider_1.InternalCallProvider.createCall(params);
    }
    receiveCallUpdate(params) {
        this.session.reactToCallUpdate(params);
    }
    async hangupExpiredCalls() {
        return CallDirector_1.mediaCallDirector.hangupExpiredCalls();
    }
    scheduleExpirationCheck() {
        CallDirector_1.mediaCallDirector.scheduleExpirationCheck();
    }
    configure(settings) {
        logger_1.logger.debug({ msg: 'Media Server Configuration' });
        this.session.configure(settings);
        this.settings = settings;
    }
    async permissionCheck(uid, callType) {
        return this.settings.permissionCheck(uid, callType);
    }
    /**
     * Receives params for a call a client wishes to do, with actors needing only their basic identification
     * Returns params for a call that should actually be done, according to server routing rules
     * Returned value also include full contact information for the actors, when such information is available on the server
     *
     * Will throw if a call can't be routed or if one of the user lacks permission for it.
     * Blocked permissions do not affect the routing rules, meaning a call may be blocked even if it would have been allowed through a different route.
     * */
    async parseCallContacts(params) {
        logger_1.logger.debug({ msg: 'MediaCallServer.parseCallContacts', params });
        // On call transfers, do not mutate the caller
        // On new calls, force the caller type to be 'user' (since the call is being created in rocket.chat first)
        const isTransfer = Boolean(params.parentCallId);
        const callerRequiredType = isTransfer ? params.caller.type : 'user';
        const requester = params.requestedBy || params.caller;
        // Internal and outgoing calls must have been requested by an internal user;
        // Incoming calls should not be passing through this function.
        if (requester.type !== 'user') {
            logger_1.logger.debug('Invalid call requester');
            throw new common_1.CallRejectedError('invalid-call-params');
        }
        // If this user can't make any call at all, fail early to avoid leaking if the callee is valid.
        if (!(await this.settings.permissionCheck(requester.id, 'any'))) {
            logger_1.logger.debug({ msg: 'User with no permission requested a call.', uid: requester.id });
            throw new common_1.CallRejectedError('forbidden');
        }
        const caller = await CallDirector_1.mediaCallDirector.cast.getContactForActor(params.caller, { requiredType: callerRequiredType });
        if (!caller) {
            logger_1.logger.debug('Failed to load caller contact information');
            throw new common_1.CallRejectedError('invalid-call-params');
        }
        // The callee contact type will determine if the call is going to go through SIP or directly to another rocket.chat user
        const callee = await CallDirector_1.mediaCallDirector.cast.getContactForActor(params.callee, this.getCalleeContactOptions());
        if (!callee) {
            logger_1.logger.debug('Failed to load callee contact information');
            throw new common_1.CallRejectedError('invalid-call-params');
        }
        if (this.settings.internalCalls.requireExtensions && !callee.sipExtension) {
            logger_1.logger.debug('Invalid target user');
            throw new common_1.CallRejectedError('invalid-call-params');
        }
        if (callee.type === 'user') {
            if (!(await this.settings.permissionCheck(requester.id, 'internal'))) {
                logger_1.logger.debug('Requester lacks permission');
                throw new common_1.CallRejectedError('forbidden');
            }
            if (!(await this.settings.permissionCheck(callee.id, 'internal'))) {
                logger_1.logger.debug('Callee lacks permission');
                throw new common_1.CallRejectedError('forbidden');
            }
            if (caller.type === 'user' && caller.id !== requester.id && !(await this.settings.permissionCheck(caller.id, 'internal'))) {
                logger_1.logger.debug('Caller lacks permission');
                throw new common_1.CallRejectedError('forbidden');
            }
        }
        else {
            if (caller.type !== 'user') {
                logger_1.logger.debug('Invalid call direction: user initiating a sip->user call');
                throw new common_1.CallRejectedError('invalid-call-params');
            }
            if (!(await this.settings.permissionCheck(requester.id, 'external'))) {
                logger_1.logger.debug('Requester lacks permission');
                throw new common_1.CallRejectedError('forbidden');
            }
        }
        return {
            ...params,
            caller: {
                ...caller,
                contractId: params.caller.contractId,
            },
            callee,
        };
    }
    getCalleeContactOptions() {
        if (!this.settings.sip.enabled) {
            return {
                requiredType: 'user',
            };
        }
        switch (this.settings.internalCalls.routeExternally) {
            case 'always':
                // Will only make sip calls
                return {
                    requiredType: 'sip',
                };
            case 'never':
                // Will not use sip when calling an user or an extension assigned to an user, but will use sip when calling an unassigned extension
                return {
                    preferredType: 'user',
                };
            case 'preferably':
                // Will only skip sip for users that don't have an assigned extension (or not call at all if `requireExtensions` is true)
                return {
                    preferredType: 'sip',
                };
        }
        return {};
    }
}
exports.MediaCallServer = MediaCallServer;
//# sourceMappingURL=MediaCallServer.js.map