"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediaCallCastDirector = void 0;
const models_1 = require("@rocket.chat/models");
const UserActorAgent_1 = require("../internal/agents/UserActorAgent");
const logger_1 = require("../logger");
const BroadcastAgent_1 = require("./BroadcastAgent");
class MediaCallCastDirector {
    async getAgentsFromCall(call) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getAgentsFromCall', callId: call?._id });
        const callerAgent = await this.getAgentFromCall(call, 'caller');
        if (!callerAgent) {
            throw new Error('Unable to find caller agent');
        }
        const calleeAgent = await this.getAgentFromCall(call, 'callee');
        if (!calleeAgent) {
            throw new Error('Unable to find callee agent');
        }
        callerAgent.oppositeAgent = calleeAgent;
        calleeAgent.oppositeAgent = callerAgent;
        return {
            caller: callerAgent,
            callee: calleeAgent,
        };
    }
    async getAgentFromCall(call, role) {
        const { [role]: actor } = call;
        return this.getAgentForActorAndRole(actor, role);
    }
    async getContactForActor(actor, options, defaultContactInfo) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getContactForActor', actor, options, defaultContactInfo });
        if (actor.type === 'user') {
            return this.getContactForUserId(actor.id, options, { ...actor, ...defaultContactInfo });
        }
        if (actor.type === 'sip') {
            return this.getContactForExtensionNumber(actor.id, options, { ...actor, ...defaultContactInfo });
        }
        return null;
    }
    getContactForUser(user, options, defaultContactInfo) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getContactForUser', user, options, defaultContactInfo });
        const actors = this.buildContactListForUser(user, defaultContactInfo);
        return this.getContactFromList(actors, options);
    }
    async getContactForUserId(userId, options, defaultContactInfo) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getContactForUserId', userId, options, defaultContactInfo });
        const user = await models_1.Users.findOneById(userId, {
            projection: { name: 1, username: 1, freeSwitchExtension: 1 },
        });
        if (!user) {
            throw new Error('invalid-callee');
        }
        return this.getContactForUser(user, options, defaultContactInfo);
    }
    async getContactForExtensionNumber(sipExtension, options, defaultContactInfo) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getContactForExtensionNumber', sipExtension, options, defaultContactInfo });
        const user = await models_1.Users.findOneByFreeSwitchExtension(sipExtension, {
            projection: { name: 1, username: 1, freeSwitchExtension: 1 },
        });
        const list = user
            ? this.buildContactListForUser(user, defaultContactInfo)
            : this.buildContactListForExtension(sipExtension, defaultContactInfo);
        return this.getContactFromList(list, options);
    }
    async getAgentForActorAndRole(actor, role) {
        if (actor.type === 'user') {
            return this.getAgentForUserActorAndRole(actor, role);
        }
        if (actor.type === 'sip') {
            return this.getAgentForSipActorAndRole(actor, role);
        }
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getAgentForActorAndRole.null', actor, role });
        return null;
    }
    async getAgentForUserActorAndRole(actor, role) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getAgentForUserActorAndRole', id: actor.id, role });
        return new UserActorAgent_1.UserActorAgent(actor, role);
    }
    async getAgentForSipActorAndRole(actor, role) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getAgentForSipActorAndRole', id: actor.id, role });
        return new BroadcastAgent_1.BroadcastActorAgent(actor, role);
    }
    buildContactListForUser(user, defaultContactInfo) {
        const { name: displayName, username, freeSwitchExtension: sipExtension, _id: id } = user;
        const data = {
            ...defaultContactInfo,
            ...(displayName && { displayName }),
            ...(username && { username }),
            ...(sipExtension && { sipExtension }),
        };
        return {
            user: {
                ...data,
                type: 'user',
                id,
            },
            sip: sipExtension
                ? {
                    ...data,
                    type: 'sip',
                    id: sipExtension,
                }
                : null,
        };
    }
    buildContactListForExtension(sipExtension, defaultContactInfo) {
        const data = {
            ...defaultContactInfo,
            ...(sipExtension && { sipExtension }),
        };
        return {
            user: null,
            sip: {
                ...data,
                type: 'sip',
                id: sipExtension,
            },
        };
    }
    getContactFromList(list, options) {
        logger_1.logger.debug({ msg: 'MediaCallCastDirector.getContactFromList', list, options });
        if (options.requiredType) {
            return list[options.requiredType] ?? null;
        }
        const preferredActor = options.preferredType && list[options.preferredType];
        return preferredActor || list.user || list.sip || null;
    }
}
exports.MediaCallCastDirector = MediaCallCastDirector;
//# sourceMappingURL=CastDirector.js.map