"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateLicenseUrl = validateLicenseUrl;
const crypto_1 = __importDefault(require("crypto"));
const isItemAllowed_1 = require("../isItemAllowed");
const logger_1 = require("../logger");
const getResultingBehavior_1 = require("./getResultingBehavior");
const validateRegex = (licenseURL, url) => {
    licenseURL = licenseURL
        .replace(/\./g, '\\.') // convert dots to literal
        .replace(/\*/g, '.*'); // convert * to .*
    const regex = new RegExp(`^${licenseURL}$`, 'i');
    return !!regex.exec(url);
};
const validateUrl = (licenseURL, url) => {
    return licenseURL.toLowerCase() === url.toLowerCase();
};
const validateHash = (licenseURL, url) => {
    const value = crypto_1.default.createHash('sha256').update(url).digest('hex');
    return licenseURL === value;
};
function validateLicenseUrl(license, options) {
    if (!(0, isItemAllowed_1.isBehaviorAllowed)('invalidate_license', options)) {
        return [];
    }
    const { validation: { serverUrls }, } = license;
    const workspaceUrl = this.getWorkspaceUrl();
    if (!workspaceUrl) {
        logger_1.logger.error('Unable to validate license URL without knowing the workspace URL.');
        return [(0, getResultingBehavior_1.getResultingBehavior)({ behavior: 'invalidate_license' }, { reason: 'url' })];
    }
    return serverUrls
        .filter((url) => {
        switch (url.type) {
            case 'regex':
                return !validateRegex(url.value, workspaceUrl);
            case 'hash':
                return !validateHash(url.value, workspaceUrl);
            case 'url':
                return !validateUrl(url.value, workspaceUrl);
        }
        return false;
    })
        .map((url) => {
        if (!options.suppressLog) {
            logger_1.logger.error({
                msg: 'Url validation failed',
                url,
                workspaceUrl,
            });
        }
        return (0, getResultingBehavior_1.getResultingBehavior)({ behavior: 'invalidate_license' }, { reason: 'url' });
    });
}
//# sourceMappingURL=validateLicenseUrl.js.map