"use strict";
/**
 * FromV2ToV3
 * Transform a License V2 into a V3 representation.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.convertToV3 = void 0;
const bundles_1 = require("./bundles");
const getTagColor_1 = require("./getTagColor");
const convertToV3 = (v2) => {
    return {
        version: '3.0',
        information: {
            autoRenew: false,
            ...((v2.meta?.trialEnd || v2.expiry) && { visualExpiration: new Date(Date.parse(v2.meta?.trialEnd || v2.expiry)).toISOString() }),
            trial: v2.meta?.trial || false,
            offline: false,
            createdAt: new Date().toISOString(),
            grantedBy: {
                method: 'manual',
                seller: 'V2',
            },
            // if no tag present, it means it is an old license, so try check for bundles and use them as tags
            tags: v2.tag
                ? [v2.tag]
                : [
                    ...v2.modules.filter(bundles_1.isBundle).map(bundles_1.getBundleFromModule).filter(Boolean).map((tag) => ({
                        name: tag,
                        color: (0, getTagColor_1.getTagColor)(tag),
                    })),
                ],
        },
        validation: {
            serverUrls: [
                {
                    value: v2.url,
                    type: 'regex',
                },
            ],
            validPeriods: [
                {
                    validUntil: new Date(Date.parse(v2.expiry)).toISOString(),
                    invalidBehavior: 'invalidate_license',
                },
            ],
            statisticsReport: {
                required: true,
            },
        },
        grantedModules: [
            ...new Set(['outbound-messaging', 'teams-voip', 'contact-id-verification', 'hide-watermark', ...v2.modules]
                .map((licenseModule) => ((0, bundles_1.isBundle)(licenseModule) ? (0, bundles_1.getBundleModules)(licenseModule) : [licenseModule]))
                .reduce((prev, curr) => [...prev, ...curr], [])
                .map((licenseModule) => ({ module: licenseModule }))),
        ],
        limits: {
            ...(v2.maxActiveUsers
                ? {
                    activeUsers: [
                        {
                            max: v2.maxActiveUsers,
                            behavior: 'prevent_action',
                        },
                    ],
                }
                : {}),
            ...(v2.maxGuestUsers
                ? {
                    guestUsers: [
                        {
                            max: v2.maxGuestUsers,
                            behavior: 'prevent_action',
                        },
                    ],
                }
                : {}),
            ...(v2.maxRoomsPerGuest
                ? {
                    roomsPerGuest: [
                        {
                            max: v2.maxRoomsPerGuest,
                            behavior: 'prevent_action',
                        },
                    ],
                }
                : {}),
            ...(v2.apps?.maxPrivateApps
                ? {
                    privateApps: [
                        {
                            max: v2.apps.maxPrivateApps,
                            behavior: 'prevent_action',
                        },
                    ],
                }
                : {}),
            ...(v2.apps?.maxMarketplaceApps
                ? {
                    marketplaceApps: [
                        {
                            max: v2.apps.maxMarketplaceApps,
                            behavior: 'prevent_action',
                        },
                    ],
                }
                : {}),
        },
        cloudMeta: v2.meta,
    };
};
exports.convertToV3 = convertToV3;
//# sourceMappingURL=convertToV3.js.map