"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Router = exports.AbstractRouter = void 0;
const logger_1 = require("@rocket.chat/logger");
const express_1 = __importDefault(require("express"));
const hono_1 = require("hono");
const qs_1 = __importDefault(require("qs")); // Using qs specifically to keep express compatibility
const honoAdapterForExpress_1 = require("./middlewares/honoAdapterForExpress");
const logger = new logger_1.Logger('HttpRouter');
function splitArray(arr) {
    const last = arr[arr.length - 1];
    const rest = arr.slice(0, -1);
    return [rest, last];
}
function coerceDatesToStrings(obj) {
    if (Array.isArray(obj)) {
        return obj.map(coerceDatesToStrings);
    }
    if (obj && typeof obj === 'object') {
        const newObj = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value instanceof Date) {
                newObj[key] = value.toISOString();
            }
            else {
                newObj[key] = coerceDatesToStrings(value);
            }
        }
        return newObj;
    }
    return obj;
}
class AbstractRouter {
}
exports.AbstractRouter = AbstractRouter;
class Router extends AbstractRouter {
    constructor(base) {
        super();
        this.base = base;
        this.typedRoutes = {};
        this.innerRouter = new hono_1.Hono();
    }
    registerTypedRoutes(method, subpath, options) {
        const path = `/${this.base}/${subpath}`.replaceAll('//', '/');
        this.typedRoutes = this.typedRoutes || {};
        this.typedRoutes[path] = this.typedRoutes[subpath] || {};
        const { query, response = {}, authRequired, body, tags, ...rest } = options;
        this.typedRoutes[path][method.toLowerCase()] = {
            responses: Object.fromEntries(Object.entries(response).map(([status, schema]) => [
                parseInt(status, 10),
                {
                    description: '',
                    content: {
                        'application/json': { schema: ('schema' in schema ? schema.schema : schema) },
                    },
                },
            ])),
            ...(query && {
                parameters: [
                    {
                        schema: query.schema,
                        in: 'query',
                        name: 'query',
                        required: true,
                    },
                ],
            }),
            ...(body && {
                requestBody: {
                    required: true,
                    content: {
                        'application/json': { schema: body.schema },
                    },
                },
            }),
            ...(authRequired && {
                ...rest,
                security: [
                    {
                        userId: [],
                        authToken: [],
                    },
                ],
            }),
            tags,
        };
    }
    async parseBodyParams({ request }) {
        try {
            let parsedBody = {};
            const contentType = request.header('content-type');
            if (contentType?.includes('application/json')) {
                parsedBody = await request.raw.clone().json();
            }
            else if (contentType?.includes('multipart/form-data')) {
                parsedBody = await request.raw.clone().formData();
            }
            else if (contentType?.includes('application/x-www-form-urlencoded')) {
                const req = await request.raw.clone().formData();
                parsedBody = Object.fromEntries(req.entries());
            }
            else {
                parsedBody = await request.raw.clone().text();
            }
            // This is necessary to keep the compatibility with the previous version, otherwise the bodyParams will be an empty string when no content-type is sent
            if (parsedBody === '') {
                return {};
            }
            if (Array.isArray(parsedBody)) {
                return parsedBody;
            }
            return { ...parsedBody };
            // eslint-disable-next-line no-empty
        }
        catch { }
        return {};
    }
    parseQueryParams(request) {
        return qs_1.default.parse(request.raw.url.split('?')?.[1] || '');
    }
    method(method, subpath, options, ...actions) {
        const [middlewares, action] = splitArray(actions);
        const convertedAction = this.convertActionToHandler(action);
        this.innerRouter[method.toLowerCase()](`/${subpath}`.replace('//', '/'), ...middlewares, async (c) => {
            const { req, res } = c;
            const queryParams = this.parseQueryParams(req);
            if (options.query) {
                const validatorFn = options.query;
                if (typeof options.query === 'function' && !validatorFn(queryParams)) {
                    logger.warn({
                        msg: 'Query parameters validation failed - route spec does not match request payload',
                        method: req.method,
                        path: req.url,
                        error: validatorFn.errors?.map((error) => error.message).join('\n '),
                        bodyParams: undefined,
                        queryParams,
                    });
                    return c.json({
                        success: false,
                        errorType: 'error-invalid-params',
                        error: validatorFn.errors?.map((error) => error.message).join('\n '),
                    }, 400);
                }
            }
            const bodyParams = await this.parseBodyParams({ request: req });
            if (options.body) {
                const validatorFn = options.body;
                if (typeof options.body === 'function' && !validatorFn(req.bodyParams || bodyParams)) {
                    logger.warn({
                        msg: 'Request body validation failed - route spec does not match request payload',
                        method: req.method,
                        path: req.url,
                        error: validatorFn.errors?.map((error) => error.message).join('\n '),
                        bodyParams,
                        queryParams: undefined,
                    });
                    return c.json({
                        success: false,
                        errorType: 'invalid-params',
                        error: validatorFn.errors?.map((error) => error.message).join('\n '),
                    }, 400);
                }
            }
            const response = await convertedAction(c);
            const { body, statusCode, headers } = response;
            if (process.env.NODE_ENV === 'test' || process.env.TEST_MODE) {
                const responseValidatorFn = options?.response?.[statusCode];
                /* c8 ignore next 3 */
                if (!responseValidatorFn && options.typed) {
                    throw new Error(`Missing response validator for endpoint ${req.method} - ${req.url} with status code ${statusCode}`);
                }
                if (responseValidatorFn && !responseValidatorFn(coerceDatesToStrings(body))) {
                    logger.warn({
                        msg: 'Response validation failed - response does not match route spec',
                        method: req.method,
                        path: req.url,
                        error: responseValidatorFn.errors?.map((error) => error.message).join('\n '),
                        originalResponse: body,
                    });
                    return c.json({
                        success: false,
                        errorType: 'error-invalid-body',
                        error: `Invalid response for endpoint ${req.method} - ${req.url}. Error: ${responseValidatorFn.errors
                            ?.map((error) => `${error.message} (${[
                            error.instancePath,
                            Object.entries(error.params)
                                .map(([key, value]) => `${key}: ${value}`)
                                .join(', '),
                        ]
                            .filter(Boolean)
                            .join(' - ')})`)
                            .join('\n')}`,
                    }, 400);
                }
            }
            const responseHeaders = Object.fromEntries(Object.entries({
                ...res.headers,
                'Content-Type': 'application/json',
                'Cache-Control': 'no-store',
                'Pragma': 'no-cache',
                ...headers,
            }).map(([key, value]) => [key.toLowerCase(), value]));
            const contentType = (responseHeaders['content-type'] || 'application/json');
            const isContentLess = (statusCode) => {
                return [101, 204, 205, 304].includes(statusCode);
            };
            if (isContentLess(statusCode)) {
                return c.status(statusCode);
            }
            Object.entries(responseHeaders).forEach(([key, value]) => {
                if (value) {
                    c.header(key, String(value));
                }
            });
            return c.body((contentType?.match(/json|javascript/) ? JSON.stringify(body) : body), statusCode);
        });
        this.registerTypedRoutes(method, subpath, options);
        return this;
    }
    convertActionToHandler(action) {
        // Default implementation simply passes through the action
        // Subclasses can override this to provide custom handling
        return action;
    }
    get(subpath, options, ...action) {
        return this.method('GET', subpath, options, ...action);
    }
    post(subpath, options, ...action) {
        return this.method('POST', subpath, options, ...action);
    }
    put(subpath, options, ...action) {
        return this.method('PUT', subpath, options, ...action);
    }
    delete(subpath, options, ...action) {
        return this.method('DELETE', subpath, options, ...action);
    }
    use(innerRouter) {
        if (innerRouter instanceof Router) {
            this.typedRoutes = {
                ...this.typedRoutes,
                ...Object.fromEntries(Object.entries(innerRouter.typedRoutes).map(([path, routes]) => [`${this.base}${path}`, routes])),
            };
            this.innerRouter.route(innerRouter.base, innerRouter.innerRouter);
        }
        if (typeof innerRouter === 'function') {
            this.innerRouter.use(innerRouter);
        }
        return this;
    }
    get router() {
        // eslint-disable-next-line new-cap
        const router = express_1.default.Router();
        const hono = new hono_1.Hono();
        router.use(this.base, (0, honoAdapterForExpress_1.honoAdapterForExpress)(hono.route(this.base, this.innerRouter).options('*', (c) => {
            return c.body('OK');
        })));
        return router;
    }
    getHonoRouter() {
        return this.innerRouter;
    }
}
exports.Router = Router;
//# sourceMappingURL=Router.js.map