import { describe, expect, it } from '@jest/globals';

import { parseEventData } from '../../src/eventParser/parseEventData';
import type { EventData } from '../../src/eventParser/parseEventData';

function createEventData(data: Record<string, string | string[] | undefined>): EventData {
	return data as unknown as EventData;
}

describe('parseEventData', () => {
	describe('simple tests', () => {
		it('should handle missing required fields', () => {
			const eventName = 'CHANNEL_CREATE';
			const eventData = createEventData({
				'Channel-Name': 'sofia/internal/1001@192.168.1.100',
				'Channel-State': 'CS_INIT',
			});

			const result = parseEventData(eventName, eventData);

			expect(result).toBeUndefined();
		});

		it('should handle invalid sequence number', () => {
			const eventName = 'CHANNEL_CREATE';
			const eventData = createEventData({
				'Channel-Name': 'sofia/internal/1001@192.168.1.100',
				'Channel-State': 'CS_INIT',
				'Event-Sequence': 'invalid',
				'Unique-ID': 'test-channel-id',
			});

			const result = parseEventData(eventName, eventData);

			expect(result).toBeUndefined();
		});
	});

	it('channel with no name, states or timestamp', () => {
		const eventName = 'CHANNEL_DESTROY';
		const eventData = createEventData({
			'Channel-State-Number': '9',
			'Channel-Call-State-Number': '9',
			'Event-Sequence': '1',
			'Unique-ID': 'test-channel-id',
			'Hangup-Cause': 'NORMAL_CLEARING',
			'variable_undefined': undefined,
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.channelName).toBe('');
		expect(result?.channelState).toBe('');
		expect(result?.channelCallState).toBe('');
		expect(result?.hangupCause).toBe('NORMAL_CLEARING');
		expect(result?.firedAt).toBeInstanceOf(Date);
	});

	it('should parse CHANNEL_BRIDGE event', () => {
		const eventName = 'CHANNEL_BRIDGE';
		const eventData = createEventData({
			'Channel-Name': 'sofia/internal/1001@192.168.1.100',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'CS_EXECUTE',
			'Channel-State-Number': '4',
			'Channel-Call-State-Number': '4',
			'Event-Sequence': '1',
			'Event-Date-Timestamp': '1749847568969044',
			'Unique-ID': 'test-channel-id',
			'Bridge-A-Unique-ID': 'bridge-a-id',
			'Bridge-B-Unique-ID': 'bridge-b-id',
			'Bridged-To': 'other-channel-id',
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'test-channel-id',
			eventName: 'CHANNEL_BRIDGE',
			sequence: 1,
			firedAt: new Date(1749847568969),
			callUniqueId: 'test-channel-id',
			channelName: 'sofia/internal/1001@192.168.1.100',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallStateNumber: '4',
			channelCallState: 'CS_EXECUTE',
			channelUsername: '1001',
			bridgeUniqueIds: ['bridge-a-id', 'bridge-b-id'],
			bridgedTo: 'other-channel-id',
			metadata: { 'Event-Sequence': '1', 'Event-Date-Timestamp': '1749847568969044' },
		});
	});

	it('should parse CHANNEL_HANGUP event', () => {
		const eventName = 'CHANNEL_HANGUP';
		const eventData = createEventData({
			'Channel-Name': 'sofia/internal/1001@192.168.1.100',
			'Channel-State': 'CS_HANGUP',
			'Channel-Call-State': 'CS_HANGUP',
			'Channel-State-Number': '6',
			'Channel-Call-State-Number': '6',
			'Event-Sequence': '1',
			'Event-Date-Timestamp': '1749847568969044',
			'Unique-ID': 'test-channel-id',
			'Hangup-Cause': 'NORMAL_CLEARING',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'test-channel-id',
			eventName: 'CHANNEL_HANGUP',
			sequence: 1,
			firedAt: new Date(1749847568969),
			callUniqueId: 'test-channel-id',
			channelName: 'sofia/internal/1001@192.168.1.100',
			channelState: 'CS_HANGUP',
			channelStateNumber: '6',
			channelCallStateNumber: '6',
			channelCallState: 'CS_HANGUP',
			channelUsername: '1001',
			hangupCause: 'NORMAL_CLEARING',
			metadata: { 'Event-Sequence': '1', 'Event-Date-Timestamp': '1749847568969044' },
		});
	});

	it('should parse CHANNEL_ANSWER event', () => {
		const eventName = 'CHANNEL_ANSWER';
		const eventData = createEventData({
			'Channel-Name': 'sofia/internal/1001@192.168.1.100',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'CS_EXECUTE',
			'Channel-State-Number': '4',
			'Channel-Call-State-Number': '4',
			'Event-Sequence': '1',
			'Event-Date-Timestamp': '1749847568969044',
			'Unique-ID': 'test-channel-id',
			'Answer-State': 'answered',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'test-channel-id',
			eventName: 'CHANNEL_ANSWER',
			sequence: 1,
			firedAt: new Date(1749847568969),
			callUniqueId: 'test-channel-id',
			channelName: 'sofia/internal/1001@192.168.1.100',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallStateNumber: '4',
			channelCallState: 'CS_EXECUTE',
			channelUsername: '1001',
			answerState: 'answered',
			metadata: { 'Event-Sequence': '1', 'Event-Date-Timestamp': '1749847568969044' },
		});
	});

	it('should parse CHANNEL_PROGRESS_MEDIA event', () => {
		const eventName = 'CHANNEL_PROGRESS_MEDIA';
		const eventData = createEventData({
			'Channel-Name': 'sofia/internal/1001@192.168.1.100',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'CS_EXECUTE',
			'Channel-State-Number': '4',
			'Channel-Call-State-Number': '4',
			'Event-Sequence': '1',
			'Event-Date-Timestamp': '1749847568969044',
			'Unique-ID': 'test-channel-id',
			'Channel-Read-Codec-Name': 'PCMU',
			'Channel-Read-Codec-Rate': '8000',
			'Channel-Write-Codec-Name': 'PCMU',
			'Channel-Write-Codec-Rate': '8000',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'test-channel-id',
			eventName: 'CHANNEL_PROGRESS_MEDIA',
			sequence: 1,
			firedAt: new Date(1749847568969),
			callUniqueId: 'test-channel-id',
			channelName: 'sofia/internal/1001@192.168.1.100',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallStateNumber: '4',
			channelCallState: 'CS_EXECUTE',
			channelUsername: '1001',
			metadata: { 'Event-Sequence': '1', 'Event-Date-Timestamp': '1749847568969044' },
			codecs: { read: { name: 'PCMU', rate: '8000' }, write: { nme: 'PCMU', rate: '8000' } },
		});
	});

	it('should parse CHANNEL_DESTROY event', () => {
		const eventName = 'CHANNEL_DESTROY';
		const eventData = createEventData({
			'Channel-Name': 'sofia/internal/1001@192.168.1.100',
			'Channel-State': 'CS_DESTROY',
			'Channel-Call-State': 'CS_DESTROY',
			'Channel-State-Number': '9',
			'Channel-Call-State-Number': '9',
			'Event-Sequence': '1',
			'Event-Date-Timestamp': '1749847568969044',
			'Unique-ID': 'test-channel-id',
			'Hangup-Cause': 'NORMAL_CLEARING',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'test-channel-id',
			eventName: 'CHANNEL_DESTROY',
			sequence: 1,
			firedAt: new Date(1749847568969),
			callUniqueId: 'test-channel-id',
			channelName: 'sofia/internal/1001@192.168.1.100',
			channelState: 'CS_DESTROY',
			channelStateNumber: '9',
			channelCallStateNumber: '9',
			channelCallState: 'CS_DESTROY',
			channelUsername: '1001',
			hangupCause: 'NORMAL_CLEARING',
			metadata: { 'Event-Sequence': '1', 'Event-Date-Timestamp': '1749847568969044' },
		});
	});
});

describe('complete call', () => {
	it('should parse CS_NEW CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39834',
			'Channel-State': 'CS_NEW',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '0',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_STATE',
			sequence: 39834,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_NEW',
			channelCallState: 'DOWN',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847568969044',
				'Event-Sequence': '39834',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: undefined,
		});
		expect(result?.legs).toBeUndefined();
		expect(result?.raw).toBeUndefined();
	});

	it('should parse CS_INIT CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39835',
			'Channel-State': 'CS_INIT',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '1',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_STATE',
			sequence: 39835,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_INIT',
			channelStateNumber: '1',
			channelCallState: 'DOWN',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847568969044',
				'Event-Sequence': '39835',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});
		expect(result?.raw).toBeUndefined();
	});

	it('should parse CHANNEL_CREATE event', () => {
		const eventName = 'CHANNEL_CREATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_CREATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39836',
			'Channel-State': 'CS_INIT',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_ep_codec_string':
				'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
			'variable_endpoint_disposition': 'DELAYED NEGOTIATION',
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_CREATE',
			sequence: 39836,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_INIT',
			channelStateNumber: '2',
			channelCallState: 'DOWN',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_CREATE',
				'Event-Date-Timestamp': '1749847568969044',
				'Event-Sequence': '39836',
			},
			variables: {
				direction: 'inbound',
				ep_codec_string:
					'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
				endpoint_disposition: 'DELAYED NEGOTIATION',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse RINGING CHANNEL_CALLSTATE event', () => {
		const eventName = 'CHANNEL_CALLSTATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_CALLSTATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39837',
			'Original-Channel-Call-State': 'DOWN',
			'Channel-Call-State-Number': '2',
			'Channel-State': 'CS_ROUTING',
			'Channel-Call-State': 'RINGING',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_CALLSTATE',
			sequence: 39837,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_ROUTING',
			channelStateNumber: '2',
			channelCallStateNumber: '2',
			channelCallState: 'RINGING',
			originalChannelCallState: 'DOWN',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_CALLSTATE',
				'Event-Date-Timestamp': '1749847568969044',
				'Event-Sequence': '39837',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CS_ROUTING CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39838',
			'Channel-State': 'CS_ROUTING',
			'Channel-Call-State': 'RINGING',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_STATE',
			sequence: 39838,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_ROUTING',
			channelStateNumber: '2',
			channelCallState: 'RINGING',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847568969044',
				'Event-Sequence': '39838',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CS_EXECUTE CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39841',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'RINGING',
			'Channel-State-Number': '4',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_STATE',
			sequence: 39841,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallState: 'RINGING',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847568969044',
				'Event-Sequence': '39841',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CHANNEL_OUTGOING event', () => {
		const eventName = 'CHANNEL_OUTGOING';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_OUTGOING',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39852',
			'Channel-State': 'CS_NONE',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '1',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Call-UUID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_OUTGOING',
			sequence: 39852,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_NONE',
			channelStateNumber: '1',
			channelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': { profileIndex: '1', profileCreatedTime: new Date(1749847568969), channelCreatedTime: new Date(1749847568969) },
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: { 'Event-Name': 'CHANNEL_OUTGOING', 'Event-Date-Timestamp': '1749847568969044', 'Event-Sequence': '39852' },
			raw: { 'Channel-Call-UUID': '31450e18-1531-46a0-9c10-4629110e3d23' },
			presenceCallDirection: 'outbound',
			caller: '1001',
			callee: '1000',
		});
	});

	it('should parse outbound CS_INIT CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39853',
			'Channel-State': 'CS_INIT',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '1',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_STATE',
			sequence: 39853,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_INIT',
			channelStateNumber: '1',
			channelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847568969044',
				'Event-Sequence': '39853',
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CHANNEL_CREATE event', () => {
		const eventName = 'CHANNEL_CREATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_CREATE',
			'Event-Date-Timestamp': '1749847569008164',
			'Event-Sequence': '39854',
			'Channel-State': 'CS_INIT',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_sip_invite_domain': '172.99.99.99',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_CREATE',
			sequence: 39854,
			firedAt: new Date(1749847569008),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_INIT',
			channelStateNumber: '2',
			channelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_CREATE',
				'Event-Date-Timestamp': '1749847569008164',
				'Event-Sequence': '39854',
			},
			variables: {
				direction: 'outbound',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				sip_invite_domain: '172.99.99.99',
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CHANNEL_ORIGINATE event', () => {
		const eventName = 'CHANNEL_ORIGINATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_ORIGINATE',
			'Event-Date-Timestamp': '1749847569008164',
			'Event-Sequence': '39855',
			'Channel-State': 'CS_INIT',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_ORIGINATE',
			sequence: 39855,
			firedAt: new Date(1749847569008),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_INIT',
			channelStateNumber: '2',
			channelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_ORIGINATE',
				'Event-Date-Timestamp': '1749847569008164',
				'Event-Sequence': '39855',
			},
			variables: {
				direction: 'outbound',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CS_ROUTING CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847569008164',
			'Event-Sequence': '39856',
			'Channel-State': 'CS_ROUTING',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_STATE',
			sequence: 39856,
			firedAt: new Date(1749847569008),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_ROUTING',
			channelStateNumber: '2',
			channelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847569008164',
				'Event-Sequence': '39856',
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CS_CONSUME_MEDIA CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847569008164',
			'Event-Sequence': '39858',
			'Channel-State': 'CS_CONSUME_MEDIA',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '7',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_STATE',
			sequence: 39858,
			firedAt: new Date(1749847569008),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_CONSUME_MEDIA',
			channelStateNumber: '7',
			channelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847569008164',
				'Event-Sequence': '39858',
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CHANNEL_PROGRESS event', () => {
		const eventName = 'CHANNEL_PROGRESS';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_PROGRESS',
			'Event-Date-Timestamp': '1749847569148194',
			'Event-Sequence': '39859',
			'Channel-State': 'CS_CONSUME_MEDIA',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '7',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '1749847569148194',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_sip_invite_domain': '172.99.99.99',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_PROGRESS',
			sequence: 39859,
			firedAt: new Date(1749847569148),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_CONSUME_MEDIA',
			channelStateNumber: '7',
			channelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_PROGRESS',
				'Event-Date-Timestamp': '1749847569148194',
				'Event-Sequence': '39859',
			},
			variables: {
				direction: 'outbound',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				sip_invite_domain: '172.99.99.99',
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound RINGING CHANNEL_CALLSTATE event', () => {
		const eventName = 'CHANNEL_CALLSTATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_CALLSTATE',
			'Event-Date-Timestamp': '1749847569148194',
			'Event-Sequence': '39860',
			'Original-Channel-Call-State': 'DOWN',
			'Channel-Call-State-Number': '2',
			'Channel-State': 'CS_CONSUME_MEDIA',
			'Channel-Call-State': 'RINGING',
			'Channel-State-Number': '7',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '1749847569148194',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_CALLSTATE',
			sequence: 39860,
			firedAt: new Date(1749847569148),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_CONSUME_MEDIA',
			channelStateNumber: '7',
			channelCallStateNumber: '2',
			channelCallState: 'RINGING',
			originalChannelCallState: 'DOWN',
			channelUsername: '1000',
			answerState: 'ringing',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_CALLSTATE',
				'Event-Date-Timestamp': '1749847569148194',
				'Event-Sequence': '39860',
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CHANNEL_PROGRESS event', () => {
		const eventName = 'CHANNEL_PROGRESS';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_PROGRESS',
			'Event-Date-Timestamp': '1749847569168181',
			'Event-Sequence': '39861',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'RINGING',
			'Channel-State-Number': '4',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_extension': '1000',
			'variable_call_timeout': '30',
			'variable_hangup_after_bridge': 'true',
			'variable_current_application_data': 'user/1000@172.99.99.99',
			'variable_current_application': 'bridge',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_originate_disposition': 'failure',
			'variable_DIALSTATUS': 'INVALIDARGS',
			'variable_originated_legs': '31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'variable_sip_to_tag': 'rUtyD23mXy9Xj',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_PROGRESS',
			sequence: 39861,
			firedAt: new Date(1749847569168),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallState: 'RINGING',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_PROGRESS',
				'Event-Date-Timestamp': '1749847569168181',
				'Event-Sequence': '39861',
			},
			variables: {
				direction: 'inbound',
				uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_extension: '1000',
				call_timeout: '30',
				hangup_after_bridge: 'true',
				current_application_data: 'user/1000@172.99.99.99',
				current_application: 'bridge',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				originate_disposition: 'failure',
				DIALSTATUS: 'INVALIDARGS',
				originated_legs: '31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				sip_to_tag: 'rUtyD23mXy9Xj',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CHANNEL_ANSWER event', () => {
		const eventName = 'CHANNEL_ANSWER';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_ANSWER',
			'Event-Date-Timestamp': '1749847572228165',
			'Event-Sequence': '39867',
			'Channel-State': 'CS_CONSUME_MEDIA',
			'Channel-Call-State': 'RINGING',
			'Channel-State-Number': '7',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'answered',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572228165',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '1749847569148194',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_is_outbound': 'true',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_ANSWER',
			sequence: 39867,
			firedAt: new Date(1749847572228),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_CONSUME_MEDIA',
			channelStateNumber: '7',
			channelCallState: 'RINGING',
			channelUsername: '1000',
			answerState: 'answered',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572228),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_ANSWER',
				'Event-Date-Timestamp': '1749847572228165',
				'Event-Sequence': '39867',
			},
			variables: {
				direction: 'outbound',
				is_outbound: 'true',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CHANNEL_PROGRESS_MEDIA event', () => {
		const eventName = 'CHANNEL_PROGRESS_MEDIA';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_PROGRESS_MEDIA',
			'Event-Date-Timestamp': '1749847572249573',
			'Event-Sequence': '39872',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'RINGING',
			'Channel-State-Number': '4',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'early',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '1749847572249573',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_extension': '1000',
			'variable_call_timeout': '30',
			'variable_hangup_after_bridge': 'true',
			'variable_current_application_data': 'user/1000@172.99.99.99',
			'variable_current_application': 'bridge',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_originate_disposition': 'failure',
			'variable_DIALSTATUS': 'INVALIDARGS',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_PROGRESS_MEDIA',
			sequence: 39872,
			firedAt: new Date(1749847572249),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallState: 'RINGING',
			channelUsername: '1001',
			answerState: 'early',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelProgressMediaTime: new Date(1749847572249),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_PROGRESS_MEDIA',
				'Event-Date-Timestamp': '1749847572249573',
				'Event-Sequence': '39872',
			},
			variables: {
				direction: 'inbound',
				uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_extension: '1000',
				call_timeout: '30',
				hangup_after_bridge: 'true',
				current_application_data: 'user/1000@172.99.99.99',
				current_application: 'bridge',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				originate_disposition: 'failure',
				DIALSTATUS: 'INVALIDARGS',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CHANNEL_OUTGOING event', () => {
		const eventName = 'CHANNEL_OUTGOING';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_OUTGOING',
			'Event-Date-Timestamp': '1749847572928202',
			'Event-Sequence': '39881',
			'Channel-State': 'CS_CONSUME_MEDIA',
			'Channel-Call-State': 'ACTIVE',
			'Channel-State-Number': '7',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'answered',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572228165',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_OUTGOING',
			sequence: 39881,
			firedAt: new Date(1749847572928),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_CONSUME_MEDIA',
			channelStateNumber: '7',
			channelCallState: 'ACTIVE',
			channelUsername: '1000',
			answerState: 'answered',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelAnsweredTime: new Date(1749847572228),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_OUTGOING',
				'Event-Date-Timestamp': '1749847572928202',
				'Event-Sequence': '39881',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CHANNEL_BRIDGE event', () => {
		const eventName = 'CHANNEL_BRIDGE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_BRIDGE',
			'Event-Date-Timestamp': '1749847572928202',
			'Event-Sequence': '39882',
			'Bridge-A-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Bridge-B-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'ACTIVE',
			'Channel-State-Number': '4',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'answered',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572928202',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '1749847572249573',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originatee',
			'Other-Leg-Direction': 'outbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Other-Leg-Profile-Created-Time': '1749847568969044',
			'Other-Leg-Channel-Created-Time': '1749847568969044',
			'Other-Leg-Channel-Answered-Time': '1749847572228165',
			'Other-Leg-Channel-Progress-Time': '1749847569148194',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_session_id': '75',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_extension': '1000',
			'variable_call_timeout': '30',
			'variable_hangup_after_bridge': 'true',
			'variable_current_application_data': 'user/1000@172.99.99.99',
			'variable_current_application': 'bridge',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_originated_legs': [
				'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			],
			'variable_endpoint_disposition': 'ANSWER',
			'variable_originate_causes': ['31450e18-1531-46a0-9c10-4629110e3d23;NONE', '31450e18-1531-46a0-9c10-4629110e3d23;NONE'],
			'variable_originate_disposition': 'SUCCESS',
			'variable_DIALSTATUS': 'SUCCESS',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_BRIDGE',
			sequence: 39882,
			firedAt: new Date(1749847572928),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallState: 'ACTIVE',
			channelUsername: '1001',
			answerState: 'answered',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			bridgeUniqueIds: ['ebc91302-1b79-4ff3-ac6b-841385e9ea03', '31450e18-1531-46a0-9c10-4629110e3d23'],
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelAnsweredTime: new Date(1749847572928),
							channelBridgedTime: new Date(1749847572928),
							channelProgressMediaTime: new Date(1749847572249),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Other-Leg',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					type: 'originatee',
					direction: 'outbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelAnsweredTime: new Date(1749847572228),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_BRIDGE',
				'Event-Date-Timestamp': '1749847572928202',
				'Event-Sequence': '39882',
			},
			variables: {
				direction: 'inbound',
				uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				session_id: '75',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_extension: '1000',
				call_timeout: '30',
				hangup_after_bridge: 'true',
				current_application_data: 'user/1000@172.99.99.99',
				current_application: 'bridge',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				originated_legs: [
					'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				],
				endpoint_disposition: 'ANSWER',
				originate_causes: ['31450e18-1531-46a0-9c10-4629110e3d23;NONE', '31450e18-1531-46a0-9c10-4629110e3d23;NONE'],
				originate_disposition: 'SUCCESS',
				DIALSTATUS: 'SUCCESS',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});
	it('should parse outbound CS_EXCHANGE_MEDIA CHANNEL_STATE event', () => {
		const eventName = 'CHANNEL_STATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_STATE',
			'Event-Date-Timestamp': '1749847572928202',
			'Event-Sequence': '39883',
			'Channel-State': 'CS_EXCHANGE_MEDIA',
			'Channel-Call-State': 'ACTIVE',
			'Channel-State-Number': '5',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'answered',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572228165',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_STATE',
			sequence: 39883,
			firedAt: new Date(1749847572928),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_EXCHANGE_MEDIA',
			channelStateNumber: '5',
			channelCallState: 'ACTIVE',
			channelUsername: '1000',
			answerState: 'answered',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelAnsweredTime: new Date(1749847572228),
							channelBridgedTime: new Date(1749847572928),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_STATE',
				'Event-Date-Timestamp': '1749847572928202',
				'Event-Sequence': '39883',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CALL_UPDATE event', () => {
		const eventName = 'CALL_UPDATE';
		const eventData = createEventData({
			'Event-Name': 'CALL_UPDATE',
			'Event-Date-Timestamp': '1749847572928202',
			'Event-Sequence': '39884',
			'Direction': 'SEND',
			'Sent-Callee-ID-Name': '1001',
			'Sent-Callee-ID-Number': '1001',
			'Bridged-To': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Channel-State': 'CS_EXCHANGE_MEDIA',
			'Channel-Call-State': 'ACTIVE',
			'Channel-State-Number': '5',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'answered',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572228165',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_is_outbound': 'true',
			'variable_uuid': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_sip_invite_domain': '172.99.99.99',
			'variable_effective_caller_id_name': 'Extension 1000',
			'variable_effective_caller_id_number': '1000',
			'variable_outbound_caller_id_name': 'FreeSWITCH',
			'variable_outbound_caller_id_number': '0000000000',
			'variable_originate_early_media': 'true',
			'variable_originating_leg_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_originate_endpoint': 'user',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_bridge_to': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_last_bridge_to': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_bridge_channel': 'sofia/internal/1001@99.219.99.169:7443',
			'variable_bridge_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_signal_bond': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_last_sent_callee_id_name': '1001',
			'variable_last_sent_callee_id_number': '1001',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CALL_UPDATE',
			sequence: 39884,
			firedAt: new Date(1749847572928),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_EXCHANGE_MEDIA',
			channelStateNumber: '5',
			channelCallState: 'ACTIVE',
			channelUsername: '1000',
			answerState: 'answered',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			bridgedTo: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelAnsweredTime: new Date(1749847572228),
							channelBridgedTime: new Date(1749847572928),
							channelProgressTime: new Date(1749847569148),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CALL_UPDATE',
				'Event-Date-Timestamp': '1749847572928202',
				'Event-Sequence': '39884',
			},
			variables: {
				direction: 'outbound',
				is_outbound: 'true',
				uuid: '31450e18-1531-46a0-9c10-4629110e3d23',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				sip_invite_domain: '172.99.99.99',
				effective_caller_id_name: 'Extension 1000',
				effective_caller_id_number: '1000',
				outbound_caller_id_name: 'FreeSWITCH',
				outbound_caller_id_number: '0000000000',
				originate_early_media: 'true',
				originating_leg_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				originate_endpoint: 'user',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				bridge_to: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				last_bridge_to: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				bridge_channel: 'sofia/internal/1001@99.219.99.169:7443',
				bridge_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				signal_bond: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				last_sent_callee_id_name: '1001',
				last_sent_callee_id_number: '1001',
			},
			raw: {
				'Direction': 'SEND',
				'Sent-Callee-ID-Name': '1001',
				'Sent-Callee-ID-Number': '1001',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});
	});

	it('should parse CALL_UPDATE event', () => {
		const eventName = 'CALL_UPDATE';
		const eventData = createEventData({
			'Event-Name': 'CALL_UPDATE',
			'Event-Date-Timestamp': '1749847572928202',
			'Event-Sequence': '39886',
			'Direction': 'SEND',
			'Sent-Callee-ID-Name': 'Outbound Call',
			'Sent-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Bridged-To': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'ACTIVE',
			'Channel-State-Number': '4',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'answered',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572928202',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '1749847572249573',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originatee',
			'Other-Leg-Direction': 'outbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Other-Leg-Profile-Created-Time': '1749847568969044',
			'Other-Leg-Channel-Created-Time': '1749847568969044',
			'Other-Leg-Channel-Answered-Time': '1749847572228165',
			'Other-Leg-Channel-Progress-Time': '1749847569148194',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_session_id': '75',
			'variable_sip_from_user': '1001',
			'variable_sip_from_port': '7443',
			'variable_sip_from_uri': '1001@99.219.99.169:7443',
			'variable_sip_from_host': '99.219.99.169',
			'variable_channel_name': 'sofia/internal/1001@99.219.99.169:7443',
			'variable_sip_local_network_addr': '99.219.99.169',
			'variable_sip_network_ip': '177.99.99.183',
			'variable_sip_network_port': '60286',
			'variable_sip_invite_stamp': '1749847568969044',
			'variable_sip_received_ip': '177.99.99.183',
			'variable_sip_received_port': '60286',
			'variable_sip_via_protocol': 'wss',
			'variable_sip_authorized': 'true',
			'variable_sip_acl_authed_by': 'any_v4.auto',
			'variable_sip_from_user_stripped': '1001',
			'variable_sofia_profile_name': 'internal',
			'variable_sofia_profile_url': 'sip:mod_sofia@99.219.99.169:5060',
			'variable_recovery_profile_name': 'internal',
			'variable_sip_allow': 'ACK, CANCEL, INVITE, MESSAGE, BYE, OPTIONS, INFO, NOTIFY, REFER',
			'variable_sip_req_user': '1000',
			'variable_sip_req_port': '7443',
			'variable_sip_req_uri': '1000@99.219.99.169:7443',
			'variable_sip_req_host': '99.219.99.169',
			'variable_sip_to_user': '1000',
			'variable_sip_to_port': '7443',
			'variable_sip_to_uri': '1000@99.219.99.169:7443',
			'variable_sip_to_host': '99.219.99.169',
			'variable_sip_contact_params': 'transport=ws;ob',
			'variable_sip_contact_user': '1001-user1-j7a419rq',
			'variable_sip_contact_uri': '1001-user1-j7a419rq@localhost',
			'variable_sip_contact_host': 'localhost',
			'variable_sip_user_agent': 'SIP.js/0.21.1',
			'variable_sip_via_host': 'localhost',
			'variable_sip_via_rport': '60286',
			'variable_max_forwards': '70',
			'variable_ep_codec_string':
				'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_extension': '1000',
			'variable_call_timeout': '30',
			'variable_hangup_after_bridge': 'true',
			'variable_current_application_data': 'user/1000@172.99.99.99',
			'variable_current_application': 'bridge',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_originated_legs': [
				'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			],
			'variable_video_media_flow': 'inactive',
			'variable_text_media_flow': 'inactive',
			'variable_rtp_use_codec_string': 'OPUS,G722,PCMU,PCMA,H264,VP8',
			'variable_remote_video_media_flow': 'inactive',
			'variable_remote_text_media_flow': 'inactive',
			'variable_remote_audio_media_flow': 'sendrecv',
			'variable_audio_media_flow': 'sendrecv',
			'variable_rtp_audio_recv_pt': '109',
			'variable_rtp_use_codec_name': 'opus',
			'variable_rtp_use_codec_fmtp': 'maxplaybackrate=48000;stereo=1;useinbandfec=1',
			'variable_rtp_use_codec_rate': '48000',
			'variable_rtp_use_codec_ptime': '20',
			'variable_rtp_use_codec_channels': '2',
			'variable_rtp_last_audio_codec_string': 'opus@48000h@20i@2c',
			'variable_read_codec': 'opus',
			'variable_original_read_codec': 'opus',
			'variable_read_rate': '48000',
			'variable_original_read_rate': '48000',
			'variable_write_codec': 'opus',
			'variable_write_rate': '48000',
			'variable_remote_audio_ip': '192.168.2.192',
			'variable_remote_audio_port': '54922',
			'variable_remote_audio_rtcp_ip': '192.168.2.192',
			'variable_remote_audio_rtcp_port': '54922',
			'variable_dtmf_type': 'rfc2833',
			'variable_local_media_ip': '172.99.99.99',
			'variable_local_media_port': '16388',
			'variable_advertised_media_ip': '99.219.99.169',
			'variable_rtp_use_timer_name': 'soft',
			'variable_rtp_use_pt': '109',
			'variable_rtp_use_ssrc': '1219061913',
			'variable_rtp_2833_send_payload': '101',
			'variable_rtp_2833_recv_payload': '101',
			'variable_remote_media_ip': '192.168.2.192',
			'variable_remote_media_port': '54922',
			'variable_sip_nat_detected': 'true',
			'variable_sip_to_tag': 'rUtyD23mXy9Xj',
			'variable_sip_from_tag': 'tgtj08fqre',
			'variable_sip_cseq': '1',
			'variable_sip_call_id': '8lapd1dj47tfei96vl6q',
			'variable_sip_full_via': 'SIP/2.0/WSS localhost;branch=z9hG4bK7118990;received=177.99.99.183;rport=60286',
			'variable_sip_full_from': '<sip:1001@99.219.99.169:7443>;tag=tgtj08fqre',
			'variable_sip_full_to': '<sip:1000@99.219.99.169:7443>;tag=rUtyD23mXy9Xj',
			'variable_send_silence_when_idle': '-1',
			'variable_rtp_has_crypto': 'srtp:dtls:AES_CM_128_HMAC_SHA1_80',
			'variable_endpoint_disposition': 'ANSWER',
			'variable_originate_causes': ['31450e18-1531-46a0-9c10-4629110e3d23;NONE', '31450e18-1531-46a0-9c10-4629110e3d23;NONE'],
			'variable_originate_disposition': 'SUCCESS',
			'variable_DIALSTATUS': 'SUCCESS',
			'variable_bridge_to': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_last_bridge_to': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_bridge_channel': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'variable_bridge_uuid': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_signal_bond': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_last_sent_callee_id_name': '1001',
			'variable_last_sent_callee_id_number': '1001',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CALL_UPDATE',
			sequence: 39886,
			firedAt: new Date(1749847572928),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallState: 'ACTIVE',
			channelUsername: '1001',
			answerState: 'answered',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			bridgedTo: '31450e18-1531-46a0-9c10-4629110e3d23',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572928),
							channelProgressMediaTime: new Date(1749847572249),
							channelBridgedTime: new Date(1749847572928),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Other-Leg',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					type: 'originatee',
					direction: 'outbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572228),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CALL_UPDATE',
				'Event-Date-Timestamp': '1749847572928202',
				'Event-Sequence': '39886',
			},
			variables: {
				direction: 'inbound',
				uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				session_id: '75',
				sip_from_user: '1001',
				sip_from_port: '7443',
				sip_from_uri: '1001@99.219.99.169:7443',
				sip_from_host: '99.219.99.169',
				channel_name: 'sofia/internal/1001@99.219.99.169:7443',
				sip_local_network_addr: '99.219.99.169',
				sip_network_ip: '177.99.99.183',
				sip_network_port: '60286',
				sip_invite_stamp: '1749847568969044',
				sip_received_ip: '177.99.99.183',
				sip_received_port: '60286',
				sip_via_protocol: 'wss',
				sip_authorized: 'true',
				sip_acl_authed_by: 'any_v4.auto',
				sip_from_user_stripped: '1001',
				sofia_profile_name: 'internal',
				sofia_profile_url: 'sip:mod_sofia@99.219.99.169:5060',
				recovery_profile_name: 'internal',
				sip_allow: 'ACK, CANCEL, INVITE, MESSAGE, BYE, OPTIONS, INFO, NOTIFY, REFER',
				sip_req_user: '1000',
				sip_req_port: '7443',
				sip_req_uri: '1000@99.219.99.169:7443',
				sip_req_host: '99.219.99.169',
				sip_to_user: '1000',
				sip_to_port: '7443',
				sip_to_uri: '1000@99.219.99.169:7443',
				sip_to_host: '99.219.99.169',
				sip_contact_params: 'transport=ws;ob',
				sip_contact_user: '1001-user1-j7a419rq',
				sip_contact_uri: '1001-user1-j7a419rq@localhost',
				sip_contact_host: 'localhost',
				sip_user_agent: 'SIP.js/0.21.1',
				sip_via_host: 'localhost',
				sip_via_rport: '60286',
				max_forwards: '70',
				ep_codec_string:
					'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_extension: '1000',
				call_timeout: '30',
				hangup_after_bridge: 'true',
				current_application_data: 'user/1000@172.99.99.99',
				current_application: 'bridge',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				originated_legs: [
					'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				],
				video_media_flow: 'inactive',
				text_media_flow: 'inactive',
				rtp_use_codec_string: 'OPUS,G722,PCMU,PCMA,H264,VP8',
				remote_video_media_flow: 'inactive',
				remote_text_media_flow: 'inactive',
				remote_audio_media_flow: 'sendrecv',
				audio_media_flow: 'sendrecv',
				rtp_audio_recv_pt: '109',
				rtp_use_codec_name: 'opus',
				rtp_use_codec_fmtp: 'maxplaybackrate=48000;stereo=1;useinbandfec=1',
				rtp_use_codec_rate: '48000',
				rtp_use_codec_ptime: '20',
				rtp_use_codec_channels: '2',
				rtp_last_audio_codec_string: 'opus@48000h@20i@2c',
				read_codec: 'opus',
				original_read_codec: 'opus',
				read_rate: '48000',
				original_read_rate: '48000',
				write_codec: 'opus',
				write_rate: '48000',
				remote_audio_ip: '192.168.2.192',
				remote_audio_port: '54922',
				remote_audio_rtcp_ip: '192.168.2.192',
				remote_audio_rtcp_port: '54922',
				dtmf_type: 'rfc2833',
				local_media_ip: '172.99.99.99',
				local_media_port: '16388',
				advertised_media_ip: '99.219.99.169',
				rtp_use_timer_name: 'soft',
				rtp_use_pt: '109',
				rtp_use_ssrc: '1219061913',
				rtp_2833_send_payload: '101',
				rtp_2833_recv_payload: '101',
				remote_media_ip: '192.168.2.192',
				remote_media_port: '54922',
				sip_nat_detected: 'true',
				sip_to_tag: 'rUtyD23mXy9Xj',
				sip_from_tag: 'tgtj08fqre',
				sip_cseq: '1',
				sip_call_id: '8lapd1dj47tfei96vl6q',
				sip_full_via: 'SIP/2.0/WSS localhost;branch=z9hG4bK7118990;received=177.99.99.183;rport=60286',
				sip_full_from: '<sip:1001@99.219.99.169:7443>;tag=tgtj08fqre',
				sip_full_to: '<sip:1000@99.219.99.169:7443>;tag=rUtyD23mXy9Xj',
				send_silence_when_idle: '-1',
				rtp_has_crypto: 'srtp:dtls:AES_CM_128_HMAC_SHA1_80',
				endpoint_disposition: 'ANSWER',
				originate_causes: ['31450e18-1531-46a0-9c10-4629110e3d23;NONE', '31450e18-1531-46a0-9c10-4629110e3d23;NONE'],
				originate_disposition: 'SUCCESS',
				DIALSTATUS: 'SUCCESS',
				bridge_to: '31450e18-1531-46a0-9c10-4629110e3d23',
				last_bridge_to: '31450e18-1531-46a0-9c10-4629110e3d23',
				bridge_channel: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
				bridge_uuid: '31450e18-1531-46a0-9c10-4629110e3d23',
				signal_bond: '31450e18-1531-46a0-9c10-4629110e3d23',
				last_sent_callee_id_name: '1001',
				last_sent_callee_id_number: '1001',
			},
			raw: {
				'Direction': 'SEND',
				'Sent-Callee-ID-Name': 'Outbound Call',
				'Sent-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});
	});

	it('should parse outbound CHANNEL_HANGUP event', () => {
		const eventName = 'CHANNEL_HANGUP';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_HANGUP',
			'Event-Date-Timestamp': '1749847575568205',
			'Event-Sequence': '39893',
			'Channel-State': 'CS_EXCHANGE_MEDIA',
			'Channel-Call-State': 'ACTIVE',
			'Channel-State-Number': '10',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'hangup',
			'Hangup-Cause': 'NORMAL_CLEARING',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572228165',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_is_outbound': 'true',
			'variable_uuid': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_session_id': '76',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_sip_invite_domain': '172.99.99.99',
			'variable_user_context': 'default',
			'variable_effective_caller_id_name': 'Extension 1000',
			'variable_effective_caller_id_number': '1000',
			'variable_outbound_caller_id_name': 'FreeSWITCH',
			'variable_outbound_caller_id_number': '0000000000',
			'variable_callgroup': 'techsupport',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_HANGUP',
			sequence: 39893,
			firedAt: new Date(1749847575568),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_EXCHANGE_MEDIA',
			channelStateNumber: '10',
			channelCallState: 'ACTIVE',
			channelUsername: '1000',
			answerState: 'hangup',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			hangupCause: 'NORMAL_CLEARING',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572228),
							channelBridgedTime: new Date(1749847572928),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_HANGUP',
				'Event-Date-Timestamp': '1749847575568205',
				'Event-Sequence': '39893',
			},
			variables: {
				direction: 'outbound',
				is_outbound: 'true',
				uuid: '31450e18-1531-46a0-9c10-4629110e3d23',
				session_id: '76',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				sip_invite_domain: '172.99.99.99',
				user_context: 'default',
				effective_caller_id_name: 'Extension 1000',
				effective_caller_id_number: '1000',
				outbound_caller_id_name: 'FreeSWITCH',
				outbound_caller_id_number: '0000000000',
				callgroup: 'techsupport',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse outbound CHANNEL_HANGUP_COMPLETE event', () => {
		const eventName = 'CHANNEL_HANGUP_COMPLETE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_HANGUP_COMPLETE',
			'Event-Date-Timestamp': '1749847575568205',
			'Event-Sequence': '39898',
			'Hangup-Cause': 'NORMAL_CLEARING',
			'Channel-State': 'CS_REPORTING',
			'Channel-Call-State': 'HANGUP',
			'Channel-State-Number': '11',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'hangup',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572228165',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '1749847575568205',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_is_outbound': 'true',
			'variable_uuid': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_user_context': 'default',
			'variable_effective_caller_id_name': 'Extension 1000',
			'variable_effective_caller_id_number': '1000',
			'variable_outbound_caller_id_name': 'FreeSWITCH',
			'variable_outbound_caller_id_number': '0000000000',
			'variable_callgroup': 'techsupport',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_HANGUP_COMPLETE',
			sequence: 39898,
			firedAt: new Date(1749847575568),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_REPORTING',
			channelStateNumber: '11',
			channelCallState: 'HANGUP',
			channelUsername: '1000',
			answerState: 'hangup',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			hangupCause: 'NORMAL_CLEARING',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572228),
							channelBridgedTime: new Date(1749847572928),
							channelHangupTime: new Date(1749847575568),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_HANGUP_COMPLETE',
				'Event-Date-Timestamp': '1749847575568205',
				'Event-Sequence': '39898',
			},
			variables: {
				direction: 'outbound',
				is_outbound: 'true',
				uuid: '31450e18-1531-46a0-9c10-4629110e3d23',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				user_context: 'default',
				effective_caller_id_name: 'Extension 1000',
				effective_caller_id_number: '1000',
				outbound_caller_id_name: 'FreeSWITCH',
				outbound_caller_id_number: '0000000000',
				callgroup: 'techsupport',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});
	it('should parse CHANNEL_UNBRIDGE event', () => {
		const eventName = 'CHANNEL_UNBRIDGE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_UNBRIDGE',
			'Event-Date-Timestamp': '1749847575588235',
			'Event-Sequence': '39899',
			'Bridge-A-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Bridge-B-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Channel-State': 'CS_EXECUTE',
			'Channel-Call-State': 'ACTIVE',
			'Channel-State-Number': '4',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'answered',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572928202',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '1749847572249573',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originatee',
			'Other-Leg-Direction': 'outbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Other-Leg-Profile-Created-Time': '1749847568969044',
			'Other-Leg-Channel-Created-Time': '1749847568969044',
			'Other-Leg-Channel-Answered-Time': '1749847572228165',
			'Other-Leg-Channel-Progress-Time': '1749847569148194',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_call_uuid': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'variable_dialed_extension': '1000',
			'variable_call_timeout': '30',
			'variable_hangup_after_bridge': 'true',
			'variable_current_application_data': 'user/1000@172.99.99.99',
			'variable_current_application': 'bridge',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_originated_legs': [
				'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			],
			'variable_endpoint_disposition': 'ANSWER',
			'variable_originate_causes': ['31450e18-1531-46a0-9c10-4629110e3d23;NONE', '31450e18-1531-46a0-9c10-4629110e3d23;NONE'],
			'variable_originate_disposition': 'SUCCESS',
			'variable_DIALSTATUS': 'SUCCESS',
			'variable_last_bridge_to': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_bridge_channel': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'variable_bridge_uuid': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_signal_bond': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_last_sent_callee_id_name': 'Outbound Call',
			'variable_last_sent_callee_id_number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'variable_sip_hangup_phrase': 'OK',
			'variable_last_bridge_hangup_cause': 'NORMAL_CLEARING',
			'variable_last_bridge_proto_specific_hangup_cause': 'sip:200',
			'variable_bridge_hangup_cause': 'NORMAL_CLEARING',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_UNBRIDGE',
			sequence: 39899,
			firedAt: new Date(1749847575588),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_EXECUTE',
			channelStateNumber: '4',
			channelCallState: 'ACTIVE',
			channelUsername: '1001',
			answerState: 'answered',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			bridgeUniqueIds: ['ebc91302-1b79-4ff3-ac6b-841385e9ea03', '31450e18-1531-46a0-9c10-4629110e3d23'],
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572928),
							channelBridgedTime: new Date(1749847572928),
							channelProgressMediaTime: new Date(1749847572249),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Other-Leg',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					type: 'originatee',
					direction: 'outbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572228),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_UNBRIDGE',
				'Event-Date-Timestamp': '1749847575588235',
				'Event-Sequence': '39899',
			},
			variables: {
				direction: 'inbound',
				uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				call_uuid: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
				dialed_extension: '1000',
				call_timeout: '30',
				hangup_after_bridge: 'true',
				current_application_data: 'user/1000@172.99.99.99',
				current_application: 'bridge',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				originated_legs: [
					'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					'31450e18-1531-46a0-9c10-4629110e3d23;Outbound Call;1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				],
				endpoint_disposition: 'ANSWER',
				originate_causes: ['31450e18-1531-46a0-9c10-4629110e3d23;NONE', '31450e18-1531-46a0-9c10-4629110e3d23;NONE'],
				originate_disposition: 'SUCCESS',
				DIALSTATUS: 'SUCCESS',
				last_bridge_to: '31450e18-1531-46a0-9c10-4629110e3d23',
				bridge_channel: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
				bridge_uuid: '31450e18-1531-46a0-9c10-4629110e3d23',
				signal_bond: '31450e18-1531-46a0-9c10-4629110e3d23',
				last_sent_callee_id_name: 'Outbound Call',
				last_sent_callee_id_number: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
				sip_hangup_phrase: 'OK',
				last_bridge_hangup_cause: 'NORMAL_CLEARING',
				last_bridge_proto_specific_hangup_cause: 'sip:200',
				bridge_hangup_cause: 'NORMAL_CLEARING',
			},
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});

		expect(result?.raw).toBeUndefined();
	});

	it('should parse CHANNEL_DESTROY event', () => {
		const eventName = 'CHANNEL_DESTROY';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_DESTROY',
			'Event-Date-Timestamp': '1749847575588235',
			'Event-Sequence': '39904',
			'Channel-State': 'CS_REPORTING',
			'Channel-Call-State': 'HANGUP',
			'Channel-State-Number': '12',
			'Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Call-Direction': 'outbound',
			'Presence-Call-Direction': 'outbound',
			'Channel-HIT-Dialplan': 'false',
			'Channel-Presence-ID': '1000@172.99.99.99',
			'Channel-Call-UUID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Answer-State': 'hangup',
			'Hangup-Cause': 'NORMAL_CLEARING',
			'Channel-Read-Codec-Name': 'opus',
			'Channel-Read-Codec-Rate': '48000',
			'Channel-Read-Codec-Bit-Rate': '0',
			'Channel-Write-Codec-Name': 'opus',
			'Channel-Write-Codec-Rate': '48000',
			'Channel-Write-Codec-Bit-Rate': '0',
			'Caller-Direction': 'outbound',
			'Caller-Logical-Direction': 'outbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Callee-ID-Name': 'Outbound Call',
			'Caller-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Caller-Unique-ID': '31450e18-1531-46a0-9c10-4629110e3d23',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '1749847572228165',
			'Caller-Channel-Progress-Time': '1749847569148194',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '1749847575568205',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '1749847572928202',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'Other-Type': 'originator',
			'Other-Leg-Direction': 'inbound',
			'Other-Leg-Logical-Direction': 'inbound',
			'Other-Leg-Username': '1001',
			'Other-Leg-Dialplan': 'XML',
			'Other-Leg-Caller-ID-Name': '1001',
			'Other-Leg-Caller-ID-Number': '1001',
			'Other-Leg-Orig-Caller-ID-Name': '1001',
			'Other-Leg-Orig-Caller-ID-Number': '1001',
			'Other-Leg-Callee-ID-Name': 'Outbound Call',
			'Other-Leg-Callee-ID-Number': '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
			'Other-Leg-Network-Addr': '177.99.99.183',
			'Other-Leg-ANI': '1001',
			'Other-Leg-Destination-Number': '1000',
			'Other-Leg-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Other-Leg-Source': 'mod_sofia',
			'Other-Leg-Context': 'default',
			'Other-Leg-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Other-Leg-Profile-Created-Time': '0',
			'Other-Leg-Channel-Created-Time': '0',
			'Other-Leg-Channel-Answered-Time': '0',
			'Other-Leg-Channel-Progress-Time': '0',
			'Other-Leg-Channel-Progress-Media-Time': '0',
			'Other-Leg-Channel-Hangup-Time': '0',
			'Other-Leg-Channel-Transfer-Time': '0',
			'Other-Leg-Channel-Resurrect-Time': '0',
			'Other-Leg-Channel-Bridged-Time': '0',
			'Other-Leg-Channel-Last-Hold': '0',
			'Other-Leg-Channel-Hold-Accum': '0',
			'Other-Leg-Screen-Bit': 'true',
			'Other-Leg-Privacy-Hide-Name': 'false',
			'Other-Leg-Privacy-Hide-Number': 'false',
			'variable_direction': 'outbound',
			'variable_is_outbound': 'true',
			'variable_uuid': '31450e18-1531-46a0-9c10-4629110e3d23',
			'variable_dialed_user': '1000',
			'variable_dialed_domain': '172.99.99.99',
			'variable_user_context': 'default',
			'variable_effective_caller_id_name': 'Extension 1000',
			'variable_effective_caller_id_number': '1000',
			'variable_outbound_caller_id_name': 'FreeSWITCH',
			'variable_outbound_caller_id_number': '0000000000',
		});

		const result = parseEventData(eventName, eventData);
		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
			eventName: 'CHANNEL_DESTROY',
			sequence: 39904,
			firedAt: new Date(1749847575588),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
			channelState: 'CS_REPORTING',
			channelStateNumber: '12',
			channelCallState: 'HANGUP',
			channelUsername: '1000',
			answerState: 'hangup',
			callDirection: 'outbound',
			channelHitDialplan: 'false',
			hangupCause: 'NORMAL_CLEARING',
			legs: {
				'31450e18-1531-46a0-9c10-4629110e3d23': {
					legName: 'Caller',
					uniqueId: '31450e18-1531-46a0-9c10-4629110e3d23',
					direction: 'outbound',
					logicalDirection: 'outbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa@localhost',
					profiles: {
						'1': {
							channelCreatedTime: new Date(1749847568969),
							profileCreatedTime: new Date(1749847568969),
							channelProgressTime: new Date(1749847569148),
							channelAnsweredTime: new Date(1749847572228),
							channelBridgedTime: new Date(1749847572928),
							channelHangupTime: new Date(1749847575568),
							profileIndex: '1',
						},
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Other-Leg',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					type: 'originator',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					calleeName: 'Outbound Call',
					calleeNumber: '1000-LJZ8A9MhHv4Eh6ZQH-h7720gsa',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: {
				'Event-Name': 'CHANNEL_DESTROY',
				'Event-Date-Timestamp': '1749847575588235',
				'Event-Sequence': '39904',
			},
			variables: {
				direction: 'outbound',
				is_outbound: 'true',
				uuid: '31450e18-1531-46a0-9c10-4629110e3d23',
				dialed_user: '1000',
				dialed_domain: '172.99.99.99',
				user_context: 'default',
				effective_caller_id_name: 'Extension 1000',
				effective_caller_id_number: '1000',
				outbound_caller_id_name: 'FreeSWITCH',
				outbound_caller_id_number: '0000000000',
			},
			raw: { 'Channel-Call-UUID': '31450e18-1531-46a0-9c10-4629110e3d23' },
			codecs: {
				read: { name: 'opus', rate: '48000' },
				write: { nme: 'opus', rate: '48000' },
			},
			presenceCallDirection: 'outbound',
			channelPresenceId: '1000@172.99.99.99',
			caller: '1001',
			callee: '1000',
		});
	});

	it('should parse CHANNEL_CALLSTATE event', () => {
		const eventName = 'CHANNEL_CALLSTATE';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_CALLSTATE',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39836',
			'Channel-State': 'CS_INIT',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_ep_codec_string':
				'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
			'variable_endpoint_disposition': 'DELAYED NEGOTIATION',
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_CALLSTATE',
			sequence: 39836,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_INIT',
			channelStateNumber: '2',
			channelCallState: 'DOWN',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': { profileIndex: '1', profileCreatedTime: new Date(1749847568969), channelCreatedTime: new Date(1749847568969) },
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: { 'Event-Name': 'CHANNEL_CALLSTATE', 'Event-Date-Timestamp': '1749847568969044', 'Event-Sequence': '39836' },
			variables: {
				direction: 'inbound',
				ep_codec_string:
					'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
				endpoint_disposition: 'DELAYED NEGOTIATION',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});
	});

	it('should parse CHANNEL_ANSWER event', () => {
		const eventName = 'CHANNEL_ANSWER';
		const eventData = createEventData({
			'Event-Name': 'CHANNEL_ANSWER',
			'Event-Date-Timestamp': '1749847568969044',
			'Event-Sequence': '39836',
			'Channel-State': 'CS_INIT',
			'Channel-Call-State': 'DOWN',
			'Channel-State-Number': '2',
			'Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Call-Direction': 'inbound',
			'Presence-Call-Direction': 'inbound',
			'Channel-HIT-Dialplan': 'true',
			'Channel-Call-UUID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Answer-State': 'ringing',
			'Caller-Direction': 'inbound',
			'Caller-Logical-Direction': 'inbound',
			'Caller-Username': '1001',
			'Caller-Dialplan': 'XML',
			'Caller-Caller-ID-Name': '1001',
			'Caller-Caller-ID-Number': '1001',
			'Caller-Orig-Caller-ID-Name': '1001',
			'Caller-Orig-Caller-ID-Number': '1001',
			'Caller-Network-Addr': '177.99.99.183',
			'Caller-ANI': '1001',
			'Caller-Destination-Number': '1000',
			'Caller-Unique-ID': 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			'Caller-Source': 'mod_sofia',
			'Caller-Context': 'default',
			'Caller-Channel-Name': 'sofia/internal/1001@99.219.99.169:7443',
			'Caller-Profile-Index': '1',
			'Caller-Profile-Created-Time': '1749847568969044',
			'Caller-Channel-Created-Time': '1749847568969044',
			'Caller-Channel-Answered-Time': '0',
			'Caller-Channel-Progress-Time': '0',
			'Caller-Channel-Progress-Media-Time': '0',
			'Caller-Channel-Hangup-Time': '0',
			'Caller-Channel-Transfer-Time': '0',
			'Caller-Channel-Resurrect-Time': '0',
			'Caller-Channel-Bridged-Time': '0',
			'Caller-Channel-Last-Hold': '0',
			'Caller-Channel-Hold-Accum': '0',
			'Caller-Screen-Bit': 'true',
			'Caller-Privacy-Hide-Name': 'false',
			'Caller-Privacy-Hide-Number': 'false',
			'variable_direction': 'inbound',
			'variable_ep_codec_string':
				'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
			'variable_endpoint_disposition': 'DELAYED NEGOTIATION',
		});

		const result = parseEventData(eventName, eventData);

		expect(result).toBeDefined();
		expect(result?.receivedAt).toBeInstanceOf(Date);

		expect(result).toMatchObject({
			channelUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			eventName: 'CHANNEL_ANSWER',
			sequence: 39836,
			firedAt: new Date(1749847568969),
			callUniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
			channelName: 'sofia/internal/1001@99.219.99.169:7443',
			channelState: 'CS_INIT',
			channelStateNumber: '2',
			channelCallState: 'DOWN',
			channelUsername: '1001',
			answerState: 'ringing',
			callDirection: 'inbound',
			channelHitDialplan: 'true',
			legs: {
				'ebc91302-1b79-4ff3-ac6b-841385e9ea03': {
					legName: 'Caller',
					uniqueId: 'ebc91302-1b79-4ff3-ac6b-841385e9ea03',
					direction: 'inbound',
					logicalDirection: 'inbound',
					username: '1001',
					callerName: '1001',
					callerNumber: '1001',
					originalCallerName: '1001',
					originalCallerNumber: '1001',
					networkAddress: '177.99.99.183',
					destinationNumber: '1000',
					source: 'mod_sofia',
					context: 'default',
					channelName: 'sofia/internal/1001@99.219.99.169:7443',
					profiles: {
						'1': { profileIndex: '1', profileCreatedTime: new Date(1749847568969), channelCreatedTime: new Date(1749847568969) },
					},
					dialplan: 'XML',
					ani: '1001',
					screenBit: 'true',
					privacyHideName: 'false',
					privacyHideNumber: 'false',
				},
			},
			metadata: { 'Event-Name': 'CHANNEL_ANSWER', 'Event-Date-Timestamp': '1749847568969044', 'Event-Sequence': '39836' },
			variables: {
				direction: 'inbound',
				ep_codec_string:
					'mod_opus.opus@48000h@20i@2c,mod_spandsp.G722@8000h@20i@64000b,CORE_PCM_MODULE.PCMU@8000h@20i@64000b,CORE_PCM_MODULE.PCMA@8000h@20i@64000b',
				endpoint_disposition: 'DELAYED NEGOTIATION',
			},
			presenceCallDirection: 'inbound',
			caller: '1001',
			callee: '1000',
		});
	});
});
