"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseEventData = parseEventData;
const filterOutMissingData_1 = require("./filterOutMissingData");
const filterStringList_1 = require("./filterStringList");
const parseChannelUsername_1 = require("./parseChannelUsername");
const parseEventCallId_1 = require("./parseEventCallId");
const parseEventLeg_1 = require("./parseEventLeg");
const parseTimestamp_1 = require("./parseTimestamp");
const logger_1 = require("../logger");
const parseEventExtensions_1 = require("./parseEventExtensions");
function parseEventData(eventName, eventData) {
    const { 'Channel-Name': channelName = '', 'Channel-State': channelState = '', 'Channel-Call-State': channelCallState = '', 'Channel-State-Number': channelStateNumber, 'Channel-Call-State-Number': channelCallStateNumber, 'Original-Channel-Call-State': originalChannelCallState, 'Event-Sequence': sequenceStr, 'Event-Date-Timestamp': timestamp, 'Unique-ID': channelUniqueId, 'Call-Direction': callDirection, 'Channel-HIT-Dialplan': channelHitDialplan, 'Answer-State': answerState, 'Hangup-Cause': hangupCause, 'Bridge-A-Unique-ID': bridgeA, 'Bridge-B-Unique-ID': bridgeB, 'Bridged-To': bridgedTo, 'Presence-Call-Direction': presenceCallDirection, 'Channel-Presence-ID': channelPresenceId, 'Channel-Read-Codec-Name': codecReadName, 'Channel-Read-Codec-Rate': codecReadRate, 'Channel-Write-Codec-Name': codecWriteName, 'Channel-Write-Codec-Rate': codecWriteRate, ...rawEventData } = eventData;
    if (!channelUniqueId || !sequenceStr) {
        logger_1.logger.error({ msg: 'Channel Event is missing either the Unique-ID or Event-Sequence', eventData });
        return;
    }
    const sequence = parseInt(sequenceStr);
    if (!sequence || typeof sequence !== 'number' || !Number.isInteger(sequence)) {
        logger_1.logger.error({ msg: 'Failed to parse Event-Sequence', eventData });
        return;
    }
    const callUniqueId = (0, parseEventCallId_1.parseEventCallId)(eventData) || channelUniqueId;
    const channelUsername = (0, parseChannelUsername_1.parseChannelUsername)(channelName);
    const firedAt = (0, parseTimestamp_1.parseTimestamp)(timestamp) || new Date();
    const callerLeg = (0, parseEventLeg_1.parseEventLeg)('Caller', eventData);
    const otherLeg = (0, parseEventLeg_1.parseEventLeg)('Other-Leg', eventData);
    const bridgeUniqueIds = [bridgeA, bridgeB].filter((bridgeId) => bridgeId);
    const legs = {
        ...(callerLeg?.uniqueId && { [callerLeg.uniqueId]: callerLeg }),
        ...(otherLeg?.uniqueId && { [otherLeg.uniqueId]: otherLeg }),
    };
    const variables = (0, filterStringList_1.filterStringList)(eventData, (key) => key.startsWith('variable_'), ([key, value]) => {
        return [key.replace('variable_', ''), value || ''];
    });
    const metadata = (0, filterStringList_1.filterStringList)(eventData, (key) => isMetadata(key));
    const unusedRawData = (0, filterStringList_1.filterStringList)(rawEventData, (key) => {
        if (isMetadata(key)) {
            return false;
        }
        if (key.startsWith('variable_')) {
            return false;
        }
        for (const { legName } of Object.values(legs)) {
            if (key.startsWith(`${legName}-`)) {
                return false;
            }
        }
        if (otherLeg && key === 'Other-Type') {
            return false;
        }
        if (key === 'Channel-Call-UUID') {
            return rawEventData['Channel-Call-UUID'] !== callUniqueId;
        }
        return true;
    });
    const event = {
        channelUniqueId,
        eventName,
        sequence,
        firedAt,
        receivedAt: new Date(),
        callUniqueId,
        channelName,
        channelState,
        channelStateNumber,
        channelCallStateNumber,
        channelCallState,
        originalChannelCallState,
        channelUsername,
        answerState,
        callDirection,
        channelHitDialplan,
        hangupCause,
        ...(bridgeUniqueIds.length && { bridgeUniqueIds }),
        bridgedTo,
        legs,
        metadata: (0, filterOutMissingData_1.filterOutMissingData)(metadata),
        ...(Object.keys(variables).length && { variables }),
        raw: (0, filterOutMissingData_1.filterOutMissingData)(unusedRawData),
        codecs: {
            ...{
                read: {
                    ...{
                        name: codecReadName,
                        rate: codecReadRate,
                    },
                },
                write: {
                    ...{
                        nme: codecWriteName,
                        rate: codecWriteRate,
                    },
                },
            },
        },
        presenceCallDirection,
        channelPresenceId,
    };
    const filteredEvent = {
        ...(0, filterOutMissingData_1.filterOutMissingData)(event),
        channelName,
        channelCallState,
        channelState,
    };
    const extensions = (0, parseEventExtensions_1.parseEventExtensions)(filteredEvent);
    return {
        ...filteredEvent,
        ...extensions,
    };
}
function isMetadata(key) {
    return key.startsWith('Event-') || key.startsWith('FreeSWITCH-') || key.startsWith('Core-');
}
//# sourceMappingURL=parseEventData.js.map