"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.computeChannelProfiles = computeChannelProfiles;
function adjustProfileTimestamps(profile) {
    const { profileIndex, profileCreatedTime, channelCreatedTime, bridgedTo, caller, callee, ...timestamps } = profile;
    // Don't mutate anything if it's the first profile
    if (!profileIndex || profileIndex === '1') {
        return { ...profile };
    }
    const newProfile = {
        channelCreatedTime,
        profileIndex,
        bridgedTo,
        callee,
        caller,
    };
    // If we don't know when the profile was created, drop every other timestamp
    if (!profileCreatedTime) {
        return newProfile;
    }
    newProfile.profileCreatedTime = profileCreatedTime;
    for (const key of Object.keys(timestamps)) {
        const value = timestamps[key];
        if (!value || typeof value === 'string') {
            continue;
        }
        if (value < profileCreatedTime) {
            continue;
        }
        newProfile[key] = value;
    }
    return newProfile;
}
function computeChannelProfiles(legProfiles) {
    const profiles = Object.values(legProfiles).map((profile) => adjustProfileTimestamps(profile));
    // Sort profiles by createdTime, temporarily filter out the ones that do not have one:
    const sortedProfiles = profiles
        .filter(({ profileCreatedTime, channelCreatedTime, profileIndex }) => profileCreatedTime || (profileIndex === '1' && channelCreatedTime))
        .sort(({ profileCreatedTime: profile1, channelCreatedTime: channel1 }, { profileCreatedTime: profile2, channelCreatedTime: channel2 }) => (profile1?.valueOf() || channel1.valueOf()) - (profile2?.valueOf() || channel2.valueOf()));
    const adjustedProfiles = [];
    let anyAnswer = false;
    let anyMedia = false;
    let anyBridge = false;
    let durationSum = 0;
    let firstProfileCreate;
    // "first" because it's an array, but it's the same channel for all so there should only be one value
    let firstChannelCreate;
    for (let i = 0; i < sortedProfiles.length; i++) {
        const nextProfileCreatedTime = sortedProfiles[i + 1]?.profileCreatedTime || undefined;
        const profile = sortedProfiles[i];
        const { channelBridgedTime, channelAnsweredTime, channelProgressMediaTime, channelHangupTime, bridgedTo, profileCreatedTime, channelCreatedTime, } = profile;
        const callEnd = channelHangupTime || nextProfileCreatedTime;
        if (channelCreatedTime && (!firstChannelCreate || firstChannelCreate > channelCreatedTime)) {
            firstChannelCreate = channelCreatedTime;
        }
        if (profileCreatedTime && (!firstProfileCreate || firstProfileCreate > profileCreatedTime)) {
            firstProfileCreate = profileCreatedTime;
        }
        const callDuration = callEnd && channelBridgedTime ? callEnd.valueOf() - channelBridgedTime.valueOf() : 0;
        const media = Boolean(channelProgressMediaTime) || sortedProfiles.length > 1;
        const answered = Boolean(channelAnsweredTime) || media;
        const bridged = Boolean(channelBridgedTime) || Boolean(bridgedTo);
        anyMedia || (anyMedia = media);
        anyAnswer || (anyAnswer = answered);
        anyBridge || (anyBridge = bridged);
        durationSum += callDuration;
        adjustedProfiles.push({
            ...profile,
            ...{
                nextProfileCreatedTime,
                callDuration,
                answered,
                media,
                bridged,
            },
        });
    }
    // Look for bridge and hangup on every channel, even if they didn't have a profile timestamp (in theory every profile will always have a created timestamp)
    let firstBridge;
    let lastCallEnd;
    for (const profile of profiles) {
        const { channelBridgedTime, channelHangupTime, nextProfileCreatedTime, bridgedTo } = profile;
        const callEnd = channelHangupTime || nextProfileCreatedTime;
        if (channelBridgedTime && (!firstBridge || firstBridge > channelBridgedTime)) {
            firstBridge = channelBridgedTime;
        }
        if ((callEnd || 0) > (lastCallEnd || 0)) {
            lastCallEnd = callEnd;
        }
        // If this profile was filtered out from the list used by the first process, add it back to the final list here
        if (!sortedProfiles.includes(profile)) {
            const bridged = Boolean(channelBridgedTime) || Boolean(bridgedTo);
            anyBridge || (anyBridge = bridged);
            adjustedProfiles.push({ ...profile, ...{ bridged } });
        }
    }
    const firstCallStart = firstBridge || firstProfileCreate || firstChannelCreate;
    const totalDuration = lastCallEnd && firstCallStart ? lastCallEnd.valueOf() - firstCallStart.valueOf() : 0;
    return {
        profiles: adjustedProfiles,
        anyMedia,
        anyAnswer,
        anyBridge,
        durationSum,
        totalDuration,
        startedAt: firstCallStart,
    };
}
//# sourceMappingURL=computeChannelProfiles.js.map