"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.computeChannelFromEvents = computeChannelFromEvents;
const tools_1 = require("@rocket.chat/tools");
const computeChannelProfiles_1 = require("./computeChannelProfiles");
const extractChannelChangesFromEvent_1 = require("./extractChannelChangesFromEvent");
const filterOutMissingData_1 = require("./filterOutMissingData");
const insertDataIntoEventProfile_1 = require("./insertDataIntoEventProfile");
const parseChannelKind_1 = require("./parseChannelKind");
function splitEventDataSections(event) {
    const { _id, channelUniqueId, _updatedAt, metadata, eventName, sequence, firedAt, receivedAt, callee, caller, ...eventData } = event;
    return {
        channelUniqueId,
        header: {
            sequence,
            eventName,
            firedAt,
            receivedAt,
            callee,
            caller,
        },
        eventData,
    };
}
async function computeChannelFromEvents(allEvents) {
    if (!allEvents.length) {
        return;
    }
    const deltas = [];
    const { channelUniqueId: uniqueId, firedAt: firstEvent } = allEvents[0];
    const callDirections = [];
    const callers = [];
    const callees = [];
    const bridgedTo = [];
    const headers = [];
    for (const event of allEvents) {
        const { callee, caller, bridgeUniqueIds, bridgedTo: eventBridgedTo } = event;
        if (event.callDirection && !callDirections.includes(event.callDirection)) {
            callDirections.push(event.callDirection);
        }
        if (callee && !callees.includes(callee)) {
            callees.push(callee);
        }
        if (caller && !callers.includes(caller)) {
            callers.push(caller);
        }
        if (bridgeUniqueIds) {
            for (const bridgeUniqueId of bridgeUniqueIds) {
                if (bridgeUniqueId && !bridgedTo.includes(bridgeUniqueId) && bridgeUniqueId !== uniqueId) {
                    bridgedTo.push(bridgeUniqueId);
                }
            }
        }
        if (eventBridgedTo && !bridgedTo.includes(eventBridgedTo)) {
            bridgedTo.push(eventBridgedTo);
        }
    }
    const flattened = allEvents.reduce((state, nextEvent) => {
        const { header, eventData, channelUniqueId } = splitEventDataSections(nextEvent);
        const { caller, callee, eventName, sequence } = header;
        // Inserts the callee and bridgedTo attributes into the profile of this event
        const eventDataEx = (0, insertDataIntoEventProfile_1.insertDataIntoEventProfile)(eventData, { caller, callee, bridgedTo: eventData.bridgedTo });
        // Make a list with every value from the event, except for the headers;
        const eventValues = (0, tools_1.convertSubObjectsIntoPaths)(eventDataEx);
        // Compare the event's list of values with the full list from all past events
        const { changedValues, newValues, changedExistingValues } = (0, extractChannelChangesFromEvent_1.extractChannelChangesFromEvent)(state, eventName, eventValues);
        const { channelState, channelCallState, originalChannelCallState, answerState } = eventData;
        const headerWithStates = {
            ...header,
            channelState,
            channelCallState,
            originalChannelCallState,
            answerState,
        };
        // Generate a "delta" entry with the data that has changed in this event
        const delta = {
            ...headerWithStates,
            newValues: (0, tools_1.convertPathsIntoSubObjects)(newValues),
            modifiedValues: (0, tools_1.convertPathsIntoSubObjects)(changedExistingValues),
        };
        // Store this delta in a list
        deltas.push((0, filterOutMissingData_1.filterOutMissingData)(delta));
        headers.push(headerWithStates);
        return {
            channelUniqueId,
            ...state,
            eventName,
            sequence,
            ...changedValues,
        };
    }, {});
    const finalState = (0, tools_1.convertPathsIntoSubObjects)(flattened);
    const computedProfiles = (0, computeChannelProfiles_1.computeChannelProfiles)(finalState?.legs?.[uniqueId]?.profiles || {});
    return {
        channel: {
            uniqueId,
            name: finalState.channelName,
            callDirection: callDirections.join('||'),
            freeSwitchUser: finalState.channelUsername,
            callers,
            callees,
            bridgedTo,
            ...{
                ...computedProfiles,
                // If we couldn't parse a startedAt, use the time of the first event
                startedAt: computedProfiles.startedAt || firstEvent,
            },
            kind: (0, parseChannelKind_1.parseChannelKind)(finalState.channelName),
            finalState,
            events: headers,
        },
        deltas,
    };
}
//# sourceMappingURL=computeChannelFromEvents.js.map