"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FreeSwitchESLClient = void 0;
const node_net_1 = require("node:net");
const emitter_1 = require("@rocket.chat/emitter");
const tools_1 = require("@rocket.chat/tools");
const esl_1 = require("esl");
const logger_1 = require("../logger");
class FreeSwitchESLClient extends emitter_1.Emitter {
    constructor(options) {
        super();
        this.options = options;
        this.state = 'none';
        this.expectingEnd = false;
        this.host = this.options.socketOptions.host;
        logger_1.logger.debug('Connecting new FreeSwitch socket');
        this.socket = new node_net_1.Socket();
        this.response = new esl_1.FreeSwitchResponse(this.socket, logger_1.logger);
        this.socket.once('connect', () => {
            logger_1.logger.debug('FreeSwitch socket connected.');
            this.authenticate();
        });
        this.socket.once('error', (error) => {
            logger_1.logger.error({ msg: 'error on connection with freeswitch server', state: this.state, error });
            this.changeState('failed');
        });
        this.socket.once('end', () => {
            if (!this.expectingEnd) {
                logger_1.logger.debug('FreeSwitchESLClient received `end` event (remote end sent a FIN packet)');
            }
            this.changeState('ended');
        });
        this.socket.on('warning', (data) => {
            logger_1.logger.warn({ msg: 'FreeSwitchClient: warning', data });
        });
        try {
            this.socket.connect(this.options.socketOptions);
        }
        catch (error) {
            this.changeState('failed');
            logger_1.logger.error({ msg: 'failed to connect to freeswitch server', error });
        }
    }
    async authenticate() {
        logger_1.logger.debug('FreeSwitch socket authenticating.');
        this.changeState('authenticating');
        try {
            // Wait for FreeSwitch to send us an authentication request
            await this.response.onceAsync('freeswitch_auth_request', this.options.timeout ?? 20000, 'FreeSwitchClient expected authentication request');
            await this.response.auth(this.options.password);
            this.changeState('transitioning');
            this.response.auto_cleanup();
            await this.transitionToReady();
        }
        catch (error) {
            logger_1.logger.error('FreeSwitchClient: initialization error', error);
            this.changeState('failed');
        }
    }
    async transitionToReady() {
        this.changeState('ready');
    }
    changeState(newState) {
        logger_1.logger.debug({ msg: 'FreeSwitchESLClient changing state .', newState, state: this.state });
        if (this.isDone()) {
            return;
        }
        this.state = newState;
        if (this.isReady()) {
            this.emit('ready');
            return;
        }
        if (this.isDone()) {
            this.emit('end');
        }
    }
    isReady() {
        return this.state === 'ready';
    }
    isDone() {
        return ['failed', 'ended'].includes(this.state);
    }
    async waitUntilUsable() {
        if (this.isReady()) {
            return;
        }
        if (this.isDone()) {
            throw new Error('connection-ended');
        }
        return new Promise((resolve, reject) => {
            let concluded = false;
            this.once('ready', () => {
                if (!concluded) {
                    concluded = true;
                    resolve();
                }
            });
            this.once('end', () => {
                if (!concluded) {
                    concluded = true;
                    reject(new Error('connection-ended'));
                }
            });
        });
    }
    endConnection() {
        this.expectingEnd = true;
        (0, tools_1.wrapExceptions)(() => this.response.end()).suppress();
    }
}
exports.FreeSwitchESLClient = FreeSwitchESLClient;
//# sourceMappingURL=client.js.map