"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MatrixMediaService = void 0;
const core_services_1 = require("@rocket.chat/core-services");
const federation_sdk_1 = require("@rocket.chat/federation-sdk");
const logger_1 = require("@rocket.chat/logger");
const models_1 = require("@rocket.chat/models");
const logger = new logger_1.Logger('federation-matrix:media-service');
class MatrixMediaService {
    static generateMXCUri(fileId, serverName) {
        return `mxc://${serverName}/${fileId}`;
    }
    static parseMXCUri(mxcUri) {
        const match = mxcUri.match(/^mxc:\/\/([^/]+)\/(.+)$/);
        if (!match) {
            logger.error('Invalid MXC URI format', { mxcUri });
            return null;
        }
        return {
            serverName: match[1],
            mediaId: match[2],
        };
    }
    static async prepareLocalFileForMatrix(fileId, serverName, matrixRoomId) {
        try {
            const file = await models_1.Uploads.findOneById(fileId);
            if (!file) {
                logger.error(`File ${fileId} not found in database`);
                throw new Error(`File ${fileId} not found`);
            }
            if (file.federation?.mxcUri) {
                return file.federation.mxcUri;
            }
            const mxcUri = this.generateMXCUri(fileId, serverName);
            await models_1.Uploads.setFederationInfo(fileId, {
                mrid: matrixRoomId,
                mxcUri,
                serverName,
                mediaId: fileId,
            });
            return mxcUri;
        }
        catch (error) {
            logger.error('Error preparing file for Matrix:', error);
            throw error;
        }
    }
    static async getLocalFileForMatrixNode(mediaId, serverName) {
        try {
            let file = await models_1.Uploads.findByFederationMediaIdAndServerName(mediaId, serverName);
            if (!file) {
                file = await models_1.Uploads.findOneById(mediaId);
            }
            if (!file) {
                return null;
            }
            return file;
        }
        catch (error) {
            logger.error('Error retrieving local file:', error);
            return null;
        }
    }
    static async downloadAndStoreRemoteFile(mxcUri, matrixRoomId, metadata) {
        try {
            const parts = this.parseMXCUri(mxcUri);
            if (!parts) {
                logger.error('Invalid MXC URI format', { mxcUri });
                throw new Error('Invalid MXC URI');
            }
            const uploadAlreadyExists = await models_1.Uploads.findByFederationMediaIdAndServerName(parts.mediaId, parts.serverName);
            if (uploadAlreadyExists) {
                return uploadAlreadyExists._id;
            }
            const buffer = await federation_sdk_1.federationSDK.downloadFromRemoteServer(parts.serverName, parts.mediaId);
            if (!buffer) {
                throw new Error('Download from remote server returned null content.');
            }
            // TODO: Make uploadFile support Partial<IUpload> to avoid calling a DB update right after the upload to set the federation info
            const uploadedFile = await core_services_1.Upload.uploadFile({
                userId: metadata.userId || 'federation',
                buffer,
                details: {
                    name: metadata.name || 'unnamed',
                    size: buffer.length,
                    type: metadata.type || 'application/octet-stream',
                    rid: metadata.roomId,
                    userId: metadata.userId || 'federation',
                },
            });
            await models_1.Uploads.setFederationInfo(uploadedFile._id, {
                mxcUri,
                mrid: matrixRoomId,
                serverName: parts.serverName,
                mediaId: parts.mediaId,
            });
            return uploadedFile._id;
        }
        catch (error) {
            logger.error('Error downloading and storing remote file:', error);
            throw error;
        }
    }
    static async getLocalFileBuffer(file) {
        return core_services_1.Upload.getFileBuffer({ file });
    }
}
exports.MatrixMediaService = MatrixMediaService;
//# sourceMappingURL=MatrixMediaService.js.map