"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMatrixMediaRoutes = void 0;
const crypto_1 = __importDefault(require("crypto"));
const federation_sdk_1 = require("@rocket.chat/federation-sdk");
const http_router_1 = require("@rocket.chat/http-router");
const Ajv_1 = require("@rocket.chat/rest-typings/dist/v1/Ajv");
const MatrixMediaService_1 = require("../../services/MatrixMediaService");
const canAccessResource_1 = require("../middlewares/canAccessResource");
const MediaDownloadParamsSchema = {
    type: 'object',
    properties: {
        mediaId: { type: 'string' },
    },
    required: ['mediaId'],
    additionalProperties: false,
};
const ErrorResponseSchema = {
    type: 'object',
    properties: {
        errcode: { type: 'string' },
        error: { type: 'string' },
    },
    required: ['errcode', 'error'],
};
const BufferResponseSchema = {
    type: 'object',
    description: 'Raw file buffer or multipart response',
};
const isMediaDownloadParamsProps = Ajv_1.ajv.compile(MediaDownloadParamsSchema);
const isErrorResponseProps = Ajv_1.ajv.compile(ErrorResponseSchema);
const isBufferResponseProps = Ajv_1.ajv.compile(BufferResponseSchema);
const SECURITY_HEADERS = {
    'X-Content-Type-Options': 'nosniff',
    'X-Frame-Options': 'DENY',
    'Content-Security-Policy': "default-src 'none'; img-src 'self'; media-src 'self'",
    'Strict-Transport-Security': 'max-age=31536000; includeSubDomains',
};
function createMultipartResponse(buffer, mimeType, fileName, metadata = {}) {
    const boundary = crypto_1.default.randomBytes(16).toString('hex');
    const parts = [];
    parts.push(`--${boundary}`, 'Content-Type: application/json', '', JSON.stringify(metadata));
    parts.push(`--${boundary}`, `Content-Type: ${mimeType}`, `Content-Disposition: attachment; filename="${fileName}"`, '');
    const headerBuffer = Buffer.from(`${parts.join('\r\n')}\r\n`);
    const endBoundary = Buffer.from(`\r\n--${boundary}--\r\n`);
    return {
        body: Buffer.concat([headerBuffer, buffer, endBoundary]),
        contentType: `multipart/mixed; boundary=${boundary}`,
    };
}
async function getMediaFile(mediaId, serverName) {
    const file = await MatrixMediaService_1.MatrixMediaService.getLocalFileForMatrixNode(mediaId, serverName);
    if (!file) {
        return null;
    }
    const buffer = await MatrixMediaService_1.MatrixMediaService.getLocalFileBuffer(file);
    return { file, buffer };
}
const getMatrixMediaRoutes = () => {
    return new http_router_1.Router('/federation')
        .get('/v1/media/download/:mediaId', {
        params: isMediaDownloadParamsProps,
        response: {
            200: isBufferResponseProps,
            401: isErrorResponseProps,
            403: isErrorResponseProps,
            404: isErrorResponseProps,
            429: isErrorResponseProps,
            500: isErrorResponseProps,
        },
        tags: ['Federation', 'Media'],
    }, (0, canAccessResource_1.canAccessResourceMiddleware)('media'), async (c) => {
        try {
            const { mediaId } = c.req.param();
            const serverName = federation_sdk_1.federationSDK.getConfig('serverName');
            // TODO: Add file streaming support
            const result = await getMediaFile(mediaId, serverName);
            if (!result) {
                return {
                    statusCode: 404,
                    body: { errcode: 'M_NOT_FOUND', error: 'Media not found' },
                };
            }
            const { file, buffer } = result;
            const mimeType = file.type || 'application/octet-stream';
            const fileName = file.name || mediaId;
            const multipartResponse = createMultipartResponse(buffer, mimeType, fileName);
            return {
                statusCode: 200,
                headers: {
                    ...SECURITY_HEADERS,
                    'content-type': multipartResponse.contentType,
                    'content-length': String(multipartResponse.body.length),
                },
                body: multipartResponse.body,
            };
        }
        catch (error) {
            return {
                statusCode: 500,
                body: { errcode: 'M_UNKNOWN', error: 'Internal server error' },
            };
        }
    })
        .get('/v1/media/thumbnail/:mediaId', {
        params: isMediaDownloadParamsProps,
        response: {
            404: isErrorResponseProps,
        },
        tags: ['Federation', 'Media'],
    }, (0, canAccessResource_1.canAccessResourceMiddleware)('media'), async (_c) => ({
        statusCode: 404,
        body: {
            errcode: 'M_UNRECOGNIZED',
            error: 'This endpoint is not implemented on the homeserver side',
        },
    }));
};
exports.getMatrixMediaRoutes = getMatrixMediaRoutes;
//# sourceMappingURL=media.js.map