"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FederationMatrix = exports.generateEd25519RandomSecretKey = exports.getUsernameServername = exports.extractDomainFromMatrixUserId = exports.fileTypes = void 0;
exports.validateFederatedUsername = validateFederatedUsername;
exports.createOrUpdateFederatedUser = createOrUpdateFederatedUser;
const core_services_1 = require("@rocket.chat/core-services");
const core_typings_1 = require("@rocket.chat/core-typings");
const federation_sdk_1 = require("@rocket.chat/federation-sdk");
const logger_1 = require("@rocket.chat/logger");
const models_1 = require("@rocket.chat/models");
const emojione_1 = __importDefault(require("emojione"));
const invite_1 = require("./api/_matrix/invite");
const message_parsers_1 = require("./helpers/message.parsers");
const MatrixMediaService_1 = require("./services/MatrixMediaService");
exports.fileTypes = {
    image: 'm.image',
    video: 'm.video',
    audio: 'm.audio',
    file: 'm.file',
};
/** helper to validate the username format */
function validateFederatedUsername(mxid) {
    if (!mxid.startsWith('@'))
        return false;
    const parts = mxid.substring(1).split(':');
    if (parts.length < 2)
        return false;
    const localpart = parts[0];
    const domainAndPort = parts.slice(1).join(':');
    const localpartRegex = /^(?:[a-z0-9._\-]|=[0-9a-fA-F]{2}){1,255}$/;
    if (!localpartRegex.test(localpart))
        return false;
    const [domain, port] = domainAndPort.split(':');
    const hostnameRegex = /^(?=.{1,253}$)([a-z0-9](?:[a-z0-9\-]{0,61}[a-z0-9])?)(?:\.[a-z0-9](?:[a-z0-9\-]{0,61}[a-z0-9])?)*$/i;
    const ipv4Regex = /^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)(\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)){3}$/;
    const ipv6Regex = /^\[([0-9a-f:.]+)\]$/i;
    if (!(hostnameRegex.test(domain) || ipv4Regex.test(domain) || ipv6Regex.test(domain))) {
        return false;
    }
    if (port !== undefined) {
        const portNum = Number(port);
        if (!/^[0-9]+$/.test(port) || portNum < 1 || portNum > 65535) {
            return false;
        }
    }
    return true;
}
const extractDomainFromMatrixUserId = (mxid) => {
    const separatorIndex = mxid.indexOf(':', 1);
    if (separatorIndex === -1) {
        throw new Error(`Invalid federated username: ${mxid}`);
    }
    return mxid.substring(separatorIndex + 1);
};
exports.extractDomainFromMatrixUserId = extractDomainFromMatrixUserId;
/**
 * Extract the username and the servername from a matrix user id
 * if the serverName is the same as the serverName in the mxid, return only the username (rocket.chat regular username)
 * otherwise, return the full mxid and the servername
 */
const getUsernameServername = (mxid, serverName) => {
    const senderServerName = (0, exports.extractDomainFromMatrixUserId)(mxid);
    // if the serverName is the same as the serverName in the mxid, return only the username (rocket.chat regular username)
    if (serverName === senderServerName) {
        const separatorIndex = mxid.indexOf(':', 1);
        if (separatorIndex === -1) {
            throw new Error(`Invalid federated username: ${mxid}`);
        }
        return [mxid.substring(1, separatorIndex), senderServerName, true]; // removers also the @
    }
    return [mxid, senderServerName, false];
};
exports.getUsernameServername = getUsernameServername;
/**
 * Helper function to create a federated user
 *
 * Because of historical reasons, we can have users only with federated flag but no federation object
 * So we need to upsert the user with the federation object
 */
async function createOrUpdateFederatedUser(options) {
    const { username, name = username, origin } = options;
    const result = await models_1.Users.updateOne({
        username,
    }, {
        $set: {
            username,
            name: name || username,
            type: 'user',
            status: core_typings_1.UserStatus.OFFLINE,
            active: true,
            roles: ['user'],
            requirePasswordChange: false,
            federated: true,
            federation: {
                version: 1,
                mui: username,
                origin,
            },
            _updatedAt: new Date(),
        },
        $setOnInsert: {
            createdAt: new Date(),
        },
    }, {
        upsert: true,
    });
    const userId = result.upsertedId || (await models_1.Users.findOneByUsername(username, { projection: { _id: 1 } }))?._id;
    if (!userId) {
        throw new Error(`Failed to create or update federated user: ${username}`);
    }
    if (typeof userId !== 'string') {
        return userId.toString();
    }
    return userId;
}
var federation_sdk_2 = require("@rocket.chat/federation-sdk");
Object.defineProperty(exports, "generateEd25519RandomSecretKey", { enumerable: true, get: function () { return federation_sdk_2.generateEd25519RandomSecretKey; } });
class FederationMatrix extends core_services_1.ServiceClass {
    constructor() {
        super(...arguments);
        this.name = 'federation-matrix';
        this.logger = new logger_1.Logger(this.name);
    }
    async created() {
        // although this is async function, it is not awaited, so we need to register the listeners before everything else
        this.onEvent('watch.settings', async ({ clientAction, setting }) => {
            if (clientAction === 'removed') {
                return;
            }
            const { _id, value } = setting;
            if (_id === 'Federation_Service_Domain' && typeof value === 'string') {
                this.serverName = value;
            }
            else if (_id === 'Federation_Service_EDU_Process_Typing' && typeof value === 'boolean') {
                this.processEDUTyping = value;
            }
            else if (_id === 'Federation_Service_EDU_Process_Presence' && typeof value === 'boolean') {
                this.processEDUPresence = value;
            }
        });
        this.onEvent('presence.status', async ({ user }) => {
            if (!this.processEDUPresence) {
                return;
            }
            if (!user.username || !user.status || user.username.includes(':')) {
                return;
            }
            const localUser = await models_1.Users.findOneByUsername(user.username, { projection: { _id: 1, federated: 1, federation: 1 } });
            if (!localUser) {
                return;
            }
            if (!(0, core_typings_1.isUserNativeFederated)(localUser)) {
                return;
            }
            // TODO: Check if it should exclude himself from the list
            const roomsUserIsMemberOf = await models_1.Subscriptions.findUserFederatedRoomIds(localUser._id).toArray();
            const statusMap = {
                [core_typings_1.UserStatus.ONLINE]: 'online',
                [core_typings_1.UserStatus.OFFLINE]: 'offline',
                [core_typings_1.UserStatus.AWAY]: 'unavailable',
                [core_typings_1.UserStatus.BUSY]: 'unavailable',
                [core_typings_1.UserStatus.DISABLED]: 'offline',
            };
            void federation_sdk_1.federationSDK.sendPresenceUpdateToRooms([
                {
                    user_id: localUser.federation.mui,
                    presence: statusMap[user.status] || 'offline',
                },
            ], roomsUserIsMemberOf.map(({ externalRoomId }) => externalRoomId).filter(Boolean));
        });
        this.serverName = (await models_1.Settings.getValueById('Federation_Service_Domain')) || '';
        this.processEDUTyping = (await models_1.Settings.getValueById('Federation_Service_EDU_Process_Typing')) || false;
        this.processEDUPresence = (await models_1.Settings.getValueById('Federation_Service_EDU_Process_Presence')) || false;
    }
    async createRoom(room, owner, members) {
        if (room.t !== 'c' && room.t !== 'p') {
            throw new Error('Room is not a public or private room');
        }
        try {
            const matrixUserId = federation_sdk_1.userIdSchema.parse(`@${owner.username}:${this.serverName}`);
            const roomName = room.name || room.fname || 'Untitled Room';
            // canonical alias computed from name
            const matrixRoomResult = await federation_sdk_1.federationSDK.createRoom(matrixUserId, roomName, room.t === 'c' ? 'public' : 'invite');
            this.logger.debug('Matrix room created:', matrixRoomResult);
            await models_1.Rooms.setAsFederated(room._id, { mrid: matrixRoomResult.room_id, origin: this.serverName });
            const federatedRoom = await models_1.Rooms.findOneById(room._id);
            if (federatedRoom && (0, core_typings_1.isRoomNativeFederated)(federatedRoom)) {
                await this.inviteUsersToRoom(federatedRoom, members.filter((m) => m !== owner.username), owner);
            }
            this.logger.debug('Room creation completed successfully', room._id);
            return matrixRoomResult;
        }
        catch (error) {
            this.logger.error('Failed to create room:', error);
            throw error;
        }
    }
    async ensureFederatedUsersExistLocally(usernames) {
        try {
            this.logger.debug('Ensuring federated users exist locally before DM creation', { memberCount: usernames.length });
            const federatedUsers = usernames.filter(validateFederatedUsername);
            for await (const username of federatedUsers) {
                const existingUser = await models_1.Users.findOneByUsername(username);
                if (existingUser && (0, core_typings_1.isUserNativeFederated)(existingUser)) {
                    continue;
                }
                await createOrUpdateFederatedUser({
                    username,
                    name: username,
                    origin: (0, exports.extractDomainFromMatrixUserId)(username),
                });
            }
        }
        catch (error) {
            this.logger.error({ msg: 'Failed to ensure federated users exist locally', error });
            throw error;
        }
    }
    async createDirectMessageRoom(room, members, creatorId) {
        try {
            this.logger.debug('Creating direct message room in Matrix', { roomId: room._id, memberCount: members.length });
            const creator = await models_1.Users.findOneById(creatorId);
            if (!creator) {
                throw new Error('Creator not found in members list');
            }
            const actualMatrixUserId = `@${creator.username}:${this.serverName}`;
            let matrixRoomResult;
            if (members.length === 2) {
                const otherMember = members.find((member) => member._id !== creatorId);
                if (!otherMember) {
                    throw new Error('Other member not found for 1-on-1 DM');
                }
                if (!(0, core_typings_1.isUserNativeFederated)(otherMember)) {
                    throw new Error('Other member is not federated');
                }
                const roomId = await federation_sdk_1.federationSDK.createDirectMessageRoom(federation_sdk_1.userIdSchema.parse(actualMatrixUserId), federation_sdk_1.userIdSchema.parse(otherMember.username));
                matrixRoomResult = { room_id: roomId };
            }
            else {
                // For group DMs (more than 2 members), create a private room
                const roomName = room.name || room.fname || `Group chat with ${members.length} members`;
                matrixRoomResult = await federation_sdk_1.federationSDK.createRoom(federation_sdk_1.userIdSchema.parse(actualMatrixUserId), roomName, 'invite');
                for await (const member of members) {
                    if (member._id === creatorId) {
                        continue;
                    }
                    if (!(0, core_typings_1.isUserNativeFederated)(member)) {
                        continue;
                    }
                    try {
                        await federation_sdk_1.federationSDK.inviteUserToRoom(federation_sdk_1.userIdSchema.parse(member.username), federation_sdk_1.roomIdSchema.parse(matrixRoomResult.room_id), federation_sdk_1.userIdSchema.parse(actualMatrixUserId));
                    }
                    catch (error) {
                        this.logger.error('Error creating or updating bridged user for DM:', error);
                    }
                }
            }
            await models_1.Rooms.setAsFederated(room._id, {
                mrid: matrixRoomResult.room_id,
                origin: this.serverName,
            });
            this.logger.debug('Direct message room creation completed successfully', room._id);
        }
        catch (error) {
            this.logger.error('Failed to create direct message room:', error);
            throw error;
        }
    }
    getMatrixMessageType(mimeType) {
        const mainType = mimeType?.split('/')[0];
        if (!mainType) {
            return exports.fileTypes.file;
        }
        return exports.fileTypes[mainType] ?? exports.fileTypes.file;
    }
    async handleFileMessage(message, matrixRoomId, matrixUserId, matrixDomain) {
        if (!message.files || message.files.length === 0) {
            return null;
        }
        const replyToMessage = await this.handleThreadedMessage(message, matrixRoomId, matrixUserId, matrixDomain);
        const quoteMessage = await this.handleQuoteMessage(message, matrixRoomId, matrixUserId, matrixDomain);
        try {
            let lastEventId = null;
            // TODO handle multiple files, we currently save thumbs on files[], we need to flag them as thumb so we can ignore them here
            const [file] = message.files;
            const mxcUri = await MatrixMediaService_1.MatrixMediaService.prepareLocalFileForMatrix(file._id, matrixDomain, matrixRoomId);
            const msgtype = this.getMatrixMessageType(file.type);
            const fileContent = {
                body: file.name,
                msgtype,
                url: mxcUri,
                info: {
                    mimetype: file.type,
                    size: file.size,
                },
            };
            lastEventId = await federation_sdk_1.federationSDK.sendFileMessage(federation_sdk_1.roomIdSchema.parse(matrixRoomId), fileContent, federation_sdk_1.userIdSchema.parse(matrixUserId), replyToMessage || quoteMessage);
            return lastEventId;
        }
        catch (error) {
            this.logger.error('Failed to handle file message', {
                messageId: message._id,
                error,
            });
            throw error;
        }
    }
    async handleTextMessage(message, matrixRoomId, matrixUserId, matrixDomain) {
        const parsedMessage = await (0, message_parsers_1.toExternalMessageFormat)({
            message: message.msg,
            externalRoomId: matrixRoomId,
            homeServerDomain: matrixDomain,
        });
        const replyToMessage = await this.handleThreadedMessage(message, matrixRoomId, matrixUserId, matrixDomain);
        const quoteMessage = await this.handleQuoteMessage(message, matrixRoomId, matrixUserId, matrixDomain);
        return federation_sdk_1.federationSDK.sendMessage(federation_sdk_1.roomIdSchema.parse(matrixRoomId), message.msg, parsedMessage, federation_sdk_1.userIdSchema.parse(matrixUserId), replyToMessage || quoteMessage);
    }
    async handleThreadedMessage(message, matrixRoomId, matrixUserId, matrixDomain) {
        if (!message.tmid) {
            return;
        }
        const threadRootMessage = await models_1.Messages.findOneById(message.tmid);
        const threadRootEventId = threadRootMessage?.federation?.eventId;
        if (!threadRootEventId) {
            throw new Error('Thread root event ID not found');
        }
        const quoteMessageEventId = message.attachments?.some((attachment) => (0, core_typings_1.isQuoteAttachment)(attachment) && Boolean(attachment.message_link))
            ? (await this.getQuoteMessage(message, matrixRoomId, matrixUserId, matrixDomain))?.eventToReplyTo
            : undefined;
        const latestThreadMessage = !quoteMessageEventId
            ? (await models_1.Messages.findLatestFederationThreadMessageByTmid(message.tmid, message._id))?.federation?.eventId ||
                federation_sdk_1.eventIdSchema.parse(threadRootEventId)
            : undefined;
        if (!quoteMessageEventId && !latestThreadMessage) {
            throw new Error('No event to reply to found');
        }
        const eventToReplyToNormalized = federation_sdk_1.eventIdSchema.parse(quoteMessageEventId ?? latestThreadMessage);
        if (quoteMessageEventId) {
            return { threadEventId: federation_sdk_1.eventIdSchema.parse(threadRootEventId), replyToEventId: eventToReplyToNormalized };
        }
        return { threadEventId: federation_sdk_1.eventIdSchema.parse(threadRootEventId), latestThreadEventId: eventToReplyToNormalized };
    }
    async handleQuoteMessage(message, matrixRoomId, matrixUserId, matrixDomain) {
        if (!message.attachments?.some((attachment) => (0, core_typings_1.isQuoteAttachment)(attachment) && Boolean(attachment.message_link))) {
            return;
        }
        const quoteMessage = await this.getQuoteMessage(message, matrixRoomId, matrixUserId, matrixDomain);
        if (!quoteMessage) {
            throw new Error('Failed to retrieve quote message');
        }
        return {
            replyToEventId: federation_sdk_1.eventIdSchema.parse(quoteMessage.eventToReplyTo),
        };
    }
    async sendMessage(message, room, user) {
        try {
            const userMui = (0, core_typings_1.isUserNativeFederated)(user) ? user.federation.mui : `@${user.username}:${this.serverName}`;
            let result;
            if (message.files && message.files.length > 0) {
                result = await this.handleFileMessage(message, room.federation.mrid, userMui, this.serverName);
            }
            else {
                result = await this.handleTextMessage(message, room.federation.mrid, userMui, this.serverName);
            }
            if (!result) {
                throw new Error('Failed to send message to Matrix - no result returned');
            }
            await models_1.Messages.setFederationEventIdById(message._id, result.eventId);
            this.logger.debug('Message sent to Matrix successfully:', result.eventId);
        }
        catch (error) {
            this.logger.error('Failed to send message to Matrix:', error);
            throw error;
        }
    }
    async getQuoteMessage(message, matrixRoomId, matrixUserId, matrixDomain) {
        if (!message.attachments) {
            return;
        }
        const messageLink = message.attachments.find((attachment) => (0, core_typings_1.isQuoteAttachment)(attachment) && Boolean(attachment.message_link)).message_link;
        if (!messageLink) {
            return;
        }
        const messageToReplyToId = messageLink.includes('msg=') && messageLink?.split('msg=').pop();
        if (!messageToReplyToId) {
            return;
        }
        const messageToReplyTo = await models_1.Messages.findOneById(messageToReplyToId);
        if (!messageToReplyTo?.federation?.eventId) {
            return;
        }
        const { formattedMessage, message: rawMessage } = await (0, message_parsers_1.toExternalQuoteMessageFormat)({
            externalRoomId: matrixRoomId,
            eventToReplyTo: messageToReplyTo.federation?.eventId,
            originalEventSender: matrixUserId,
            message: message.msg,
            homeServerDomain: matrixDomain,
        });
        return {
            formattedMessage,
            rawMessage,
            eventToReplyTo: messageToReplyTo.federation.eventId,
        };
    }
    async deleteMessage(matrixRoomId, message) {
        try {
            if (!(0, core_typings_1.isMessageFromMatrixFederation)(message) || (0, core_typings_1.isDeletedMessage)(message)) {
                return;
            }
            const matrixEventId = message.federation?.eventId;
            if (!matrixEventId) {
                throw new Error(`No Matrix event ID mapping found for message ${message._id}`);
            }
            // TODO fix branded EventID and remove type casting
            // TODO message.u?.username is not the user who removed the message
            const eventId = await federation_sdk_1.federationSDK.redactMessage(federation_sdk_1.roomIdSchema.parse(matrixRoomId), federation_sdk_1.eventIdSchema.parse(matrixEventId));
            this.logger.debug('Message Redaction sent to Matrix successfully:', eventId);
        }
        catch (error) {
            this.logger.error('Failed to send redaction to Matrix:', error);
            throw error;
        }
    }
    async inviteUsersToRoom(room, matrixUsersUsername, inviter) {
        try {
            const inviterUserId = `@${inviter.username}:${this.serverName}`;
            await Promise.all(matrixUsersUsername.map(async (username) => {
                if (validateFederatedUsername(username)) {
                    return federation_sdk_1.federationSDK.inviteUserToRoom(federation_sdk_1.userIdSchema.parse(username), federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.userIdSchema.parse(inviterUserId));
                }
                // if inviter is an external user it means we receive the invite from the endpoint
                // since we accept from there we can skip accepting here
                if ((0, core_typings_1.isUserNativeFederated)(inviter)) {
                    this.logger.debug('Inviter is native federated, skip accept invite');
                    return;
                }
                const result = await federation_sdk_1.federationSDK.inviteUserToRoom(federation_sdk_1.userIdSchema.parse(`@${username}:${this.serverName}`), federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.userIdSchema.parse(inviterUserId));
                return (0, invite_1.acceptInvite)(result.event, username);
            }));
        }
        catch (error) {
            this.logger.error({ msg: 'Failed to invite an user to Matrix:', err: error });
            throw error;
        }
    }
    async sendReaction(messageId, reaction, user) {
        try {
            const message = await models_1.Messages.findOneById(messageId);
            if (!message) {
                throw new Error(`Message ${messageId} not found`);
            }
            const room = await models_1.Rooms.findOneById(message.rid);
            if (!room || !(0, core_typings_1.isRoomNativeFederated)(room)) {
                throw new Error(`No Matrix room mapping found for room ${message.rid}`);
            }
            const matrixEventId = message.federation?.eventId;
            if (!matrixEventId) {
                throw new Error(`No Matrix event ID mapping found for message ${messageId}`);
            }
            const reactionKey = emojione_1.default.shortnameToUnicode(reaction);
            const userMui = (0, core_typings_1.isUserNativeFederated)(user) ? user.federation.mui : `@${user.username}:${this.serverName}`;
            const eventId = await federation_sdk_1.federationSDK.sendReaction(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.eventIdSchema.parse(matrixEventId), reactionKey, federation_sdk_1.userIdSchema.parse(userMui));
            await models_1.Messages.setFederationReactionEventId(user.username || '', messageId, reaction, eventId);
            this.logger.debug('Reaction sent to Matrix successfully:', eventId);
        }
        catch (error) {
            this.logger.error('Failed to send reaction to Matrix:', error);
            throw error;
        }
    }
    async removeReaction(messageId, reaction, user, oldMessage) {
        try {
            const message = await models_1.Messages.findOneById(messageId);
            if (!message) {
                this.logger.error(`Message ${messageId} not found`);
                return;
            }
            const targetEventId = message.federation?.eventId;
            if (!targetEventId) {
                this.logger.warn(`No federation event ID found for message ${messageId}`);
                return;
            }
            const room = await models_1.Rooms.findOneById(message.rid);
            if (!room || !(0, core_typings_1.isRoomNativeFederated)(room)) {
                this.logger.error(`No Matrix room mapping found for room ${message.rid}`);
                return;
            }
            const reactionKey = emojione_1.default.shortnameToUnicode(reaction);
            const userMui = (0, core_typings_1.isUserNativeFederated)(user) ? user.federation.mui : `@${user.username}:${this.serverName}`;
            const reactionData = oldMessage.reactions?.[reaction];
            if (!reactionData?.federationReactionEventIds) {
                return;
            }
            for await (const [eventId, username] of Object.entries(reactionData.federationReactionEventIds)) {
                if (username !== user.username) {
                    continue;
                }
                const redactionEventId = await federation_sdk_1.federationSDK.unsetReaction(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.eventIdSchema.parse(eventId), reactionKey, federation_sdk_1.userIdSchema.parse(userMui));
                if (!redactionEventId) {
                    this.logger.warn('No reaction event found to remove in Matrix');
                    return;
                }
                await models_1.Messages.unsetFederationReactionEventId(eventId, messageId, reaction);
                break;
            }
        }
        catch (error) {
            this.logger.error('Failed to remove reaction from Matrix:', error);
            throw error;
        }
    }
    async getEventById(eventId) {
        return federation_sdk_1.federationSDK.getEventById(eventId);
    }
    async leaveRoom(roomId, user, kicker) {
        if (kicker && (0, core_typings_1.isUserNativeFederated)(kicker)) {
            this.logger.debug('Only local users can remove others, ignoring action');
            return;
        }
        try {
            const room = await models_1.Rooms.findOneById(roomId);
            if (!room || !(0, core_typings_1.isRoomNativeFederated)(room)) {
                this.logger.debug(`Room ${roomId} is not federated, skipping leave operation`);
                return;
            }
            const actualMatrixUserId = (0, core_typings_1.isUserNativeFederated)(user) ? user.federation.mui : `@${user.username}:${this.serverName}`;
            await federation_sdk_1.federationSDK.leaveRoom(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.userIdSchema.parse(actualMatrixUserId));
            this.logger.info(`User ${user.username} left Matrix room ${room.federation.mrid} successfully`);
        }
        catch (error) {
            this.logger.error('Failed to leave room in Matrix:', error);
            throw error;
        }
    }
    async kickUser(room, removedUser, userWhoRemoved) {
        try {
            const actualKickedMatrixUserId = (0, core_typings_1.isUserNativeFederated)(removedUser)
                ? removedUser.federation.mui
                : `@${removedUser.username}:${this.serverName}`;
            const actualSenderMatrixUserId = (0, core_typings_1.isUserNativeFederated)(userWhoRemoved)
                ? userWhoRemoved.federation.mui
                : `@${userWhoRemoved.username}:${this.serverName}`;
            await federation_sdk_1.federationSDK.kickUser(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.userIdSchema.parse(actualKickedMatrixUserId), federation_sdk_1.userIdSchema.parse(actualSenderMatrixUserId), `Kicked by ${userWhoRemoved.username}`);
            this.logger.info(`User ${removedUser.username} was kicked from Matrix room ${room.federation.mrid} by ${userWhoRemoved.username}`);
        }
        catch (error) {
            this.logger.error('Failed to kick user from Matrix room:', error);
            throw error;
        }
    }
    async updateMessage(room, message) {
        try {
            const matrixEventId = message.federation?.eventId;
            if (!matrixEventId) {
                throw new Error(`No Matrix event ID mapping found for message ${message._id}`);
            }
            const user = await models_1.Users.findOneById(message.u._id, { projection: { _id: 1, username: 1, federation: 1, federated: 1 } });
            if (!user) {
                this.logger.error(`No user found for ID ${message.u._id}`);
                return;
            }
            const userMui = (0, core_typings_1.isUserNativeFederated)(user) ? user.federation.mui : `@${user.username}:${this.serverName}`;
            const parsedMessage = await (0, message_parsers_1.toExternalMessageFormat)({
                message: message.msg,
                externalRoomId: room.federation.mrid,
                homeServerDomain: this.serverName,
            });
            const eventId = await federation_sdk_1.federationSDK.updateMessage(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), message.msg, parsedMessage, federation_sdk_1.userIdSchema.parse(userMui), federation_sdk_1.eventIdSchema.parse(matrixEventId));
            this.logger.debug('Message updated in Matrix successfully:', eventId);
        }
        catch (error) {
            this.logger.error('Failed to update message in Matrix:', error);
            throw error;
        }
    }
    async updateRoomName(rid, displayName, user) {
        const room = await models_1.Rooms.findOneById(rid);
        if (!room || !(0, core_typings_1.isRoomNativeFederated)(room)) {
            throw new Error(`No Matrix room mapping found for room ${rid}`);
        }
        if ((0, core_typings_1.isUserNativeFederated)(user)) {
            this.logger.debug('Only local users can change the name of a room, ignoring action');
            return;
        }
        const userMui = `@${user.username}:${this.serverName}`;
        await federation_sdk_1.federationSDK.updateRoomName(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), displayName, federation_sdk_1.userIdSchema.parse(userMui));
    }
    async updateRoomTopic(room, topic, user) {
        if ((0, core_typings_1.isUserNativeFederated)(user)) {
            this.logger.debug('Only local users can change the topic of a room, ignoring action');
            return;
        }
        const userMui = `@${user.username}:${this.serverName}`;
        await federation_sdk_1.federationSDK.setRoomTopic(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.userIdSchema.parse(userMui), topic);
    }
    async addUserRoleRoomScoped(room, senderId, userId, role) {
        if (role === 'leader') {
            throw new Error('Leader role is not supported');
        }
        const userSender = await models_1.Users.findOneById(senderId);
        if (!userSender) {
            throw new Error(`No user found for ID ${senderId}`);
        }
        if ((0, core_typings_1.isUserNativeFederated)(userSender)) {
            this.logger.debug('Only local users can change roles of other users in a room, ignoring action');
            return;
        }
        const senderMui = `@${userSender.username}:${this.serverName}`;
        const user = await models_1.Users.findOneById(userId);
        if (!user) {
            throw new Error(`No user found for ID ${userId}`);
        }
        const userMui = (0, core_typings_1.isUserNativeFederated)(user) ? user.federation.mui : `@${user.username}:${this.serverName}`;
        let powerLevel = 0;
        if (role === 'owner') {
            powerLevel = 100;
        }
        else if (role === 'moderator') {
            powerLevel = 50;
        }
        await federation_sdk_1.federationSDK.setPowerLevelForUser(federation_sdk_1.roomIdSchema.parse(room.federation.mrid), federation_sdk_1.userIdSchema.parse(senderMui), federation_sdk_1.userIdSchema.parse(userMui), powerLevel);
    }
    async notifyUserTyping(rid, user, isTyping) {
        if (!this.processEDUTyping) {
            return;
        }
        if (!rid || !user) {
            return;
        }
        const room = await models_1.Rooms.findOneById(rid);
        if (!room || !(0, core_typings_1.isRoomNativeFederated)(room)) {
            return;
        }
        const localUser = await models_1.Users.findOneByUsername(user, {
            projection: { _id: 1, username: 1, federation: 1, federated: 1 },
        });
        if (!localUser) {
            return;
        }
        const userMui = (0, core_typings_1.isUserNativeFederated)(localUser) ? localUser.federation.mui : `@${localUser.username}:${this.serverName}`;
        void federation_sdk_1.federationSDK.sendTypingNotification(room.federation.mrid, userMui, isTyping);
    }
    async verifyMatrixIds(matrixIds) {
        const results = Object.fromEntries(await Promise.all(matrixIds.map(async (matrixId) => {
            // Split only on the first ':' (after the leading '@') so we keep any port in the homeserver
            const separatorIndex = matrixId.indexOf(':', 1);
            if (separatorIndex === -1) {
                return [matrixId, 'UNABLE_TO_VERIFY'];
            }
            const userId = matrixId.slice(0, separatorIndex);
            const homeserverUrl = matrixId.slice(separatorIndex + 1);
            if (homeserverUrl === this.serverName) {
                const user = await models_1.Users.findOneByUsername(userId.slice(1));
                return [matrixId, user ? 'VERIFIED' : 'UNVERIFIED'];
            }
            if (!homeserverUrl) {
                return [matrixId, 'UNABLE_TO_VERIFY'];
            }
            try {
                const result = await federation_sdk_1.federationSDK.queryProfileRemote({ homeserverUrl, userId: matrixId });
                if ('errcode' in result && result.errcode === 'M_NOT_FOUND') {
                    return [matrixId, 'UNVERIFIED'];
                }
                return [matrixId, 'VERIFIED'];
            }
            catch (e) {
                return [matrixId, 'UNABLE_TO_VERIFY'];
            }
        })));
        return results;
    }
}
exports.FederationMatrix = FederationMatrix;
//# sourceMappingURL=FederationMatrix.js.map