import { Buffer } from 'buffer';
import { ProxiedApp } from './ProxiedApp';
import { AppBridges } from './bridges';
import { AppStatus, AppStatusUtils } from '../definition/AppStatus';
import { AppMethod } from '../definition/metadata';
import { UserType } from '../definition/users';
import { AppCompiler, AppFabricationFulfillment, AppPackageParser } from './compiler';
import { InvalidLicenseError } from './errors';
import { InvalidInstallationError } from './errors/InvalidInstallationError';
import { AppAccessorManager, AppApiManager, AppExternalComponentManager, AppLicenseManager, AppListenerManager, AppSchedulerManager, AppSettingsManager, AppSlashCommandManager, AppVideoConfProviderManager } from './managers';
import { AppOutboundCommunicationProviderManager } from './managers/AppOutboundCommunicationProviderManager';
import { AppRuntimeManager } from './managers/AppRuntimeManager';
import { AppSignatureManager } from './managers/AppSignatureManager';
import { UIActionButtonManager } from './managers/UIActionButtonManager';
import { defaultPermissions } from './permissions/AppPermissions';
import { EmptyRuntime } from './runtime/EmptyRuntime';
import { AppLogStorage, AppMetadataStorage } from './storage';
import { AppSourceStorage } from './storage/AppSourceStorage';
import { AppInstallationSource } from './storage/IAppStorageItem';
export class AppManager {
  static Instance;
  // apps contains all of the Apps
  apps;
  appMetadataStorage;
  appSourceStorage;
  logStorage;
  bridges;
  parser;
  compiler;
  accessorManager;
  listenerManager;
  commandManager;
  apiManager;
  externalComponentManager;
  settingsManager;
  licenseManager;
  schedulerManager;
  uiActionButtonManager;
  videoConfProviderManager;
  outboundCommunicationProviderManager;
  signatureManager;
  runtime;
  isLoaded;
  constructor({ metadataStorage, logStorage, bridges, sourceStorage }){
    // Singleton style. There can only ever be one AppManager instance
    if (typeof AppManager.Instance !== 'undefined') {
      throw new Error('There is already a valid AppManager instance');
    }
    if (metadataStorage instanceof AppMetadataStorage) {
      this.appMetadataStorage = metadataStorage;
    } else {
      throw new Error('Invalid instance of the AppMetadataStorage');
    }
    if (logStorage instanceof AppLogStorage) {
      this.logStorage = logStorage;
    } else {
      throw new Error('Invalid instance of the AppLogStorage');
    }
    if (bridges instanceof AppBridges) {
      this.bridges = bridges;
    } else {
      throw new Error('Invalid instance of the AppBridges');
    }
    if (sourceStorage instanceof AppSourceStorage) {
      this.appSourceStorage = sourceStorage;
    } else {
      throw new Error('Invalid instance of the AppSourceStorage');
    }
    this.apps = new Map();
    this.parser = new AppPackageParser();
    this.compiler = new AppCompiler();
    this.accessorManager = new AppAccessorManager(this);
    this.listenerManager = new AppListenerManager(this);
    this.commandManager = new AppSlashCommandManager(this);
    this.apiManager = new AppApiManager(this);
    this.externalComponentManager = new AppExternalComponentManager();
    this.settingsManager = new AppSettingsManager(this);
    this.licenseManager = new AppLicenseManager(this);
    this.schedulerManager = new AppSchedulerManager(this);
    this.uiActionButtonManager = new UIActionButtonManager(this);
    this.videoConfProviderManager = new AppVideoConfProviderManager(this);
    this.outboundCommunicationProviderManager = new AppOutboundCommunicationProviderManager(this);
    this.signatureManager = new AppSignatureManager(this);
    this.runtime = new AppRuntimeManager(this);
    this.isLoaded = false;
    AppManager.Instance = this;
  }
  /** Gets the instance of the storage connector. */ getStorage() {
    return this.appMetadataStorage;
  }
  /** Gets the instance of the log storage connector. */ getLogStorage() {
    return this.logStorage;
  }
  /** Gets the instance of the App package parser. */ getParser() {
    return this.parser;
  }
  /** Gets the compiler instance. */ getCompiler() {
    return this.compiler;
  }
  /** Gets the accessor manager instance. */ getAccessorManager() {
    return this.accessorManager;
  }
  /** Gets the instance of the Bridge manager. */ getBridges() {
    return this.bridges;
  }
  /** Gets the instance of the listener manager. */ getListenerManager() {
    return this.listenerManager;
  }
  /** Gets the command manager's instance. */ getCommandManager() {
    return this.commandManager;
  }
  getVideoConfProviderManager() {
    return this.videoConfProviderManager;
  }
  getOutboundCommunicationProviderManager() {
    return this.outboundCommunicationProviderManager;
  }
  getLicenseManager() {
    return this.licenseManager;
  }
  /** Gets the api manager's instance. */ getApiManager() {
    return this.apiManager;
  }
  /** Gets the external component manager's instance. */ getExternalComponentManager() {
    return this.externalComponentManager;
  }
  /** Gets the manager of the settings, updates and getting. */ getSettingsManager() {
    return this.settingsManager;
  }
  getSchedulerManager() {
    return this.schedulerManager;
  }
  getUIActionButtonManager() {
    return this.uiActionButtonManager;
  }
  getSignatureManager() {
    return this.signatureManager;
  }
  getRuntime() {
    return this.runtime;
  }
  /** Gets whether the Apps have been loaded or not. */ areAppsLoaded() {
    return this.isLoaded;
  }
  setSourceStorage(storage) {
    this.appSourceStorage = storage;
  }
  /**
	 * Goes through the entire loading up process.
	 * Expect this to take some time, as it goes through a very
	 * long process of loading all the Apps up.
	 */ async load() {
    // You can not load the AppManager system again
    // if it has already been loaded.
    if (this.isLoaded) {
      return true;
    }
    const items = await this.appMetadataStorage.retrieveAll();
    for (const item of items.values()){
      try {
        const appPackage = await this.appSourceStorage.fetch(item);
        const unpackageResult = await this.getParser().unpackageApp(appPackage);
        const app = await this.getCompiler().toSandBox(this, item, unpackageResult);
        this.apps.set(item.id, app);
      } catch (e) {
        console.warn(`Error while compiling the App "${item.info.name} (${item.id})":`);
        console.error(e);
        const prl = new ProxiedApp(this, item, new EmptyRuntime(item.id));
        this.apps.set(item.id, prl);
      }
    }
    this.isLoaded = true;
    return true;
  }
  async enableAll() {
    const affs = [];
    // Let's initialize them
    for (const rl of this.apps.values()){
      const aff = new AppFabricationFulfillment();
      aff.setAppInfo(rl.getInfo());
      aff.setImplementedInterfaces(rl.getImplementationList());
      aff.setApp(rl);
      affs.push(aff);
      if (AppStatusUtils.isDisabled(await rl.getStatus())) {
        // Usually if an App is disabled before it's initialized,
        // then something (such as an error) occured while
        // it was compiled or something similar.
        // We still have to validate its license, though
        await rl.validateLicense();
        continue;
      }
      await this.initializeApp(rl, true).catch(console.error);
    }
    // Let's ensure the required settings are all set
    for (const rl of this.apps.values()){
      if (AppStatusUtils.isDisabled(await rl.getStatus())) {
        continue;
      }
      if (!this.areRequiredSettingsSet(rl.getStorageItem())) {
        await rl.setStatus(AppStatus.INVALID_SETTINGS_DISABLED).catch(console.error);
      }
    }
    // Now let's enable the apps which were once enabled
    // but are not currently disabled.
    for (const app of this.apps.values()){
      const status = await app.getStatus();
      if (!AppStatusUtils.isDisabled(status) && AppStatusUtils.isEnabled(app.getPreviousStatus())) {
        await this.enableApp(app).catch(console.error);
      } else if (!AppStatusUtils.isError(status)) {
        this.listenerManager.lockEssentialEvents(app);
        this.uiActionButtonManager.clearAppActionButtons(app.getID());
      }
    }
    return affs;
  }
  async unload(isManual) {
    // If the AppManager hasn't been loaded yet, then
    // there is nothing to unload
    if (!this.isLoaded) {
      return;
    }
    for (const app of this.apps.values()){
      const status = await app.getStatus();
      if (status === AppStatus.INITIALIZED) {
        await this.purgeAppConfig(app);
      } else if (!AppStatusUtils.isDisabled(status)) {
        await this.disable(app.getID(), isManual ? AppStatus.MANUALLY_DISABLED : AppStatus.DISABLED);
      }
      this.listenerManager.releaseEssentialEvents(app);
      app.getRuntimeController().stopApp();
    }
    // Remove all the apps from the system now that we have unloaded everything
    this.apps.clear();
    this.isLoaded = false;
  }
  /** Gets the Apps which match the filter passed in. */ async get(filter) {
    let rls = [];
    if (typeof filter === 'undefined') {
      this.apps.forEach((rl)=>rls.push(rl));
      return rls;
    }
    let nothing = true;
    if (typeof filter.enabled === 'boolean' && filter.enabled) {
      for (const rl of this.apps.values()){
        if (AppStatusUtils.isEnabled(await rl.getStatus())) {
          rls.push(rl);
        }
      }
      nothing = false;
    }
    if (typeof filter.disabled === 'boolean' && filter.disabled) {
      for (const rl of this.apps.values()){
        if (AppStatusUtils.isDisabled(await rl.getStatus())) {
          rls.push(rl);
        }
      }
      nothing = false;
    }
    if (nothing) {
      this.apps.forEach((rl)=>rls.push(rl));
    }
    if (typeof filter.ids !== 'undefined') {
      rls = rls.filter((rl)=>filter.ids.includes(rl.getID()));
    }
    if (typeof filter.installationSource !== 'undefined') {
      rls = rls.filter((rl)=>rl.getInstallationSource() === filter.installationSource);
    }
    if (typeof filter.name === 'string') {
      rls = rls.filter((rl)=>rl.getName() === filter.name);
    } else if (filter.name instanceof RegExp) {
      rls = rls.filter((rl)=>filter.name.test(rl.getName()));
    }
    return rls;
  }
  /** Gets a single App by the id passed in. */ getOneById(appId) {
    return this.apps.get(appId);
  }
  getPermissionsById(appId) {
    const app = this.apps.get(appId);
    if (!app) {
      return [];
    }
    const { permissionsGranted } = app.getStorageItem();
    return permissionsGranted || defaultPermissions;
  }
  async enable(id) {
    const rl = this.apps.get(id);
    if (!rl) {
      throw new Error(`No App by the id "${id}" exists.`);
    }
    const status = await rl.getStatus();
    if (AppStatusUtils.isEnabled(status)) {
      return true;
    }
    if (status === AppStatus.COMPILER_ERROR_DISABLED) {
      throw new Error('The App had compiler errors, can not enable it.');
    }
    const storageItem = await this.appMetadataStorage.retrieveOne(id);
    if (!storageItem) {
      throw new Error(`Could not enable an App with the id of "${id}" as it doesn't exist.`);
    }
    const isSetup = await this.runStartUpProcess(storageItem, rl, false);
    return isSetup;
  }
  async disable(id, status = AppStatus.DISABLED, silent) {
    if (!AppStatusUtils.isDisabled(status)) {
      throw new Error('Invalid disabled status');
    }
    const app = this.apps.get(id);
    if (!app) {
      throw new Error(`No App by the id "${id}" exists.`);
    }
    if (AppStatusUtils.isEnabled(await app.getStatus())) {
      await app.call(AppMethod.ONDISABLE).catch((e)=>console.warn('Error while disabling:', e));
    }
    await this.purgeAppConfig(app, {
      keepScheduledJobs: true,
      keepSlashcommands: true,
      keepOutboundCommunicationProviders: true
    });
    await app.setStatus(status, silent);
    const storageItem = await this.appMetadataStorage.retrieveOne(id);
    app.getStorageItem().marketplaceInfo = storageItem.marketplaceInfo;
    await app.validateLicense().catch(()=>{});
    return true;
  }
  async migrate(id) {
    const app = this.apps.get(id);
    if (!app) {
      throw new Error(`No App by the id "${id}" exists.`);
    }
    await app.call(AppMethod.ONUPDATE).catch((e)=>console.warn('Error while migrating:', e));
    await this.purgeAppConfig(app, {
      keepScheduledJobs: true
    });
    const storageItem = await this.appMetadataStorage.retrieveOne(id);
    app.getStorageItem().marketplaceInfo = storageItem.marketplaceInfo;
    await app.validateLicense().catch(()=>{});
    storageItem.migrated = true;
    storageItem.signature = await this.getSignatureManager().signApp(storageItem);
    const { marketplaceInfo, signature, migrated, _id } = storageItem;
    const stored = await this.appMetadataStorage.updatePartialAndReturnDocument({
      marketplaceInfo,
      signature,
      migrated,
      _id
    });
    await this.updateLocal(stored, app);
    await this.bridges.getAppActivationBridge().doAppUpdated(app).catch(()=>{});
    return true;
  }
  async addLocal(appId) {
    const storageItem = await this.appMetadataStorage.retrieveOne(appId);
    if (!storageItem) {
      throw new Error(`App with id ${appId} couldn't be found`);
    }
    const appPackage = await this.appSourceStorage.fetch(storageItem);
    if (!appPackage) {
      throw new Error(`Package file for app "${storageItem.info.name}" (${appId}) couldn't be found`);
    }
    const parsedPackage = await this.getParser().unpackageApp(appPackage);
    const app = await this.getCompiler().toSandBox(this, storageItem, parsedPackage);
    this.apps.set(app.getID(), app);
    await this.loadOne(appId);
  }
  async add(appPackage, installationParameters) {
    const { enable = true, marketplaceInfo, permissionsGranted, user } = installationParameters;
    const aff = new AppFabricationFulfillment();
    const result = await this.getParser().unpackageApp(appPackage);
    const undoSteps = [];
    aff.setAppInfo(result.info);
    aff.setImplementedInterfaces(result.implemented.getValues());
    const descriptor = {
      id: result.info.id,
      info: result.info,
      status: enable ? AppStatus.MANUALLY_ENABLED : AppStatus.MANUALLY_DISABLED,
      settings: {},
      implemented: result.implemented.getValues(),
      installationSource: marketplaceInfo ? AppInstallationSource.MARKETPLACE : AppInstallationSource.PRIVATE,
      marketplaceInfo,
      permissionsGranted,
      languageContent: result.languageContent
    };
    try {
      descriptor.sourcePath = await this.appSourceStorage.store(descriptor, appPackage);
      undoSteps.push(()=>this.appSourceStorage.remove(descriptor));
    } catch (error) {
      aff.setStorageError('Failed to store app package');
      return aff;
    }
    let app;
    try {
      app = await this.getCompiler().toSandBox(this, descriptor, result);
    } catch (error) {
      await Promise.all(undoSteps.map((undoer)=>undoer()));
      throw error;
    }
    undoSteps.push(()=>this.getRuntime().stopRuntime(app.getRuntimeController()).catch(()=>{}));
    // Create a user for the app
    try {
      await this.createAppUser(result.info);
      undoSteps.push(()=>this.removeAppUser(app));
    } catch (err) {
      aff.setAppUserError({
        username: `${result.info.nameSlug}.bot`,
        message: 'Failed to create an app user for this app.'
      });
      await Promise.all(undoSteps.map((undoer)=>undoer()));
      return aff;
    }
    descriptor.signature = await this.getSignatureManager().signApp(descriptor);
    const created = await this.appMetadataStorage.create(descriptor);
    if (!created) {
      aff.setStorageError('Failed to create the App, the storage did not return it.');
      await Promise.all(undoSteps.map((undoer)=>undoer()));
      return aff;
    }
    app.getStorageItem()._id = created._id;
    this.apps.set(app.getID(), app);
    aff.setApp(app);
    // Let everyone know that the App has been added
    await this.bridges.getAppActivationBridge().doAppAdded(app).catch(()=>{
    // If an error occurs during this, oh well.
    });
    await this.installApp(app, user);
    // Should enable === true, then we go through the entire start up process
    // Otherwise, we only initialize it.
    if (enable) {
      // Start up the app
      await this.runStartUpProcess(created, app, false);
    } else {
      await this.initializeApp(app);
    }
    return aff;
  }
  /**
	 * Uninstalls specified app from the server and remove
	 * all database records regarding it
	 *
	 * @returns the instance of the removed ProxiedApp
	 */ async remove(id, uninstallationParameters) {
    const app = this.apps.get(id);
    const { user } = uninstallationParameters;
    // First remove the app
    await this.uninstallApp(app, user);
    await this.removeLocal(id);
    // Then let everyone know that the App has been removed
    await this.bridges.getAppActivationBridge().doAppRemoved(app).catch();
    return app;
  }
  /**
	 * Removes the app instance from the local Apps container
	 * and every type of data associated with it
	 */ async removeLocal(id) {
    const app = this.apps.get(id);
    if (AppStatusUtils.isEnabled(await app.getStatus())) {
      await this.disable(id);
    }
    await this.purgeAppConfig(app);
    this.listenerManager.releaseEssentialEvents(app);
    await this.removeAppUser(app);
    await this.bridges.getPersistenceBridge().purge(app.getID());
    await this.appMetadataStorage.remove(app.getID());
    await this.appSourceStorage.remove(app.getStorageItem()).catch(()=>{});
    // Errors here don't really prevent the process from dying, so we don't really need to do anything on the catch
    await this.getRuntime().stopRuntime(app.getRuntimeController()).catch(()=>{});
    this.apps.delete(app.getID());
  }
  async update(appPackage, permissionsGranted, updateOptions = {
    loadApp: true
  }) {
    const aff = new AppFabricationFulfillment();
    const result = await this.getParser().unpackageApp(appPackage);
    aff.setAppInfo(result.info);
    aff.setImplementedInterfaces(result.implemented.getValues());
    const old = await this.appMetadataStorage.retrieveOne(result.info.id);
    if (!old) {
      throw new Error('Can not update an App that does not currently exist.');
    }
    // If there is any error during disabling, it doesn't really matter
    await this.disable(old.id).catch(()=>{});
    const descriptor = {
      ...old,
      id: result.info.id,
      info: result.info,
      languageContent: result.languageContent,
      implemented: result.implemented.getValues()
    };
    if (!permissionsGranted) {
      delete descriptor.permissionsGranted;
    } else {
      descriptor.permissionsGranted = permissionsGranted;
    }
    try {
      descriptor.sourcePath = await this.appSourceStorage.update(descriptor, appPackage);
    } catch (error) {
      aff.setStorageError('Failed to storage app package');
      return aff;
    }
    descriptor.signature = await this.signatureManager.signApp(descriptor);
    const stored = await this.appMetadataStorage.updatePartialAndReturnDocument(descriptor, {
      unsetPermissionsGranted: typeof permissionsGranted === 'undefined'
    });
    // Errors here don't really prevent the process from dying, so we don't really need to do anything on the catch
    await this.getRuntime().stopRuntime(this.apps.get(old.id).getRuntimeController()).catch(()=>{});
    const app = await this.getCompiler().toSandBox(this, descriptor, result);
    // Ensure there is an user for the app
    try {
      await this.createAppUser(result.info);
    } catch (err) {
      aff.setAppUserError({
        username: `${result.info.nameSlug}.bot`,
        message: 'Failed to create an app user for this app.'
      });
      return aff;
    }
    aff.setApp(app);
    if (updateOptions.loadApp) {
      const shouldEnableApp = AppStatusUtils.isEnabled(old.status);
      if (shouldEnableApp) {
        await this.updateAndStartupLocal(stored, app);
      } else {
        await this.updateAndInitializeLocal(stored, app);
      }
      await this.bridges.getAppActivationBridge().doAppUpdated(app).catch(()=>{});
    }
    await this.updateApp(app, updateOptions.user, old.info.version);
    return aff;
  }
  /**
	 * Updates the local instance of an app.
	 *
	 * If the second parameter is a Buffer of an app package,
	 * unpackage and instantiate the app's main class
	 *
	 * With an instance of a ProxiedApp, start it up and replace
	 * the reference in the local app collection
	 */ async updateLocal(stored, appPackageOrInstance) {
    const app = await (async ()=>{
      if (appPackageOrInstance instanceof Buffer) {
        const parseResult = await this.getParser().unpackageApp(appPackageOrInstance);
        // Errors here don't really prevent the process from dying, so we don't really need to do anything on the catch
        await this.getRuntime().stopRuntime(this.apps.get(stored.id).getRuntimeController()).catch(()=>{});
        return this.getCompiler().toSandBox(this, stored, parseResult);
      }
      if (appPackageOrInstance instanceof ProxiedApp) {
        return appPackageOrInstance;
      }
    })();
    // We don't keep slashcommands here as the update could potentially not provide the same list
    await this.purgeAppConfig(app, {
      keepScheduledJobs: true
    });
    this.apps.set(app.getID(), app);
    return app;
  }
  async updateAndStartupLocal(stored, appPackageOrInstance) {
    const app = await this.updateLocal(stored, appPackageOrInstance);
    await this.runStartUpProcess(stored, app, true);
  }
  async updateAndInitializeLocal(stored, appPackageOrInstance) {
    const app = await this.updateLocal(stored, appPackageOrInstance);
    await this.initializeApp(app, true);
  }
  getLanguageContent() {
    const langs = {};
    this.apps.forEach((rl)=>{
      const content = rl.getStorageItem().languageContent;
      Object.keys(content).forEach((key)=>{
        langs[key] = Object.assign(langs[key] || {}, content[key]);
      });
    });
    return langs;
  }
  async changeStatus(appId, status) {
    switch(status){
      case AppStatus.MANUALLY_DISABLED:
      case AppStatus.MANUALLY_ENABLED:
        break;
      default:
        throw new Error('Invalid status to change an App to, must be manually disabled or enabled.');
    }
    const rl = this.apps.get(appId);
    if (!rl) {
      throw new Error('Can not change the status of an App which does not currently exist.');
    }
    const storageItem = await rl.getStorageItem();
    if (AppStatusUtils.isEnabled(status)) {
      // Then enable it
      if (AppStatusUtils.isEnabled(await rl.getStatus())) {
        throw new Error('Can not enable an App which is already enabled.');
      }
      await this.enable(rl.getID());
      storageItem.status = AppStatus.MANUALLY_ENABLED;
      await this.appMetadataStorage.updateStatus(storageItem._id, AppStatus.MANUALLY_ENABLED);
    } else {
      if (!AppStatusUtils.isEnabled(await rl.getStatus())) {
        throw new Error('Can not disable an App which is not enabled.');
      }
      await this.disable(rl.getID(), AppStatus.MANUALLY_DISABLED);
      storageItem.status = AppStatus.MANUALLY_DISABLED;
      await this.appMetadataStorage.updateStatus(storageItem._id, AppStatus.MANUALLY_DISABLED);
    }
    return rl;
  }
  async updateAppsMarketplaceInfo(appsOverview) {
    await Promise.all(appsOverview.map(async ({ latest: appInfo })=>{
      if (!appInfo.subscriptionInfo) {
        return;
      }
      const app = this.apps.get(appInfo.id);
      if (!app) {
        return;
      }
      const appStorageItem = app.getStorageItem();
      const { subscriptionInfo } = appStorageItem.marketplaceInfo?.[0] || {};
      if (subscriptionInfo && subscriptionInfo.license.license === appInfo.subscriptionInfo.license.license) {
        return;
      }
      appStorageItem.marketplaceInfo[0].subscriptionInfo = appInfo.subscriptionInfo;
      appStorageItem.signature = await this.getSignatureManager().signApp(appStorageItem);
      return this.appMetadataStorage.updatePartialAndReturnDocument({
        _id: appStorageItem._id,
        marketplaceInfo: appStorageItem.marketplaceInfo,
        signature: appStorageItem.signature
      });
    })).catch(()=>{});
    const queue = [];
    this.apps.forEach((app)=>queue.push(app.validateLicense().then(async ()=>{
        if (await app.getStatus() !== AppStatus.INVALID_LICENSE_DISABLED) {
          return;
        }
        return app.setStatus(AppStatus.DISABLED);
      }).catch(async (error)=>{
        if (!(error instanceof InvalidLicenseError)) {
          console.error(error);
          return;
        }
        await this.purgeAppConfig(app, {
          keepScheduledJobs: true
        });
        return app.setStatus(AppStatus.INVALID_LICENSE_DISABLED);
      }).then(async ()=>{
        const status = await app.getStatus();
        if (status === app.getPreviousStatus()) {
          return;
        }
        const storageItem = app.getStorageItem();
        storageItem.status = status;
        return this.appMetadataStorage.updateStatus(storageItem._id, storageItem.status).catch(console.error);
      })));
    await Promise.all(queue);
  }
  /**
	 * Goes through the entire loading up process.
	 *
	 * @param appId the id of the application to load
	 */ async loadOne(appId, silenceStatus = false) {
    const rl = this.apps.get(appId);
    if (!rl) {
      throw new Error(`No App found by the id of: "${appId}"`);
    }
    const item = rl.getStorageItem();
    await this.initializeApp(rl, silenceStatus);
    if (!this.areRequiredSettingsSet(item)) {
      await rl.setStatus(AppStatus.INVALID_SETTINGS_DISABLED);
    }
    if (!AppStatusUtils.isDisabled(await rl.getStatus()) && AppStatusUtils.isEnabled(rl.getPreviousStatus())) {
      await this.enableApp(rl, silenceStatus);
    }
    return this.apps.get(item.id);
  }
  async runStartUpProcess(storageItem, app, silenceStatus) {
    if (await app.getStatus() !== AppStatus.INITIALIZED) {
      const isInitialized = await this.initializeApp(app, silenceStatus);
      if (!isInitialized) {
        return false;
      }
    }
    if (!this.areRequiredSettingsSet(storageItem)) {
      await app.setStatus(AppStatus.INVALID_SETTINGS_DISABLED, silenceStatus);
      return false;
    }
    return this.enableApp(app, silenceStatus);
  }
  async installApp(app, user) {
    let result;
    const context = {
      user
    };
    try {
      await app.call(AppMethod.ONINSTALL, context);
      result = true;
    } catch (e) {
      const status = AppStatus.ERROR_DISABLED;
      result = false;
      await app.setStatus(status);
    }
    return result;
  }
  async updateApp(app, user, oldAppVersion) {
    let result;
    try {
      await app.call(AppMethod.ONUPDATE, {
        oldAppVersion,
        user
      });
      result = true;
    } catch (e) {
      const status = AppStatus.ERROR_DISABLED;
      result = false;
      await app.setStatus(status);
    }
    return result;
  }
  async initializeApp(app, silenceStatus = false) {
    let result;
    try {
      await app.validateLicense();
      await app.validateInstallation();
      await app.call(AppMethod.INITIALIZE);
      await app.setStatus(AppStatus.INITIALIZED, silenceStatus);
      await this.commandManager.registerCommands(app.getID());
      result = true;
    } catch (e) {
      let status = AppStatus.ERROR_DISABLED;
      if (e instanceof InvalidLicenseError) {
        status = AppStatus.INVALID_LICENSE_DISABLED;
      }
      if (e instanceof InvalidInstallationError) {
        status = AppStatus.INVALID_INSTALLATION_DISABLED;
      }
      await this.purgeAppConfig(app);
      result = false;
      await app.setStatus(status, silenceStatus);
    }
    return result;
  }
  async purgeAppConfig(app, opts = {}) {
    if (!opts.keepScheduledJobs) {
      await this.schedulerManager.cleanUp(app.getID());
    }
    if (!opts.keepSlashcommands) {
      await this.commandManager.unregisterCommands(app.getID());
    }
    this.listenerManager.unregisterListeners(app);
    this.listenerManager.lockEssentialEvents(app);
    this.externalComponentManager.unregisterExternalComponents(app.getID());
    await this.apiManager.unregisterApis(app.getID());
    this.accessorManager.purifyApp(app.getID());
    this.uiActionButtonManager.clearAppActionButtons(app.getID());
    this.videoConfProviderManager.unregisterProviders(app.getID());
    await this.outboundCommunicationProviderManager.unregisterProviders(app.getID(), {
      keepReferences: opts.keepOutboundCommunicationProviders
    });
  }
  /**
	 * Determines if the App's required settings are set or not.
	 * Should a packageValue be provided and not empty, then it's considered set.
	 */ areRequiredSettingsSet(storageItem) {
    let result = true;
    for (const setk of Object.keys(storageItem.settings)){
      const sett = storageItem.settings[setk];
      // If it's not required, ignore
      if (!sett.required) {
        continue;
      }
      if (sett.value !== 'undefined' || sett.packageValue !== 'undefined') {
        continue;
      }
      result = false;
    }
    return result;
  }
  async enableApp(app, silenceStatus = false) {
    let enable;
    let status = AppStatus.ERROR_DISABLED;
    try {
      await app.validateLicense();
      await app.validateInstallation();
      enable = await app.call(AppMethod.ONENABLE);
      if (enable) {
        status = AppStatus.MANUALLY_ENABLED;
      } else {
        status = AppStatus.DISABLED;
        console.warn(`The App (${app.getID()}) disabled itself when being enabled. \nCheck the "onEnable" implementation for details.`);
      }
    } catch (e) {
      enable = false;
      if (e instanceof InvalidLicenseError) {
        status = AppStatus.INVALID_LICENSE_DISABLED;
      }
      if (e instanceof InvalidInstallationError) {
        status = AppStatus.INVALID_INSTALLATION_DISABLED;
      }
      console.error(e);
    }
    if (enable) {
      this.externalComponentManager.registerExternalComponents(app.getID());
      await this.apiManager.registerApis(app.getID());
      this.listenerManager.registerListeners(app);
      this.listenerManager.releaseEssentialEvents(app);
      this.videoConfProviderManager.registerProviders(app.getID());
      await this.outboundCommunicationProviderManager.registerProviders(app.getID());
    } else {
      await this.purgeAppConfig(app, {
        keepScheduledJobs: true,
        keepSlashcommands: true,
        keepOutboundCommunicationProviders: true
      });
    }
    await app.setStatus(status, silenceStatus);
    return enable;
  }
  async createAppUser(appInfo) {
    const appUser = await this.bridges.getUserBridge().getAppUser(appInfo.id);
    if (appUser) {
      return appUser.id;
    }
    const userData = {
      username: `${appInfo.nameSlug}.bot`,
      name: appInfo.name,
      roles: [
        'app'
      ],
      appId: appInfo.id,
      type: UserType.APP,
      status: 'online',
      isEnabled: true
    };
    return this.bridges.getUserBridge().create(userData, appInfo.id, {
      avatarUrl: appInfo.iconFileContent || appInfo.iconFile,
      joinDefaultChannels: true,
      sendWelcomeEmail: false
    });
  }
  async removeAppUser(app) {
    const appUser = await this.bridges.getUserBridge().getAppUser(app.getID());
    if (!appUser) {
      return true;
    }
    return this.bridges.getUserBridge().remove(appUser, app.getID());
  }
  async uninstallApp(app, user) {
    let result;
    const context = {
      user
    };
    try {
      await app.call(AppMethod.ONUNINSTALL, context);
      result = true;
    } catch (e) {
      const status = AppStatus.ERROR_DISABLED;
      result = false;
      await app.setStatus(status);
    }
    return result;
  }
}
export const getPermissionsByAppId = (appId)=>{
  if (!AppManager.Instance) {
    console.error('AppManager should be instantiated first');
    return [];
  }
  return AppManager.Instance.getPermissionsById(appId);
};
//# sourceMappingURL=data:application/json;base64,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