if (!Deno.args.includes('--subprocess')) {
  Deno.stderr.writeSync(new TextEncoder().encode(`
            This is a Deno wrapper for Rocket.Chat Apps. It is not meant to be executed stand-alone;
            It is instead meant to be executed as a subprocess by the Apps-Engine framework.
       `));
  Deno.exit(1001);
}
import { JsonRpcError } from 'jsonrpc-lite';
import * as Messenger from './lib/messenger.ts';
import { decoder } from './lib/codec.ts';
import { AppObjectRegistry } from './AppObjectRegistry.ts';
import { Logger } from './lib/logger.ts';
import slashcommandHandler from './handlers/slashcommand-handler.ts';
import videoConferenceHandler from './handlers/videoconference-handler.ts';
import apiHandler from './handlers/api-handler.ts';
import handleApp from './handlers/app/handler.ts';
import handleScheduler from './handlers/scheduler-handler.ts';
import registerErrorListeners from './error-handlers.ts';
import { sendMetrics } from './lib/metricsCollector.ts';
import outboundMessageHandler from './handlers/outboundcomms-handler.ts';
const COMMAND_PING = '_zPING';
async function requestRouter({ type, payload }) {
  const methodHandlers = {
    app: handleApp,
    api: apiHandler,
    slashcommand: slashcommandHandler,
    videoconference: videoConferenceHandler,
    outboundCommunication: outboundMessageHandler,
    scheduler: handleScheduler,
    ping: (_method, _params)=>'pong'
  };
  // We're not handling notifications at the moment
  if (type === 'notification') {
    return Messenger.sendInvalidRequestError();
  }
  const { id, method, params } = payload;
  const logger = new Logger(method);
  AppObjectRegistry.set('logger', logger);
  const app = AppObjectRegistry.get('app');
  if (app) {
    // Same logic as applied in the ProxiedApp class previously
    app.logger = logger;
  }
  const [methodPrefix] = method.split(':');
  const handler = methodHandlers[methodPrefix];
  if (!handler) {
    return Messenger.errorResponse({
      error: {
        message: 'Method not found',
        code: -32601
      },
      id
    });
  }
  const result = await handler(method, params);
  if (result instanceof JsonRpcError) {
    return Messenger.errorResponse({
      id,
      error: result
    });
  }
  return Messenger.successResponse({
    id,
    result
  });
}
function handleResponse(response) {
  let event;
  if (response.type === 'error') {
    event = new ErrorEvent(`response:${response.payload.id}`, {
      error: response.payload
    });
  } else {
    event = new CustomEvent(`response:${response.payload.id}`, {
      detail: response.payload
    });
  }
  Messenger.RPCResponseObserver.dispatchEvent(event);
}
async function main() {
  Messenger.sendNotification({
    method: 'ready'
  });
  for await (const message of decoder.decodeStream(Deno.stdin.readable)){
    try {
      // Process PING command first as it is not JSON RPC
      if (message === COMMAND_PING) {
        void Messenger.pongResponse();
        void sendMetrics();
        continue;
      }
      const JSONRPCMessage = Messenger.parseMessage(message);
      if (Messenger.isRequest(JSONRPCMessage)) {
        void requestRouter(JSONRPCMessage);
        continue;
      }
      if (Messenger.isResponse(JSONRPCMessage)) {
        handleResponse(JSONRPCMessage);
      }
    } catch (error) {
      if (Messenger.isErrorResponse(error)) {
        await Messenger.errorResponse(error);
      } else {
        await Messenger.sendParseError();
      }
    }
  }
}
registerErrorListeners();
main();
//# sourceMappingURL=data:application/json;base64,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