import { writeAll } from 'https://deno.land/std@0.216.0/io/write_all.ts';
import * as jsonrpc from 'jsonrpc-lite';
import { AppObjectRegistry } from '../AppObjectRegistry.ts';
import { encoder } from './codec.ts';
export function isRequest(message) {
  return message.type === 'request' || message.type === 'notification';
}
export function isResponse(message) {
  return message.type === 'success' || message.type === 'error';
}
export function isErrorResponse(message) {
  return message instanceof jsonrpc.ErrorObject;
}
const COMMAND_PONG = '_zPONG';
export const RPCResponseObserver = new EventTarget();
export const Queue = new class Queue {
  queue = [];
  isProcessing = false;
  async processQueue() {
    if (this.isProcessing) {
      return;
    }
    this.isProcessing = true;
    while(this.queue.length){
      const message = this.queue.shift();
      if (message) {
        await Transport.send(message);
      }
    }
    this.isProcessing = false;
  }
  enqueue(message) {
    this.queue.push(encoder.encode(message));
    this.processQueue();
  }
  getCurrentSize() {
    return this.queue.length;
  }
}();
export const Transport = new class Transporter {
  selectedTransport;
  constructor(){
    this.selectedTransport = this.stdoutTransport.bind(this);
  }
  async stdoutTransport(message) {
    await writeAll(Deno.stdout, message);
  }
  async noopTransport(_message) {}
  selectTransport(transport) {
    switch(transport){
      case 'stdout':
        this.selectedTransport = this.stdoutTransport.bind(this);
        break;
      case 'noop':
        this.selectedTransport = this.noopTransport.bind(this);
        break;
    }
  }
  send(message) {
    return this.selectedTransport(message);
  }
}();
export function parseMessage(message) {
  let parsed;
  if (typeof message === 'string') {
    parsed = jsonrpc.parse(message);
  } else {
    parsed = jsonrpc.parseObject(message);
  }
  if (Array.isArray(parsed)) {
    throw jsonrpc.error(null, jsonrpc.JsonRpcError.invalidRequest(null));
  }
  if (parsed.type === 'invalid') {
    throw jsonrpc.error(null, parsed.payload);
  }
  return parsed;
}
export async function sendInvalidRequestError() {
  const rpc = jsonrpc.error(null, jsonrpc.JsonRpcError.invalidRequest(null));
  await Queue.enqueue(rpc);
}
export async function sendInvalidParamsError(id) {
  const rpc = jsonrpc.error(id, jsonrpc.JsonRpcError.invalidParams(null));
  await Queue.enqueue(rpc);
}
export async function sendParseError() {
  const rpc = jsonrpc.error(null, jsonrpc.JsonRpcError.parseError(null));
  await Queue.enqueue(rpc);
}
export async function sendMethodNotFound(id) {
  const rpc = jsonrpc.error(id, jsonrpc.JsonRpcError.methodNotFound(null));
  await Queue.enqueue(rpc);
}
export async function errorResponse({ error: { message, code = -32000, data = {} }, id }) {
  const logger = AppObjectRegistry.get('logger');
  if (logger?.hasEntries()) {
    data.logs = logger.getLogs();
  }
  const rpc = jsonrpc.error(id, new jsonrpc.JsonRpcError(message, code, data));
  await Queue.enqueue(rpc);
}
export async function successResponse({ id, result }) {
  const payload = {
    value: result
  };
  const logger = AppObjectRegistry.get('logger');
  if (logger?.hasEntries()) {
    payload.logs = logger.getLogs();
  }
  const rpc = jsonrpc.success(id, payload);
  await Queue.enqueue(rpc);
}
export function pongResponse() {
  return Promise.resolve(Queue.enqueue(COMMAND_PONG));
}
export async function sendRequest(requestDescriptor) {
  const request = jsonrpc.request(Math.random().toString(36).slice(2), requestDescriptor.method, requestDescriptor.params);
  // TODO: add timeout to this
  const responsePromise = new Promise((resolve, reject)=>{
    const handler = (event)=>{
      if (event instanceof ErrorEvent) {
        reject(event.error);
      }
      if (event instanceof CustomEvent) {
        resolve(event.detail);
      }
      RPCResponseObserver.removeEventListener(`response:${request.id}`, handler);
    };
    RPCResponseObserver.addEventListener(`response:${request.id}`, handler);
  });
  await Queue.enqueue(request);
  return responsePromise;
}
export function sendNotification({ method, params }) {
  const request = jsonrpc.notification(method, params);
  Queue.enqueue(request);
}
export function log(params) {
  sendNotification({
    method: 'log',
    params
  });
}
//# sourceMappingURL=data:application/json;base64,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