// @deno-types="../../acorn.d.ts"
export function getFunctionIdentifier(ancestors, functionNodeIndex) {
  const parent = ancestors[functionNodeIndex - 1];
  // If there is a parent node and it's not a computed property, we can try to
  // extract an identifier for our function from it. This needs to be done first
  // because when functions are assigned to named symbols, this will be the only
  // way to call it, even if the function itself has an identifier
  // Consider the following block:
  //
  // const foo = function bar() {}
  //
  // Even though the function itself has a name, the only way to call it in the
  // program is wiht `foo()`
  if (parent && !parent.computed) {
    // Several node types can have an id prop of type Identifier
    const { id } = parent;
    if (id?.type === 'Identifier') {
      return id.name;
    }
    // Usually assignments to object properties (MethodDefinition, Property)
    const { key } = parent;
    if (key?.type === 'Identifier') {
      return key.name;
    }
    // Variable assignments have left hand side that can be used as Identifier
    const { left } = parent;
    // Simple assignment: `const fn = () => {}`
    if (left?.type === 'Identifier') {
      return left.name;
    }
    // Object property assignment: `obj.fn = () => {}`
    if (left?.type === 'MemberExpression' && !left.computed) {
      return left.property.name;
    }
  }
  // nodeIndex needs to be the index of a Function node (either FunctionDeclaration or FunctionExpression)
  const currentNode = ancestors[functionNodeIndex];
  // Function declarations or expressions can be directly named
  if (currentNode.id?.type === 'Identifier') {
    return currentNode.id.name;
  }
}
export function wrapWithAwait(node) {
  if (!node.type.endsWith('Expression')) {
    throw new Error(`Can't wrap "${node.type}" with await`);
  }
  const innerNode = {
    ...node
  };
  node.type = 'AwaitExpression';
  // starting here node has become an AwaitExpression
  node.argument = innerNode;
  Object.keys(node).forEach((key)=>![
      'type',
      'argument'
    ].includes(key) && delete node[key]);
}
export function asyncifyScope(ancestors, state) {
  const functionNodeIndex = ancestors.findLastIndex((n)=>'async' in n);
  if (functionNodeIndex === -1) return;
  // At this point this is a node with an "async" property, so it has to be
  // of type Function - let TS know about that
  const functionScopeNode = ancestors[functionNodeIndex];
  if (functionScopeNode.async) {
    return;
  }
  functionScopeNode.async = true;
  // If the parent of a function node is a call expression, we're talking about an IIFE
  // Should we care about this case as well?
  // const parentNode = ancestors[functionScopeIndex-1];
  // if (parentNode?.type === 'CallExpression' && ancestors[functionScopeIndex-2] && ancestors[functionScopeIndex-2].type !== 'AwaitExpression') {
  //   pendingOperations.push(buildFunctionPredicate(getFunctionIdentifier(ancestors, functionScopeIndex-2)));
  // }
  const identifier = getFunctionIdentifier(ancestors, functionNodeIndex);
  // We can't fix calls of functions which name we can't determine at compile time
  if (!identifier) return;
  state.functionIdentifiers.add(identifier);
}
export function buildFixModifiedFunctionsOperation(functionIdentifiers) {
  return function _fixModifiedFunctionsOperation(node, state, ancestors) {
    if (node.type !== 'CallExpression') return;
    let isWrappable = false;
    // This node is a simple call to a function, like `fn()`
    isWrappable = node.callee.type === 'Identifier' && functionIdentifiers.has(node.callee.name);
    // This node is a call to an object property or instance method, like `obj.fn()`, but not computed like `obj[fn]()`
    isWrappable ||= node.callee.type === 'MemberExpression' && !node.callee.computed && node.callee.property?.type === 'Identifier' && functionIdentifiers.has(node.callee.property.name);
    // This is a weird dereferencing technique used by bundlers, and since we'll be dealing with bundled sources we have to check for it
    // e.g. `r=(0,fn)(e)`
    if (!isWrappable && node.callee.type === 'SequenceExpression') {
      const [, secondExpression] = node.callee.expressions;
      isWrappable = secondExpression?.type === 'Identifier' && functionIdentifiers.has(secondExpression.name);
      isWrappable ||= secondExpression?.type === 'MemberExpression' && !secondExpression.computed && secondExpression.property.type === 'Identifier' && functionIdentifiers.has(secondExpression.property.name);
    }
    if (!isWrappable) return;
    // ancestors[ancestors.length-1] === node, so here we're checking for parent node
    const parentNode = ancestors[ancestors.length - 2];
    if (!parentNode || parentNode.type === 'AwaitExpression') return;
    wrapWithAwait(node);
    asyncifyScope(ancestors, state);
    state.isModified = true;
  };
}
export const checkReassignmentOfModifiedIdentifiers = (node, { functionIdentifiers }, _ancestors)=>{
  if (node.type === 'AssignmentExpression') {
    if (node.operator !== '=') return;
    let identifier = '';
    if (node.left.type === 'Identifier') identifier = node.left.name;
    if (node.left.type === 'MemberExpression' && !node.left.computed) {
      identifier = node.left.property.name;
    }
    if (!identifier || node.right.type !== 'Identifier' || !functionIdentifiers.has(node.right.name)) return;
    functionIdentifiers.add(identifier);
    return;
  }
  if (node.type === 'VariableDeclarator') {
    if (node.id.type !== 'Identifier' || functionIdentifiers.has(node.id.name)) return;
    if (node.init?.type !== 'Identifier' || !functionIdentifiers.has(node.init?.name)) return;
    functionIdentifiers.add(node.id.name);
    return;
  }
  // "Property" is for plain objects, "PropertyDefinition" is for classes
  // but both share the same structure
  if (node.type === 'Property' || node.type === 'PropertyDefinition') {
    if (node.key.type !== 'Identifier' || functionIdentifiers.has(node.key.name)) return;
    if (node.value?.type !== 'Identifier' || !functionIdentifiers.has(node.value.name)) return;
    functionIdentifiers.add(node.key.name);
    return;
  }
};
export const fixLivechatIsOnlineCalls = (node, state, ancestors)=>{
  if (node.type !== 'MemberExpression' || node.computed) return;
  if (node.property.name !== 'isOnline') return;
  if (node.object.type !== 'CallExpression') return;
  if (node.object.callee.type !== 'MemberExpression') return;
  if (node.object.callee.property.name !== 'getLivechatReader') return;
  let parentIndex = ancestors.length - 2;
  let targetNode = ancestors[parentIndex];
  if (targetNode.type !== 'CallExpression') {
    targetNode = node;
  } else {
    parentIndex--;
  }
  // If we're already wrapped with an await, nothing to do
  if (ancestors[parentIndex].type === 'AwaitExpression') return;
  // If we're in the middle of a chained member access, we can't wrap with await
  if (ancestors[parentIndex].type === 'MemberExpression') return;
  wrapWithAwait(targetNode);
  asyncifyScope(ancestors, state);
  state.isModified = true;
};
export const fixRoomUsernamesCalls = (node, state, ancestors)=>{
  if (node.type !== 'MemberExpression' || node.computed) return;
  if (node.property.name !== 'usernames') return;
  let parentIndex = ancestors.length - 2;
  let targetNode = ancestors[parentIndex];
  if (targetNode.type !== 'CallExpression') {
    targetNode = node;
  } else {
    parentIndex--;
  }
  // If we're already wrapped with an await, nothing to do
  if (ancestors[parentIndex].type === 'AwaitExpression') return;
  wrapWithAwait(targetNode);
  asyncifyScope(ancestors, state);
  state.isModified = true;
};
//# sourceMappingURL=data:application/json;base64,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