import { BlockBuilder } from './BlockBuilder.ts';
import { require } from '../../../lib/require.ts';
const { RocketChatAssociationModel } = require('@rocket.chat/apps-engine/definition/metadata/RocketChatAssociations.js');
export class MessageBuilder {
  kind;
  msg;
  changes = {};
  attachmentsChanged = false;
  customFieldsChanged = false;
  constructor(message){
    this.kind = RocketChatAssociationModel.MESSAGE;
    this.msg = message || {};
  }
  setData(data) {
    delete data.id;
    this.msg = data;
    return this;
  }
  setUpdateData(data, editor) {
    this.msg = data;
    this.msg.editor = editor;
    this.msg.editedAt = new Date();
    this.changes = structuredClone(this.msg);
    return this;
  }
  setThreadId(threadId) {
    this.msg.threadId = threadId;
    this.changes.threadId = threadId;
    return this;
  }
  getThreadId() {
    return this.msg.threadId;
  }
  setRoom(room) {
    this.msg.room = room;
    this.changes.room = room;
    return this;
  }
  getRoom() {
    return this.msg.room;
  }
  setSender(sender) {
    this.msg.sender = sender;
    this.changes.sender = sender;
    return this;
  }
  getSender() {
    return this.msg.sender;
  }
  setText(text) {
    this.msg.text = text;
    this.changes.text = text;
    return this;
  }
  getText() {
    return this.msg.text;
  }
  setEmojiAvatar(emoji) {
    this.msg.emoji = emoji;
    this.changes.emoji = emoji;
    return this;
  }
  getEmojiAvatar() {
    return this.msg.emoji;
  }
  setAvatarUrl(avatarUrl) {
    this.msg.avatarUrl = avatarUrl;
    this.changes.avatarUrl = avatarUrl;
    return this;
  }
  getAvatarUrl() {
    return this.msg.avatarUrl;
  }
  setUsernameAlias(alias) {
    this.msg.alias = alias;
    this.changes.alias = alias;
    return this;
  }
  getUsernameAlias() {
    return this.msg.alias;
  }
  addAttachment(attachment) {
    if (!this.msg.attachments) {
      this.msg.attachments = [];
    }
    this.msg.attachments.push(attachment);
    this.attachmentsChanged = true;
    return this;
  }
  setAttachments(attachments) {
    this.msg.attachments = attachments;
    this.attachmentsChanged = true;
    return this;
  }
  getAttachments() {
    return this.msg.attachments;
  }
  replaceAttachment(position, attachment) {
    if (!this.msg.attachments?.[position]) {
      throw new Error(`No attachment found at the index of "${position}" to replace.`);
    }
    this.msg.attachments[position] = attachment;
    this.attachmentsChanged = true;
    return this;
  }
  removeAttachment(position) {
    if (!this.msg.attachments?.[position]) {
      throw new Error(`No attachment found at the index of "${position}" to remove.`);
    }
    this.msg.attachments.splice(position, 1);
    this.attachmentsChanged = true;
    return this;
  }
  setEditor(user) {
    this.msg.editor = user;
    this.changes.editor = user;
    return this;
  }
  getEditor() {
    return this.msg.editor;
  }
  setGroupable(groupable) {
    this.msg.groupable = groupable;
    this.changes.groupable = groupable;
    return this;
  }
  getGroupable() {
    return this.msg.groupable;
  }
  setParseUrls(parseUrls) {
    this.msg.parseUrls = parseUrls;
    this.changes.parseUrls = parseUrls;
    return this;
  }
  getParseUrls() {
    return this.msg.parseUrls;
  }
  getMessage() {
    if (!this.msg.room) {
      throw new Error('The "room" property is required.');
    }
    return this.msg;
  }
  addBlocks(blocks) {
    if (!Array.isArray(this.msg.blocks)) {
      this.msg.blocks = [];
    }
    if (blocks instanceof BlockBuilder) {
      this.msg.blocks.push(...blocks.getBlocks());
    } else {
      this.msg.blocks.push(...blocks);
    }
    return this;
  }
  setBlocks(blocks) {
    const blockArray = blocks instanceof BlockBuilder ? blocks.getBlocks() : blocks;
    this.msg.blocks = blockArray;
    this.changes.blocks = blockArray;
    return this;
  }
  getBlocks() {
    return this.msg.blocks;
  }
  addCustomField(key, value) {
    if (!this.msg.customFields) {
      this.msg.customFields = {};
    }
    if (this.msg.customFields[key]) {
      throw new Error(`The message already contains a custom field by the key: ${key}`);
    }
    if (key.includes('.')) {
      throw new Error(`The given key contains a period, which is not allowed. Key: ${key}`);
    }
    this.msg.customFields[key] = value;
    this.customFieldsChanged = true;
    return this;
  }
  getChanges() {
    const changes = structuredClone(this.changes);
    if (this.attachmentsChanged) {
      changes.attachments = structuredClone(this.msg.attachments);
    }
    if (this.customFieldsChanged) {
      changes.customFields = structuredClone(this.msg.customFields);
    }
    return changes;
  }
}
//# sourceMappingURL=data:application/json;base64,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