import { JsonRpcError } from 'jsonrpc-lite';
import handleConstructApp from './construct.ts';
import handleInitialize from './handleInitialize.ts';
import handleGetStatus from './handleGetStatus.ts';
import handleSetStatus from './handleSetStatus.ts';
import handleOnEnable from './handleOnEnable.ts';
import handleOnInstall from './handleOnInstall.ts';
import handleOnDisable from './handleOnDisable.ts';
import handleOnUninstall from './handleOnUninstall.ts';
import handleOnPreSettingUpdate from './handleOnPreSettingUpdate.ts';
import handleOnSettingUpdated from './handleOnSettingUpdated.ts';
import handleListener from '../listener/handler.ts';
import handleUIKitInteraction, { uikitInteractions } from '../uikit/handler.ts';
import { AppObjectRegistry } from '../../AppObjectRegistry.ts';
import handleOnUpdate from './handleOnUpdate.ts';
export default async function handleApp(method, params) {
  const [, appMethod] = method.split(':');
  try {
    // We don't want the getStatus method to generate logs, so we handle it separately
    if (appMethod === 'getStatus') {
      return await handleGetStatus();
    }
    // `app` will be undefined if the method here is "app:construct"
    const app = AppObjectRegistry.get('app');
    app?.getLogger().debug(`'${appMethod}' is being called...`);
    if (uikitInteractions.includes(appMethod)) {
      return handleUIKitInteraction(appMethod, params).then((result)=>{
        if (result instanceof JsonRpcError) {
          app?.getLogger().debug(`'${appMethod}' was unsuccessful.`, result.message);
        } else {
          app?.getLogger().debug(`'${appMethod}' was successfully called! The result is:`, result);
        }
        return result;
      });
    }
    if (appMethod.startsWith('check') || appMethod.startsWith('execute')) {
      return handleListener(appMethod, params).then((result)=>{
        if (result instanceof JsonRpcError) {
          app?.getLogger().debug(`'${appMethod}' was unsuccessful.`, result.message);
        } else {
          app?.getLogger().debug(`'${appMethod}' was successfully called! The result is:`, result);
        }
        return result;
      });
    }
    let result;
    switch(appMethod){
      case 'construct':
        result = await handleConstructApp(params);
        break;
      case 'initialize':
        result = await handleInitialize();
        break;
      case 'setStatus':
        result = await handleSetStatus(params);
        break;
      case 'onEnable':
        result = await handleOnEnable();
        break;
      case 'onDisable':
        result = await handleOnDisable();
        break;
      case 'onInstall':
        result = await handleOnInstall(params);
        break;
      case 'onUninstall':
        result = await handleOnUninstall(params);
        break;
      case 'onPreSettingUpdate':
        result = await handleOnPreSettingUpdate(params);
        break;
      case 'onSettingUpdated':
        result = await handleOnSettingUpdated(params);
        break;
      case 'onUpdate':
        result = await handleOnUpdate(params);
        break;
      default:
        throw new JsonRpcError('Method not found', -32601);
    }
    app?.getLogger().debug(`'${appMethod}' was successfully called! The result is:`, result);
    return result;
  } catch (e) {
    if (!(e instanceof Error)) {
      return new JsonRpcError('Unknown error', -32000, e);
    }
    if (e.cause?.includes('invalid_param_type')) {
      return JsonRpcError.invalidParams(null);
    }
    if (e.cause?.includes('invalid_app')) {
      return JsonRpcError.internalError({
        message: 'App unavailable'
      });
    }
    return new JsonRpcError(e.message, -32000, e);
  }
}
//# sourceMappingURL=data:application/json;base64,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