import { AppObjectRegistry } from '../../AppObjectRegistry.ts';
import { require } from '../../lib/require.ts';
import { sanitizeDeprecatedUsage } from '../../lib/sanitizeDeprecatedUsage.ts';
import { AppAccessorsInstance } from '../../lib/accessors/mod.ts';
import { Socket } from 'node:net';
const ALLOWED_NATIVE_MODULES = [
  'path',
  'url',
  'crypto',
  'buffer',
  'stream',
  'net',
  'http',
  'https',
  'zlib',
  'util',
  'punycode',
  'os',
  'querystring',
  'fs'
];
const ALLOWED_EXTERNAL_MODULES = [
  'uuid'
];
function prepareEnvironment() {
  // Deno does not behave equally to Node when it comes to piping content to a socket
  // So we intervene here
  const originalFinal = Socket.prototype._final;
  Socket.prototype._final = function _final(cb) {
    // Deno closes the readable stream in the Socket earlier than Node
    // The exact reason for that is yet unknown, so we'll need to simply delay the execution
    // which allows data to be read in a response
    setTimeout(()=>originalFinal.call(this, cb), 1);
  };
}
// As the apps are bundled, the only times they will call require are
// 1. To require native modules
// 2. To require external npm packages we may provide
// 3. To require apps-engine files
function buildRequire() {
  return (module)=>{
    // Normalize Node built-in specifiers: accept both 'crypto' and 'node:crypto'
    const normalized = module.replace('node:', '');
    if (ALLOWED_NATIVE_MODULES.includes(normalized)) {
      return require(`node:${normalized}`);
    }
    if (ALLOWED_EXTERNAL_MODULES.includes(module)) {
      return require(`npm:${module}`);
    }
    if (module.startsWith('@rocket.chat/apps-engine')) {
      // Our `require` function knows how to handle these
      return require(module);
    }
    throw new Error(`Module ${module} is not allowed`);
  };
}
function wrapAppCode(code) {
  return new Function('require', `
        const { Buffer } = require('buffer');
        const exports = {};
        const module = { exports };
        const _error = console.error.bind(console);
        const _console = {
            log: _error,
            error: _error,
            debug: _error,
            info: _error,
            warn: _error,
        };

        const result = (async (exports,module,require,Buffer,console,globalThis,Deno) => {
            ${code};
        })(exports,module,require,Buffer,_console,undefined,undefined);

        return result.then(() => module.exports);`);
}
export default async function handleConstructApp(params) {
  if (!Array.isArray(params)) {
    throw new Error('Invalid params', {
      cause: 'invalid_param_type'
    });
  }
  const [appPackage] = params;
  if (!appPackage?.info?.id || !appPackage?.info?.classFile || !appPackage?.files) {
    throw new Error('Invalid params', {
      cause: 'invalid_param_type'
    });
  }
  prepareEnvironment();
  AppObjectRegistry.set('id', appPackage.info.id);
  const source = sanitizeDeprecatedUsage(appPackage.files[appPackage.info.classFile]);
  const require = buildRequire();
  const exports = await wrapAppCode(source)(require);
  // This is the same naive logic we've been using in the App Compiler
  // Applying the correct type here is quite difficult because of the dynamic nature of the code
  // deno-lint-ignore no-explicit-any
  const appClass = Object.values(exports)[0];
  const logger = AppObjectRegistry.get('logger');
  const app = new appClass(appPackage.info, logger, AppAccessorsInstance.getDefaultAppAccessors());
  if (typeof app.getName !== 'function') {
    throw new Error('App must contain a getName function');
  }
  if (typeof app.getNameSlug !== 'function') {
    throw new Error('App must contain a getNameSlug function');
  }
  if (typeof app.getVersion !== 'function') {
    throw new Error('App must contain a getVersion function');
  }
  if (typeof app.getID !== 'function') {
    throw new Error('App must contain a getID function');
  }
  if (typeof app.getDescription !== 'function') {
    throw new Error('App must contain a getDescription function');
  }
  if (typeof app.getRequiredApiVersion !== 'function') {
    throw new Error('App must contain a getRequiredApiVersion function');
  }
  AppObjectRegistry.set('app', app);
  return true;
}
//# sourceMappingURL=data:application/json;base64,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